/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.io.File;
import java.math.RoundingMode;
import java.text.NumberFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Helper for report creation.
 */
public class ReportHelper implements Component, Serviceable, Initializable
{
    /** The avalon role */
    public static final String ROLE = ReportHelper.class.getName();
    
    /** Number format with 2 digits after the comma */
    public static final NumberFormat FORMAT_2_DIGITS = NumberFormat.getInstance(Locale.FRANCE);
    static
    {
        FORMAT_2_DIGITS.setRoundingMode(RoundingMode.HALF_UP);
        FORMAT_2_DIGITS.setMaximumFractionDigits(2);
        FORMAT_2_DIGITS.setGroupingUsed(false);
    }
    
    /** Comparator on content titles */
    public static final Comparator<Content> CONTENT_TITLE_COMPARATOR = Comparator.<Content, String>comparing(Content::getTitle).thenComparing(Content::hashCode);
    
    private static final String __READABLE_DF = "dd/MM/yyyy";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The root orgunit provider */
    protected RootOrgUnitProvider _rootOrgUnitProvider;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;

    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    private File _pilotageFolder;
    
    @Override
    public void initialize() throws Exception
    {
        _pilotageFolder = new File(AmetysHomeHelper.getAmetysHomeData(), "odf/gestion");
        FileUtils.forceMkdir(_pilotageFolder);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _rootOrgUnitProvider = (RootOrgUnitProvider) manager.lookup(RootOrgUnitProvider.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    /**
     * Get the current date to the following format : 'dd/MM/yyyy'
     * @return The date as a {@link String}
     */
    public String getReadableCurrentDate()
    {
        return LocalDate.now().format(DateTimeFormatter.ofPattern(__READABLE_DF));
    }
    
    /**
     * Get the current org unit, if empty returns all children of the root org unit.
     * @param orgUnitId the org unit id, can be null or empty
     * @return a list of org unit, corresponding to the selected org unit or to the child org units of the root org unit
     */
    public List<OrgUnit> getOrgUnits(String orgUnitId)
    {
        return Optional.ofNullable(orgUnitId)
            .filter(StringUtils::isNotEmpty)
            .map(_resolver::<OrgUnit>resolveById)
            .map(List::of)
            .orElseGet(this::_getRootOrgUnits);
    }
    
    private List<OrgUnit> _getRootOrgUnits()
    {
        return Stream.of(_rootOrgUnitProvider.getRoot().<ContentValue[]>getValue(OrgUnit.CHILD_ORGUNITS))
            .map(ContentValue::getContentIfExists)
            .flatMap(Optional::stream)
            .map(OrgUnit.class::cast)
            .toList();
    }

    /**
     * Get the acronym if exists or UAI code of the orgunit given.
     * @param orgUnit The orgUnit
     * @return The acronym if it exists, otherwise the UAI code
     */
    public String getAcronymOrUaiCode(OrgUnit orgUnit)
    {
        return Optional.of(orgUnit)
            .map(OrgUnit::getAcronym)
            .filter(StringUtils::isNotBlank)
            .orElseGet(orgUnit::getUAICode);
    }

    /**
     * Format the given long
     * @param number the long
     * @return string representation of this long
     */
    public String formatNumberToSax(Long number)
    {
        return number > 0 ? String.valueOf(number) : "";
    }

    /**
     * Get the list of courses underneath the given ametys object
     * @param programItem The program item to gather the courses from
     * @return the map representation of the tree of ametys objects
     */
    public Map<ProgramItem, Object> getCoursesFromContent(ProgramItem programItem)
    {
        Map<ProgramItem, Object> contentTree = new LinkedHashMap<>();
        for (ProgramItem childProgramItem : _odfHelper.getChildProgramItems(programItem))
        {
            if (childProgramItem instanceof Course)
            {
                contentTree.put(childProgramItem, getCoursesFromContent(childProgramItem));
            }
            else
            {
                Map<ProgramItem, Object> childTree = getCoursesFromContent(childProgramItem);
                if (MapUtils.isNotEmpty(childTree))
                {
                    contentTree.put(childProgramItem, childTree);
                }
            }
        }

        return contentTree;
    }

    /**
     * Generates SAX events for a multiple enumerated attribute. The attribute must be of type content or string
     * @param handler The handler
     * @param content The content
     * @param attributeName The attribute name
     * @param tagName The name of the tag
     * @throws SAXException if an error occurs
     */
    public void saxContentAttribute(ContentHandler handler, ModifiableDefaultContent content, String attributeName, String tagName) throws SAXException
    {
        Locale lang = LocaleUtils.toLocale(content.getLanguage());
        
        if (!ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(content.getType(attributeName).getId()))
        {
            throw new IllegalArgumentException("The attribute '" + attributeName + "' should be of type '" + ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID + "'.");
        }
        
        Object value = content.getValue(attributeName);
        if (value != null)
        {
            Stream<ContentValue> values = content.isMultiple(attributeName)
                    ? Stream.of((ContentValue[]) value)
                    : Stream.of((ContentValue) value);
            
            String tagValue =
                    values.map(ContentValue::getContentIfExists)
                        .flatMap(Optional::stream)
                        .map(c -> c.getTitle(lang))
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.joining(", "));
            
            XMLUtils.createElement(handler, tagName, tagValue);
        }
    }
    
    /**
     * Sax the "natures d'enseignement" from the reference table.
     * @param handler The handler
     * @param logger The logger
     * @throws SAXException if an error occurs
     */
    public void saxNaturesEnseignement(ContentHandler handler, Logger logger) throws SAXException
    {
        String lang = Config.getInstance().getValue("odf.programs.lang");
        Locale langAsLocale = LocaleUtils.toLocale(lang);

        Map<String, List<OdfReferenceTableEntry>> itemsByCategory =
            _refTableHelper.getItems(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE)
                .stream()
                .collect(Collectors.groupingBy(item -> ContentDataHelper.getContentIdFromContentData(item.getContent(), "category")));
        
        AttributesImpl rootAttrs = new AttributesImpl();
        rootAttrs.addCDATAAttribute("id", OdfReferenceTableHelper.ENSEIGNEMENT_NATURE);
        XMLUtils.startElement(handler, "items", rootAttrs);
        for (String categoryId : itemsByCategory.keySet())
        {
            AttributesImpl attr = new AttributesImpl();
            try
            {
                Optional<Content> category = Optional.ofNullable(categoryId)
                    .filter(StringUtils::isNotBlank)
                    .map(_resolver::<Content>resolveById);
                attr.addCDATAAttribute("title", category.map(c -> c.getTitle(langAsLocale)).orElse(StringUtils.EMPTY));
                attr.addCDATAAttribute("code", category.map(c -> c.<String>getValue("code")).orElse(StringUtils.EMPTY));
                attr.addCDATAAttribute("order", String.valueOf(category.map(c -> c.<Long>getValue("order")).orElse(Long.MAX_VALUE)));
            }
            catch (UnknownAmetysObjectException e)
            {
                if (StringUtils.isNotEmpty(categoryId))
                {
                    logger.warn("There is no content matching with the ID {}.", categoryId);
                }
            }
            
            XMLUtils.startElement(handler, "category", attr);
            for (OdfReferenceTableEntry item : itemsByCategory.get(categoryId))
            {
                attr = new AttributesImpl();
                attr.addCDATAAttribute("id", item.getId());
                attr.addCDATAAttribute("code", item.getCode());
                attr.addCDATAAttribute("order", item.getOrder().toString());
                attr.addCDATAAttribute("archived", item.isArchived().toString());
                attr.addCDATAAttribute("shortLabel", item.getContent().getValue("shortLabel", false, StringUtils.EMPTY));
                XMLUtils.createElement(handler, "item", attr, item.getLabel(lang));
            }
            XMLUtils.endElement(handler, "category");
        }
        XMLUtils.endElement(handler, "items");
    }
    
    /**
     * Get the pilotage folder.
     * @return The pilotage folder
     */
    public File getPilotageFolder()
    {
        return _pilotageFolder;
    }
    
    /**
     * Try to delete reports.
     * @param reports Reports to delete
     * @return A {@link Map} with name of deleted (deletedReports key) and undeleted (undeletedReports key) reports.
     */
    @Callable(rights = "ODF_Rights_Pilotage_Report_Delete")
    public Map<String, List<String>> deleteReports(List<String> reports)
    {
        Map<String, List<String>> result = new HashMap<>();
        
        File pilotageFolder = getPilotageFolder();
        for (String report : reports)
        {
            File reportFile = new File(pilotageFolder, report);
            FileUtils.deleteQuietly(reportFile);
            result.computeIfAbsent(reportFile.exists() ? "undeletedReports" : "deletedReports", __ -> new ArrayList<>()).add(report);
        }
        
        return result;
    }
    
    /**
     * Sax MCC workflow for a given container
     * @param handler the handler
     * @param container the container
     * @throws SAXException if a saxing error occurred
     */
    public void saxMCCWorkflow(ContentHandler handler, Container container) throws SAXException
    {
        XMLUtils.startElement(handler, "mcc-workflow");
        for (MCCWorkflowStep step : _mccWorkflowHelper.getCurrentSteps(container))
        {
            XMLUtils.createElement(handler, "workflow-step", step.name());
        }
        XMLUtils.endElement(handler, "mcc-workflow");
    }
}
