/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.validators;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.View;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Check the unicity on changing catalog by content type, code, catalog and lang.
 */
public class UnicityValidator extends AbstractContentValidator implements Serviceable
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content types helper */
    protected ContentTypesHelper _contentTypeHelper;
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** The languages manager */
    protected LanguagesManager _languagesManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _languagesManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
    }
    
    public ValidationResult validate(Content content)
    {
        // Nothing to do
        return ValidationResult.empty();
    }

    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        if (values.containsKey("catalog"))
        {
            String oldCatalog = content.getValue("catalog");
            String newCatalog = (String) ((SynchronizableValue) values.get("catalog")).getLocalValue();
            
            if (!newCatalog.equals(oldCatalog))
            {
                Expression expression = new AndExpression(
                    new ContentTypeExpression(Operator.EQ, content.getTypes()),
                    new LanguageExpression(Operator.EQ, content.getLanguage()),
                    new StringExpression("code", Operator.EQ, content.<String>getValue("code")),
                    new StringExpression("catalog", Operator.EQ, newCatalog)
                );
                String query = ContentQueryHelper.getContentXPathQuery(expression);
                
                if (_resolver.query(query).stream().findAny().isPresent())
                {
                    Map<String, I18nizableTextParameter> params = new HashMap<>();
                    params.put("contentType", _contentTypeHelper.getContentTypeLabel(content));
                    params.put("code",  new I18nizableText(content.<String>getValue("code")));
                    params.put("catalog", new I18nizableText(_catalogsManager.getCatalog(newCatalog).getTitle()));
                    params.put("lang", _languagesManager.getLanguage(content.getLanguage()).getLabel());
                    result.addError(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_UNICITY_VALIDATOR_ERROR", params));
                }
            }
        }
        
        return result;
    }
}
