/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree;
import org.ametys.runtime.model.View;

/**
 * Class to generate maquette extract.
 */
public abstract class AbstractMaquetteExtract extends AbstractExtract implements Initializable
{
    /** Unique instance of ProgramItemComparator */
    protected static final ProgramItemComparator PROGRAM_ITEM_COMPARATOR = new ProgramItemComparator();
    
    /** The maquette report view for courses */
    protected View _courseView;

    /** The maquette report view for courselists */
    protected View _courseListView;
    
    /** The maquette report view for containers */
    protected View _containerView;

    /** The maquette report view for subprograms */
    protected View _subProgramView;
    
    /** The maquette report view for program */
    protected View _programView;
    
    /** The content type EP */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _courseView = getView(CourseFactory.COURSE_CONTENT_TYPE, "report-maquette");
        _courseListView = getView(CourseListFactory.COURSE_LIST_CONTENT_TYPE, "report-maquette");
        _containerView = getView(ContainerFactory.CONTAINER_CONTENT_TYPE, "report-maquette");
        _subProgramView = getView(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, "report-maquette");
        _programView = getView(ProgramFactory.PROGRAM_CONTENT_TYPE, "report-maquette");
    }

    /**
     * Get the view from the content type.
     * @param contentTypeId The content type id
     * @param viewName The view name
     * @return The view
     */
    protected View getView(String contentTypeId, String viewName)
    {
        ContentType courseContentType = _contentTypeEP.getExtension(contentTypeId);
        return  courseContentType.getView(viewName);
    }

    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_DOC, OUTPUT_FORMAT_XLS);
    }

    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        // Check the type of the program item (currently, by default, only programs are accepted)
        ProgramItem programItem = _resolver.resolveById(programItemId);
        if (!(programItem instanceof Program program))
        {
            throw new UnsupportedOperationException("The report '" + getType() + "' can be launch only on programs.");
        }
        
        ProgramItemTree programTree = createTreeFromProgramItem(programItem, reportParameters);
        
        try
        {
            handler.startDocument();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType());
            XMLUtils.startElement(handler, "report", attrs);
            
            // SAX other global informations
            saxGlobalInformations(handler, programItem, reportParameters);

            _reportHelper.saxNaturesEnseignement(handler, getLogger());

            saxTree(handler, programTree);
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Maquette' extract for program '{}' ({})", program.getTitle(), program.getCode(), e);
        }
    }

    /**
     * Add and populate the program item to the {@link ProgramItemTree}
     * @param programItem The program item to add
     * @param reportParameters The report parameters
     * @return a tree of the program
     */
    protected ProgramItemTree createTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        ProgramItemTree programItemTree = new ProgramItemTree(programItem);
        populateTree(programItemTree);
        return programItemTree;
    }
    
    /**
     * Populate the MCC tree.
     * @param tree The MCC tree
     */
    protected abstract void populateTree(ProgramItemTree tree);
    
    /**
     * SAX the MCC tree for the report
     * @param handler The handler
     * @param tree The MCC tree
     * @throws SAXException if an error occurs
     */
    protected void saxTree(ContentHandler handler, ProgramItemTree tree) throws SAXException
    {
        ProgramItem programItem = tree.getCurrent();
        
        if (programItem instanceof Program program)
        {
            saxContent(handler, "program", program, _programView, tree, true);
        }
        else if (programItem instanceof SubProgram subProgram)
        {
            saxContent(handler, "subProgram", subProgram, _subProgramView, tree, true);
        }
        else if (programItem instanceof Container container)
        {
            saxContent(handler, "container", container, _containerView, tree, true);
        }
        else if (programItem instanceof CourseList courseList)
        {
            saxContent(handler, "courseList", courseList, _courseListView, tree, true);
        }
        else if (programItem instanceof Course course)
        {
            saxContent(handler, "course", course, _courseView, tree, true);
        }
    }
    
    /**
     * Sax a content on a view to the handler and sax children.
     * @param handler The handler
     * @param tagName The root tag name
     * @param content The content
     * @param view The view
     * @param tree The tree for children
     * @param saxHierarchy <code>true</code> to sax the hierarchy of the current content
     * @throws SAXException if an error occurs
     */
    protected void saxContent(ContentHandler handler, String tagName, Content content, View view, ProgramItemTree tree, boolean saxHierarchy) throws SAXException
    {
        AttributesImpl attr = getProgramItemAttributes((ProgramItem) content, tree);

        XMLUtils.startElement(handler, tagName, attr);
        
        content.dataToSAX(handler, view);
        
        if (saxHierarchy)
        {
            for (ProgramItemTree child : tree.getChildren())
            {
                saxTree(handler, child);
            }
        }
        
        XMLUtils.endElement(handler, tagName);
    }
    
    /**
     * Get program item's attribute to sax
     * @param programItem the current program item
     * @param tree the program item tree
     * @return the program item attribues to sax
     */
    protected AttributesImpl getProgramItemAttributes(ProgramItem programItem, ProgramItemTree tree)
    {
        Content content = (Content) programItem;
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("path", tree.getPath().toString());
        return attrs;
    }

    /**
     * Comparator for sorting by (subProgram | container) > courseList (mandatory > choice > optional)
     */
    protected static final class ProgramItemComparator implements Comparator<ProgramItem>
    {
        public int compare(ProgramItem p1, ProgramItem p2)
        {
            if (p1 instanceof CourseList && !(p2 instanceof CourseList))
            {
                // Order courselist after subprogram or container
                return 1;
            }
            else if (!(p1 instanceof CourseList) && p2 instanceof CourseList)
            {
                // Order courselist after subprogram or container
                return -1;
            }
            else if (p1 instanceof CourseList cl1 && p2 instanceof CourseList cl2)
            {
                // Order courselist according to their type (mandatory > choice > optional)
                ChoiceType t1 = cl1.getType();
                ChoiceType t2 = cl2.getType();
                
                if (t1.equals(t2))
                {
                    return 0;
                }
                else if (t1.equals(ChoiceType.MANDATORY) || !t2.equals(ChoiceType.MANDATORY) && t1.equals(ChoiceType.CHOICE))
                {
                    return -1;
                }
                else
                {
                    return 1;
                }
            }
            else
            {
                // No order between other program items (subprograms or semesters)
                return 0;
            }
        }
    }
    
    /**
     * Get the children of the current program item, course lists are at the end of the list (after containers)
     * @param programItem the current program item
     * @return the ordered children
     */
    protected List<ProgramItem> getOrderedChildren(ProgramItem programItem)
    {
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        // Order tree elements (course list last, ordered by their type)
        children.sort(PROGRAM_ITEM_COMPARATOR);
        return children;
    }
    
    /**
     * Sax the additional global informations of the report.
     * @param handler The handler
     * @param programItem The program item on which the report is launched
     * @param reportParameters The report parameters
     * @throws SAXException If an error occurs
     */
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        String catalog = programItem.getCatalog();
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", catalog);
        XMLUtils.createElement(handler, "catalog", attrs, _catalogsManager.getCatalog(catalog).getTitle());
        XMLUtils.createElement(handler, "title", ((Content) programItem).getTitle());
        XMLUtils.createElement(handler, "lang", programItem.getLanguage());
    }
}
