/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.odfpilotage.report.consistency.AnalysisExtensionPoint;
import org.ametys.plugins.odfpilotage.report.consistency.ConsistencyAnalysis;
import org.ametys.plugins.odfpilotage.report.consistency.ConsistencyAnalysisResult;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.OrgUnitConsistencyExtractSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.ProgramConsistencyExtractSchedulable;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Class to generate consistency extract as DOC.
 */
public class ConsistencyExtract extends AbstractExtract
{
    /** The key for the analysis */
    public static final String PARAMETER_ANALYSIS = "analysis";
    
    /** The analysis extension point. */
    protected AnalysisExtensionPoint _analysisEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _analysisEP = (AnalysisExtensionPoint) manager.lookup(AnalysisExtensionPoint.ROLE);
        super.service(manager);
    }
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        String type = "coherence";
        
        // If report parameters are given, add the short id of the analysis type or "all"
        if (!shortName && reportParameters != null)
        {
            type += "-";
            type += Optional.ofNullable(reportParameters.get(PARAMETER_ANALYSIS))
                .map(_analysisEP::getExtension)
                .map(ConsistencyAnalysis::getShortId)
                .orElse("all");
        }
        
        return type;
    }
    
    @Override
    protected boolean isGeneric()
    {
        return false;
    }

    @Override
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return schedulable instanceof OrgUnitConsistencyExtractSchedulable || schedulable instanceof ProgramConsistencyExtractSchedulable;
    }
    
    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        ProgramItem programItem = _resolver.resolveById(programItemId);
        
        List<ConsistencyAnalysis> analyses = Optional.ofNullable(reportParameters.get(PARAMETER_ANALYSIS))
            .map(_analysisEP::getExtension)
            .map(List::of)
            .orElseGet(() ->
                _analysisEP.getExtensionsIds()
                    .stream()
                    .map(_analysisEP::getExtension)
                    .toList()
            );
        
        try
        {
            handler.startDocument();

            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            
            XMLUtils.startElement(handler, "report", attrs);
            _saxReport(handler, programItem, analyses);
            XMLUtils.endElement(handler, "report");
            
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Cohérence' report for program '{}' ({})", ((Content) programItem).getTitle(), programItem.getCode(), e);
        }
    }
    
    private void _saxReport(ContentHandler handler, ProgramItem programItem, List<ConsistencyAnalysis> analyses) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("title", ((Content) programItem).getTitle());
        XMLUtils.startElement(handler, "programItem", attr);

        for (ConsistencyAnalysis analysis : analyses)
        {
            XMLUtils.startElement(handler, "analysis");
            
            // Title
            analysis.getLabel().toSAX(handler, "title");
            
            // Result
            ConsistencyAnalysisResult result = analysis.analyze(programItem);
            
            // Status
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("bgColor", result.getStatus().getBgColor());
            attrs.addCDATAAttribute("fontColor", result.getStatus().getFontColor());
            XMLUtils.startElement(handler, "status", attrs);
            result.getStatusText().toSAX(handler);
            XMLUtils.endElement(handler, "status");
            
            // Introduction
            result.getIntroText().toSAX(handler, "intro");
            
            // Columns
            XMLUtils.startElement(handler, "columns");
            Map<String, I18nizableText> columns = result.getColumns();
            List<String> indentableColumns = result.getIndentableColumns();
            for (String key : columns.keySet())
            {
                AttributesImpl columnAttrs = new AttributesImpl();
                columnAttrs.addCDATAAttribute("name", key);
                columnAttrs.addCDATAAttribute("indentable", String.valueOf(indentableColumns.contains(key)));
                
                XMLUtils.startElement(handler, "column", columnAttrs);
                columns.get(key).toSAX(handler);
                XMLUtils.endElement(handler, "column");
            }
            XMLUtils.endElement(handler, "columns");
            
            // Lines list
            XMLUtils.startElement(handler, "lines");
            for (Map<String, Object> line : result.getLines())
            {
                XMLUtils.startElement(handler, "line");
                for (Map.Entry<String, Object> entry : line.entrySet())
                {
                    if (entry.getValue() instanceof I18nizableText text)
                    {
                        text.toSAX(handler, entry.getKey());
                    }
                    else
                    {
                        XMLUtils.createElement(handler, entry.getKey(), entry.getValue().toString());
                    }
                }
                XMLUtils.endElement(handler, "line");
            }
            
            

            XMLUtils.endElement(handler, "lines");
            
            XMLUtils.endElement(handler, "analysis");
        }

        XMLUtils.endElement(handler, "programItem");
    }
}
