/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

import com.google.common.collect.ImmutableMap;

/**
 * Class to generate the volume horaire report.
 */
public class VolumeHoraireReport extends AbstractReport
{
    private static final Map<ChoiceType, String> __COURSELIST_TYPE_2_LABEL = ImmutableMap.of(
        ChoiceType.MANDATORY, "Obligatoire",
        ChoiceType.OPTIONAL, "Facultatif",
        ChoiceType.CHOICE, "A choix"
    );

    private String _natureSemester;
    private String _natureYear;
    private String _natureUE;
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        return "volumehoraire";
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_XLS;
    }
    
    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_XLS);
    }

    @Override
    protected void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        _natureYear = _odfHelper.getYearId().orElse(null);
        _natureSemester = Optional.of(_refTableHelper.getItemFromCode(OdfReferenceTableHelper.CONTAINER_NATURE, "semestre")).map(OdfReferenceTableEntry::getId).orElse(null);
        _natureUE = Optional.ofNullable(_refTableHelper.getItemFromCode(OdfReferenceTableHelper.COURSE_NATURE, "UE")).map(OdfReferenceTableEntry::getId).orElse(null);
        
        OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
        List<Program> selectedPrograms = _odfHelper.getProgramsFromOrgUnit(orgUnit, catalog, lang);
        
        // Initialize data
        Map<String, Map<String, String>> calculatedElps = _volumeHoraire(selectedPrograms);
        
        try
        {
            handler.startDocument();
        
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            XMLUtils.startElement(handler, "report", attrs);
            
            // SAX dynamic informations
            _reportHelper.saxNaturesEnseignement(handler, getLogger());

            XMLUtils.startElement(handler, "lines");
            for (Program program : selectedPrograms)
            {
                _saxUEsForProgram(handler, program, calculatedElps);
            }
            XMLUtils.endElement(handler, "lines");
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Volume horaire' report for orgunit '{}'", orgUnit.getTitle(), e);
        }
    }

    /**
     * Processing of the hourly volume for each UE.
     * @param selectedPrograms The programs to explore
     */
    private Map<String, Map<String, String>> _volumeHoraire(List<Program> selectedPrograms)
    {
        Map<String, Map<String, String>> calculatedElps = new HashMap<>();
        
        // Descendre au niveau des courses de type UE
        Set<Course> courses = _getUEsFromPrograms(selectedPrograms);
        
        // Calculer pour chaque ELP
        for (Course course : courses)
        {
            String coursePrefix = course.getTitle();
            
            // Calcul des volumes pour l'UE
            getLogger().info("[{}] Calcul des volumes horaires...", coursePrefix);
            Map<String, Pair<Double, Double>> volumesByNature = _calculVolumeByEnseignement(course, 1);
            
            Map<String, String> ueData = new HashMap<>();
            
            // Période et type de période
            ContentValue periodValue = course.getValue("period");
            if (periodValue != null)
            {
                try
                {
                    Content period = periodValue.getContent();
                    String periodType = Optional.ofNullable(period.<ContentValue>getValue("type"))
                        .flatMap(ContentValue::getContentIfExists)
                        .map(OdfReferenceTableEntry::new)
                        .map(OdfReferenceTableEntry::getCode)
                        .orElse(StringUtils.EMPTY);
                    
                    ueData.put("periode", period.getTitle());
                    ueData.put("typePeriode", periodType);
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().error("Impossible de retrouver la période : {}", periodValue, e);
                }
            }
            
            ueData.put("code", course.getDisplayCode());
            ueData.put("codeELP", course.getValue("elpCode", false, StringUtils.EMPTY));
            ueData.put("shortLabel", course.getValue("shortLabel", false, StringUtils.EMPTY));
            ueData.put("title", course.getTitle());
            Double ects = course.getValue("ects");
            ueData.put("ects", ects == null ? StringUtils.EMPTY : String.valueOf(ects));

            // Mutualisation
            Set<Container> steps = _pilotageHelper.getSteps(course);
            ueData.put("isShared", steps.size() > 1 ? "X" : StringUtils.EMPTY);
            
            Pair<StepHolderStatus, Container> stepHolder = _pilotageHelper.getStepHolder(course);
            if (stepHolder.getKey().equals(StepHolderStatus.SINGLE))
            {
                ueData.put("stepHolder", _getStepHolder(stepHolder.getValue()));
            }
            
            for (String nature : volumesByNature.keySet())
            {
                Pair<Double, Double> volumes = volumesByNature.get(nature);
                ueData.put("nbHours#" + nature + "#average", String.valueOf(volumes.getLeft()));
                ueData.put("nbHours#" + nature + "#total", String.valueOf(volumes.getRight()));
            }
            
            calculatedElps.put(course.getId(), ueData);
        }
        
        return calculatedElps;
    }
    
    private String _getStepHolder(Container stepHolder)
    {
        StringBuilder stepHolderAsString = new StringBuilder(stepHolder.getDisplayCode())
                .append("#");
        
        List<String> etpCodes = new ArrayList<>();
        Optional.ofNullable(stepHolder.getValue("etpCode"))
                .map(String.class::cast)
                .filter(StringUtils::isNotEmpty)
                .ifPresent(etpCodes::add);
        Optional.ofNullable(stepHolder.getValue("vrsEtpCode"))
                .map(String.class::cast)
                .filter(StringUtils::isNotEmpty)
                .ifPresent(etpCodes::add);
        
        if (!etpCodes.isEmpty())
        {
            // Add the ETP codes as prefixes for the title
            stepHolderAsString.append("[")
                .append(StringUtils.join(etpCodes, "-"))
                .append("] ");
        }
        
        stepHolderAsString.append(stepHolder.getTitle());
        return stepHolderAsString.toString();
    }
    
    private Set<Course> _getUEsFromPrograms(List<Program> selectedPrograms)
    {
        Set<Course> courses = new LinkedHashSet<>();
        
        if (_natureUE != null)
        {
            selectedPrograms.forEach(program -> courses.addAll(_getCoursesForProgramItem(program)));
        }
        
        return courses;
    }
    
    private List<Course> _getCoursesForProgramItem(ProgramItem programItem)
    {
        List<Course> courses = new ArrayList<>();
        
        if (programItem instanceof Course course && _natureUE.equals(course.getCourseType()))
        {
            courses.add(course);
        }
        else
        {
            _odfHelper.getChildProgramItems(programItem).forEach(child -> courses.addAll(_getCoursesForProgramItem(child)));
        }
        
        return courses;
    }

    private Map<String, Pair<Double, Double>> _calculVolumeByEnseignement(CourseList courseList, float initialWeight, Map<String, Pair<Double, Double>> volumesByNature)
    {
        Map<String, Pair<Double, Double>> volumes = volumesByNature;
        
        ChoiceType courseListType = courseList.getType();
        if (courseListType == null)
        {
            getLogger().error("The list '{}' ({}) doesn't have a valid type.", courseList.getTitle(), courseList.getCode());
        }
        else if (!courseListType.equals(ChoiceType.OPTIONAL) && courseList.hasCourses())
        {
            float weight = initialWeight * _getWeight(courseList);
            if (weight > 0)
            {
                for (Course course : courseList.getCourses())
                {
                    Map<String, Pair<Double, Double>> courseVolumes = _calculVolumeByEnseignement(course, weight);
                    for (String nature : courseVolumes.keySet())
                    {
                        Pair<Double, Double> courseVolume = courseVolumes.getOrDefault(nature, Pair.of(0.0, 0.0));
                        Pair<Double, Double> volume = volumes.getOrDefault(nature, Pair.of(0.0, 0.0));
                        volumes.put(nature, Pair.of(courseVolume.getLeft() + volume.getLeft(), courseVolume.getRight() + volume.getRight()));
                    }
                }
            }
        }
        
        return volumes;
    }
    
    private float _getWeight(CourseList courseList)
    {
        ChoiceType courseListType = courseList.getType();

        if (!courseListType.equals(ChoiceType.OPTIONAL) && courseList.hasCourses())
        {
            if (courseListType.equals(ChoiceType.MANDATORY))
            {
                return 1.0f;
            }
            
            if (courseListType.equals(ChoiceType.CHOICE))
            {
                return courseList.getMinNumberOfCourses() / (float) courseList.getCourses().size();
            }
        }
        
        return 0.0f;
    }
    
    private Map<String, Pair<Double, Double>> _calculVolumeByEnseignement(Course course, float weight)
    {
        Map<String, Pair<Double, Double>> volumesByNature = new HashMap<>();
        
        if (course.hasCourseLists())
        {
            // Parcours de toutes les UEs en dessous
            for (CourseList courseList : course.getCourseLists())
            {
                volumesByNature = _calculVolumeByEnseignement(courseList, weight, volumesByNature);
            }
        }
        else
        {
            // Calcul
            Map<String, Double> volumes = new HashMap<>();
            for (CoursePart coursePart : course.getCourseParts())
            {
                String nature = coursePart.getNature();
                double nbHours = coursePart.getNumberOfHours();
                volumes.put(nature, volumes.getOrDefault(nature, 0.0) + nbHours);
            }
            
            for (String nature : volumes.keySet())
            {
                Double nbHours = volumes.get(nature);
                volumesByNature.put(nature, Pair.of(nbHours * weight, nbHours));
            }
        }
        
        return volumesByNature;
    }
    
    private void _saxUEsForProgram(ContentHandler handler, Program program, Map<String, Map<String, String>> calculatedElps) throws SAXException
    {
        _saxUEsWithStructure(handler, program, new HashMap<>(), calculatedElps, 1.0f);
    }
    
    private void _saxUEsWithStructure(ContentHandler handler, ProgramItem programItem, Map<String, String> structureData, Map<String, Map<String, String>> calculatedElps, float initialWeight) throws SAXException
    {
        float weight = initialWeight;
        
        Map<String, String> currentStructureData = new HashMap<>(structureData);
        String title = ((Content) programItem).getTitle();
        
        if (programItem instanceof Program program)
        {
            currentStructureData.put("program", title);
            String degree = _refTableHelper.getItemLabel(program.getDegree(), program.getLanguage());
            currentStructureData.put("degree", degree);
        }
        else if (programItem instanceof SubProgram)
        {
            currentStructureData.put("parcours", title);
        }
        else if (programItem instanceof Container container)
        {
            String containerNature = container.getNature();
            if (containerNature.equals(_natureSemester))
            {
                currentStructureData.put("semestre", title);
            }
            else if (containerNature.equals(_natureYear))
            {
                currentStructureData.put("annee", title);
                currentStructureData.put("etpCode", container.getValue("etpCode", false, StringUtils.EMPTY));
                currentStructureData.put("vrsEtpCode", container.getValue("vrsEtpCode", false, StringUtils.EMPTY));
            }
        }
        else if (programItem instanceof CourseList courseList)
        {
            if (!currentStructureData.containsKey("listType"))
            {
                ChoiceType courseListType = courseList.getType();
                if (courseListType == null)
                {
                    getLogger().error("The course list '{}' hasn't a type.", title);
                }
                else
                {
                    String courseListTypeTraduction = __COURSELIST_TYPE_2_LABEL.get(courseListType);
                    if (courseListTypeTraduction == null)
                    {
                        getLogger().error("Invalid course list type '{}' for '{}'.", courseListType, title);
                    }
                    else
                    {
                        currentStructureData.put("listType", courseListTypeTraduction);
                    }
                    
                    weight *= _getWeight(courseList);
                }
            }
        }
        else if (programItem instanceof Course course)
        {
            if (!currentStructureData.containsKey("CodeAnu"))
            {
                String codeAnu = Optional.ofNullable(course.getValue("CodeAnu"))
                                         .map(codeAsLong -> String.valueOf(codeAsLong))
                                         .orElse(StringUtils.EMPTY);
                currentStructureData.put("CodeAnu", codeAnu);
            }
            
            Map<String, String> ueData = calculatedElps.get(course.getId());
            if (ueData != null)
            {
                _saxUE(handler, currentStructureData, ueData, weight);
            }
        }
        
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            _saxUEsWithStructure(handler, child, currentStructureData, calculatedElps, weight);
        }
    }
    
    private void _saxUE(ContentHandler handler, Map<String, String> structureData, Map<String, String> ueData, float weight) throws SAXException
    {
        XMLUtils.startElement(handler, "line");
        
        // Sax structure data
        for (Map.Entry<String, String> cell : structureData.entrySet())
        {
            XMLUtils.createElement(handler, cell.getKey(), cell.getValue());
        }
        
        // Sax UE data
        for (Map.Entry<String, String> cell : ueData.entrySet())
        {
            String key = cell.getKey();
            if ("stepHolder".equals(key))
            {
                _stepHolderToSAX(handler, cell.getValue());
            }
            else if (key.startsWith("nbHours#"))
            {
                AttributesImpl cellAttrs = new AttributesImpl();
                String[] tokens = key.split("#");
                key = tokens[0];
                cellAttrs.addCDATAAttribute("nature", tokens[1]);
                cellAttrs.addCDATAAttribute("type", tokens[2]);
                
                // Ponderate if average nb hours
                String value = tokens[2].equals("average") ? String.valueOf(Double.parseDouble(cell.getValue()) * weight) : cell.getValue();
                
                XMLUtils.createElement(handler, key, cellAttrs, value);
            }
            else
            {
                XMLUtils.createElement(handler, key, cell.getValue());
            }
        }
        
        XMLUtils.endElement(handler, "line");
    }
    
    private void _stepHolderToSAX(ContentHandler handler, String stepAsString) throws SAXException
    {
        XMLUtils.startElement(handler, "stepHolder");
        
        String[] tokens = stepAsString.split("#");
        XMLUtils.createElement(handler, "code", tokens[0]);
        XMLUtils.createElement(handler, "title", tokens[1]);

        XMLUtils.endElement(handler, "stepHolder");
    }
}
