/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportTarget;
import org.ametys.plugins.odfpilotage.report.ReportExtensionPoint;

/**
 * Implementation for Pilotage search tools
 */
public class PilotageSearchClientSideElement extends StaticClientSideElement
{
    private static final String CRITERIA_ENUMERATION_VALUE_FIELD = "value";
    private static final String CRITERIA_ENUMERATION_LABEL_FIELD = "label";
    
    /** The report extension point */
    protected ReportExtensionPoint _reportEP;
    
    /** The catalog manager */
    protected CatalogsManager _catalogManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _reportEP = (ReportExtensionPoint) manager.lookup(ReportExtensionPoint.ROLE);
        _catalogManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    /**
     * Get the search model criteria configuration
     * @return The search model criteria configuration
     */
    @Callable (rights = "ODF_Rights_Pilotage_Tool")
    public Map<String, Object> getSearchModelCriteriaConfiguration()
    {
        Map<String, Object> enumerations = new LinkedHashMap<>();
        
        enumerations.put("reportTypes", _getReportTypeEnumeration());
        enumerations.put("targets", _getTargetEnumeration());
        enumerations.put("catalogs", _getCatalogEnumeration());
        
        return enumerations;
    }
    
    private List<Map<String, Object>> _getReportTypeEnumeration()
    {
        return _reportEP.getExtensionsIds()
                .stream()
                .map(_reportEP::getExtension)
                .map(extension -> Map.of(CRITERIA_ENUMERATION_VALUE_FIELD, extension.getId(), CRITERIA_ENUMERATION_LABEL_FIELD, extension.getLabel()))
                .toList();
    }
    
    private List<Map<String, Object>> _getTargetEnumeration()
    {
        List<Map<String, Object>> targetsLabels = new ArrayList<>();
        PilotageReportTarget[] targets = PilotageReportTarget.values();
        for (PilotageReportTarget target : targets)
        {
            targetsLabels.add(Map.of(CRITERIA_ENUMERATION_VALUE_FIELD, target.name(), CRITERIA_ENUMERATION_LABEL_FIELD, target.getLabel()));
        }
        
        return targetsLabels;
    }
    
    private List<Map<String, String>> _getCatalogEnumeration()
    {
        return _catalogManager.getCatalogs().stream()
                                            .map(catalog -> Map.of(CRITERIA_ENUMERATION_VALUE_FIELD, catalog.getName(), CRITERIA_ENUMERATION_LABEL_FIELD, catalog.getTitle()))
                                            .toList();
    }
}
