/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.ThematicsHelper;

/**
 * Observer to remove derogation and additional rule if the container restriction are modified.
 */
public class UpdateContainerRulesStep2Observer extends AbstractRulesStepObserver
{
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;

    /** The rules manager */
    protected RulesManager _rulesManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _thematicsHelper = (ThematicsHelper) manager.lookup(ThematicsHelper.ROLE);
        _rulesManager = (RulesManager) manager.lookup(RulesManager.ROLE);
    }
    
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_MODIFIED;
    }
    
    @Override
    protected boolean supportsContent(Content content)
    {
        return _odfHelper.isContainerOfTypeYear(content);
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Container content = (Container) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        String regimeId = Optional.ofNullable(content.<ContentValue>getValue(PilotageHelper.CONTAINER_MCC_REGIME))
                .map(ContentValue::getContentId)
                .orElse(null);
        String oldRegimeId = _getOldRegimeIds(content).stream().findFirst().orElse(null);
        
        String nbSessions = content.getValue(PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS);
        String oldNbSessions = _getOldNumberOfSessions(content);
        
        // Only process if the mccRegime or mccNbSessions have been modified
        if (!StringUtils.equals(regimeId, oldRegimeId) || !StringUtils.equals(StringUtils.defaultString(nbSessions), StringUtils.defaultString(oldNbSessions)))
        {
            _deleteOldRulesOnContainer(content);
        }
    }

    /**
     * Delete old rules on container
     * @param content the container
     */
    protected void _deleteOldRulesOnContainer(Container content)
    {
        // Get all compatible thematics with the given container
        List<String> compatibleThematicCode = _thematicsHelper.getCompatibleThematics(content)
            .stream()
            .map(t -> t.<String>getValue(RulesManager.THEMATIC_CODE))
            .toList();
        
        // Get the rules not compatible anymore with the container (positions in repeater)
        Set<Integer> incompatibleRulesPosition = Optional.ofNullable(content)
                .map(c -> c.<ModifiableIndexableRepeater>getValue(RulesManager.CONTAINER_RULES))
                .map(ModifiableIndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .filter(e -> !this._matchRules(e, compatibleThematicCode))
                .map(ModifiableIndexableRepeaterEntry::getPosition)
                .collect(Collectors.toSet());

        // Remove incompatible rules
        _rulesManager.deleteRules(content, incompatibleRulesPosition);
    }
    
    private boolean _matchRules(ModifiableIndexableRepeaterEntry e, List<String> compatibleThematicCode)
    {
        // If it is an additional rule, the thematic code is filled
        if (e.hasValue(RulesManager.RULE_THEMATIC_CODE))
        {
            return compatibleThematicCode.contains(e.getValue(RulesManager.RULE_THEMATIC_CODE));
        }
        
        // If it is a derogated rule, the rule code starts with its thematic code then a hyphen ([ThematicCode]-[Number])
        String code = e.getValue(RulesManager.RULE_CODE);
        return compatibleThematicCode.stream()
                .map(c -> c + "-")
                .anyMatch(code::startsWith);
    }
}
