/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.cost.entity;

import java.util.HashMap;
import java.util.Map;

/**
 * This class store all overridden data by the user
 */
public class OverriddenData
{
    /** Map of overridden volume of hours */
    private Map<String, Double> _volumeOfHours;
    
    /** Map of overridden number of groups */
    private Map<String, Long> _groups;
    
    /** Map of overridden effectives */
    private Map<String, Long> _effectives;
    
    
    /** 
     * The constructor
     * @param overriddenData overridden data by the user
     */
    public OverriddenData(Map<String, Map<String, String>> overriddenData)
    {
        _volumeOfHours = new HashMap<>();
        _groups = new HashMap<>();
        _effectives = new HashMap<>();
        
        for (Map.Entry<String, Map<String, String>> json : overriddenData.entrySet())
        {
            String contentId = json.getKey();
            Map<String, String> overriddenValues = json.getValue();
            for (String key : overriddenValues.keySet())
            {
                switch (key)
                {
                    case "effectivesGlobal":
                        _effectives.put(contentId, Long.parseLong(overriddenValues.get(key)));
                        break;
                    case "groups":
                        _groups.put(contentId, Long.parseLong(overriddenValues.get(key)));
                        break;
                    case "nbHours":
                        _volumeOfHours.put(contentId, Double.parseDouble(overriddenValues.get(key)));
                        break;
                    default:
                        // Ignore
                        break;
                }
            }
        }
    }
    
    /** 
     * The constructor
     */
    public OverriddenData()
    {
        this(Map.of());
    }
    
    /**
     * Get the overridden volume of hours of a program item
     * @param programItemId the program item id
     * @return the overridden volume of hours
     */
    public Double getVolumeOfHours(String programItemId)
    {
        return _volumeOfHours.get(programItemId);
    } 
    
    /**
     * Get the overridden effective of a program item
     * @param programItemId the program item id
     * @return the overridden effective
     */
    public Long getEffective(String programItemId)
    {
        return _effectives.get(programItemId);
    } 
    
    /**
     * Get the overridden number groups of a program item
     * @param programItemId the program item id
     * @return the overridden number of groups
     */
    public Long getNumberOfGroups(String programItemId)
    {
        return _groups.get(programItemId);
    }
}
