/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.dashboard;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.search.SearchResults;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ContentQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper.PilotageStatus;
import org.ametys.plugins.odfpilotage.property.ContainerMCCWorkflowStatusProperty;
import org.ametys.plugins.odfpilotage.property.InvalidContainerDataProperty;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.ThematicsHelper;
import org.ametys.plugins.odfpilotage.rule.enumerators.NbSessionsEnumerator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The pilotage dashboard helper
 */
public class PilotageDashboardHelper extends AbstractLogEnabled implements Component, Serviceable
{
    private static final DecimalFormat _DECIMAL_FORMAT = new DecimalFormat("#.#");

    private static final String _PILOTAGE_DASHBOARD_RIGHT = "ODF_Rights_Pilotage_Dashboard";

    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The content searcher factory */
    protected ContentSearcherFactory _contentSearcherFactory;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) smanager.lookup(ContentSearcherFactory.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
        _thematicsHelper = (ThematicsHelper) smanager.lookup(ThematicsHelper.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
        _pilotageHelper = (PilotageHelper) smanager.lookup(PilotageHelper.ROLE);
    }
    
    private boolean _checkRightOnOrgunit(OrgUnit orgUnit)
    {
        if (_rightManager.currentUserHasRight(_PILOTAGE_DASHBOARD_RIGHT, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            return true;
        }
        
        return orgUnit != null ? _rightManager.currentUserHasRight(_PILOTAGE_DASHBOARD_RIGHT, orgUnit) == RightResult.RIGHT_ALLOW : false;
    }
    
    /**
     * Get the dashboard information
     * isRulesEnabled: <code>true</code> if rules are enabled
     * yearId: the year id
     * @return the map of dashboard information
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getDashboardInfo()
    {
        Optional<String> yearId = _odfHelper.getYearId();
        if (yearId.isPresent())
        {
            return Map.of(
                "isRulesEnabled", RulesManager.isRulesEnabled(),
                "yearId", yearId.get()
            );
        }
        
        return Map.of("error", "no-year");
    }
    
    /**
     * Get orgUnits when the current user has right 'ODF_Rights_Pilotage_Dashboard'
     * @param params the params
     * @return the list of orgUnits
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getOrgUnits(Map<String, Object> params)
    {
        List<Map<String, Object>> orgUnitsAsMap = new ArrayList<>();
        // Add all options if the current user is a super amdin with the right 'ODF_Rights_Pilotage_Dashboard' on general context
        if (_rightManager.currentUserHasRight(_PILOTAGE_DASHBOARD_RIGHT, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            orgUnitsAsMap.add(Map.of(
                    "id", " ",
                    "label", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_ORGUNITS_ALL")
            ));
        }
        
        Expression expr = new ContentTypeExpression(org.ametys.plugins.repository.query.expression.Expression.Operator.EQ, OrgUnitFactory.ORGUNIT_CONTENT_TYPE);
        String xPathQuery = QueryHelper.getXPathQuery(null, OrgUnitFactory.ORGUNIT_NODETYPE, expr);
        List<OrgUnit> orgUnits = _resolver.query(xPathQuery)
                .stream()
                .filter(OrgUnit.class::isInstance)
                .map(OrgUnit.class::cast)
                .filter(this::_checkRightOnOrgunit)
                .toList();
        for (OrgUnit orgUnit : orgUnits)
        {
            orgUnitsAsMap.add(Map.of(
                    "id", orgUnit.getId(),
                    "label", orgUnit.getTitle()
            ));
        }
        
        return Map.of("orgUnits", orgUnitsAsMap);
    }
    
    /**
     * Get the rules statistics
     * @param catalog the catalog
     * @param orgUnitId the orgUnit id. Can be null or empty
     * @return the statistics for the derogated thematics
     * @throws IllegalAccessException if an illegal access error occurred
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getRulesStats(String catalog, String orgUnitId) throws IllegalAccessException
    {
        OrgUnit orgUnit = StringUtils.isNotBlank(orgUnitId) ? _resolver.resolveById(orgUnitId) : null;
        if (!_checkRightOnOrgunit(orgUnit))
        {
            throw new IllegalAccessException("Unable to get stats on orgUnit with id '" + orgUnitId + "' because user has no sufficient right");
        }
        
        List<Map<String, Object>> stats = new ArrayList<>();
        if (_odfHelper.getYearId().isEmpty())
        {
            return stats;
        }
        
        try
        {
            SearchResults<Content> search = _contentSearcherFactory.create(ContainerFactory.CONTAINER_CONTENT_TYPE)
                .withLimits(0, Integer.MAX_VALUE)
                .withFacets(List.of("overridenThematics"))
                .setCheckRights(false)
                .searchWithFacets(_getFilterQuery(ContainerFactory.CONTAINER_CONTENT_TYPE, catalog, orgUnitId));
            
            long total = search.getTotalCount();
            
            Map<String, Map<String, Integer>> facetResults = search.getFacetResults();
            Map<String, Integer> overridenThematics = facetResults.get("overridenThematics");
            List<Entry<String, Integer>> sortedThematics = overridenThematics.entrySet()
                .stream()
                .sorted((e1, e2) -> Integer.compare(e2.getValue(), e1.getValue()))
                .toList();
                
            boolean hasRight = _thematicsHelper.hasHandleThematicRight();
            NbSessionsEnumerator nbSessionsEnumerator = new NbSessionsEnumerator();
            for (Entry<String, Integer> entry : sortedThematics)
            {
                Content thematic = _resolver.resolveById(entry.getKey());
                stats.add(_thematic2JSON(
                    thematic,
                    entry.getValue(),
                    total,
                    nbSessionsEnumerator,
                    hasRight)
                );
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred requesting solr", e);
        }
        
        return stats;
    }
    
    private Map<String, Object> _thematic2JSON(Content thematic, Integer number, Long total, NbSessionsEnumerator nbSessionsEnumerator, boolean hasRight) throws Exception
    {
        String regimesAsString = Stream.of(thematic.getValue("regime", false, new ContentValue[0]))
                .map(c -> _getRegimeLabel(c))
                .collect(Collectors.joining(", "));
            
        String nbSessionsValue = thematic.getValue("nbSessions");
        I18nizableText nbSessions = StringUtils.isNotBlank(nbSessionsValue) ? nbSessionsEnumerator.getEntry(nbSessionsValue) : null;
        
        Map<String, Object> thematicAsJson = new HashMap<>();
        thematicAsJson.put("thematic", thematic.getId());
        thematicAsJson.put("thematicTitle", thematic.getTitle());
        thematicAsJson.put("number", number);
        thematicAsJson.put("pourcent", total != 0L ? _DECIMAL_FORMAT.format(number.floatValue() * 100 / total) : 0.0);
        thematicAsJson.put("regime", regimesAsString);
        thematicAsJson.put("nbSessions", nbSessions);
        thematicAsJson.put("total", total);
        thematicAsJson.put("hasRight", hasRight);
        
        return thematicAsJson;
    }
    
    private String _getRegimeLabel(ContentValue contentValue)
    {
        if (contentValue.hasValue("shortLabel"))
        {
            return contentValue.getValue("shortLabel");
        }
        
        return contentValue.getContent().getTitle();
    }
    
    /**
     * Get the pilotage workflow statistics
     * @param catalog the catalog
     * @param orgUnitId the orgUnit id. Can be null or empty
     * @return the statistics for each pilotage workflow status
     * @throws IllegalAccessException if an illegal access error occurred
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getPilotageWorkflowStats(String catalog, String orgUnitId) throws IllegalAccessException
    {
        OrgUnit orgUnit = StringUtils.isNotBlank(orgUnitId) ? _resolver.resolveById(orgUnitId) : null;
        if (!_checkRightOnOrgunit(orgUnit))
        {
            throw new IllegalAccessException("Unable to get stats on orgUnit with id '" + orgUnitId + "' because user has no sufficient right");
        }
        
        List<Map<String, Object>> stats = new ArrayList<>();
        
        SearchResults<Content> results = _searchWithFacets(ProgramFactory.PROGRAM_CONTENT_TYPE, catalog, orgUnitId, List.of("mccPilotageStatus"));
        SearchResult programSearchResult = _getFacetResult(results, "mccPilotageStatus");
        stats.add(_status2JSON(
                "program",
                "mccPilotageStatus",
                programSearchResult,
                PilotageStatus.NONE.name(),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_NONE_LABEL")
        ));
        
        stats.add(_status2JSON(
                "program",
                "mccPilotageStatus",
                programSearchResult,
                PilotageStatus.MENTION_VALIDATED.name(),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_MENTION_LABEL")
        ));
        
        stats.add(_status2JSON(
                "program",
                "mccPilotageStatus",
                programSearchResult,
                PilotageStatus.ORGUNIT_VALIDATED.name(),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_ORGUNIT_LABEL")
        ));
        
        stats.add(_status2JSON(
                "program",
                "mccPilotageStatus",
                programSearchResult,
                PilotageStatus.CFVU_VALIDATED.name(),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_CFVU_LABEL")
        ));
        
        return stats;
    }
    
    /**
     * Get the pilotage MCC workflow statistics
     * @param catalog the catalog
     * @param orgUnitId the orgUnit id. Can be null or empty
     * @return the statistics for each pilotage MCC workflow status
     * @throws IllegalAccessException if an illegal access error occurred
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getPilotageMCCWorkflowStats(String catalog, String orgUnitId) throws IllegalAccessException
    {
        OrgUnit orgUnit = StringUtils.isNotBlank(orgUnitId) ? _resolver.resolveById(orgUnitId) : null;
        if (!_checkRightOnOrgunit(orgUnit))
        {
            throw new IllegalAccessException("Unable to get stats on orgUnit with id '" + orgUnitId + "' because user has no sufficient right");
        }
        
        List<Map<String, Object>> stats = new ArrayList<>();
        if (_odfHelper.getYearId().isEmpty())
        {
            return stats;
        }
        
        SearchResults<Content> results = _searchWithFacets(ContainerFactory.CONTAINER_CONTENT_TYPE, catalog, orgUnitId, List.of(ContainerMCCWorkflowStatusProperty.PROPERTY_NAME, InvalidContainerDataProperty.PROPERTY_NAME));
        SearchResult containerSearchResult = _getFacetResult(results, ContainerMCCWorkflowStatusProperty.PROPERTY_NAME);
        
        if (_pilotageHelper.getMCCRegimePolicy().equals("FORCE"))
        {
            SearchResult invalidDataSearchResult = _getFacetResult(results, InvalidContainerDataProperty.PROPERTY_NAME);
            stats.add(_status2JSON(
                "container",
                InvalidContainerDataProperty.PROPERTY_NAME,
                invalidDataSearchResult,
                "true",
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_CONTAINER_INVALID_DATA")
            ));
        }
        
        Arrays.stream(MCCWorkflowStep.values())
            .filter(step -> !step.equals(MCCWorkflowStep.RULES_VALIDATED) || RulesManager.isRulesEnabled())
            .forEach(step -> {
                stats.add(_status2JSON(
                        "container",
                        ContainerMCCWorkflowStatusProperty.PROPERTY_NAME,
                        containerSearchResult,
                        step.name(),
                        step.label()
                ));
            });
        
        return stats;
    }

    private SearchResults<Content> _searchWithFacets(String contentType, String catalog, String orgUnitId, List<String> facetNames)
    {
        try
        {
            return _contentSearcherFactory.create(contentType)
                    .withLimits(0, Integer.MAX_VALUE)
                    .withFacets(facetNames)
                    .setCheckRights(false)
                    .searchWithFacets(_getFilterQuery(contentType, catalog, orgUnitId));
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred requesting solr", e);
        }
        
        return null;
    }
    
    private SearchResult _getFacetResult(SearchResults<Content> search, String facetName)
    {
        Map<String, Map<String, Integer>> facetResults = search.getFacetResults();
        Map<String, Integer> mccWorkflowStatus = facetResults.get(facetName);
        
        long total = search.getTotalCount();

        return new SearchResult(mccWorkflowStatus, total);
    }
    
    private Map<String, Object> _status2JSON(String type, String attributName, SearchResult searchResult, String status, I18nizableText label)
    {
        Long total = searchResult.totalCount();
        Map<String, Integer> facetResults = searchResult.facetResults();
        Integer number = facetResults.getOrDefault(status, 0);
        return Map.of(
                "label", label,
                "number", number,
                "pourcent", total != 0 ? _DECIMAL_FORMAT.format(number.floatValue() * 100 / total) : 0.0,
                "status", status,
                "total", total,
                "type", type,
                "attributName", attributName
        );
    }
    
    private Query _getFilterQuery(String contentType, String catalog, String orgUnitId)
    {
        List<Query> queries = new ArrayList<>();
        
        queries.add(new StringQuery("catalog", catalog));
        if (StringUtils.isNotBlank(orgUnitId))
        {
            queries.add(new ContentQuery("orgUnit", Operator.EQ, orgUnitId, _resolver, _contentHelper));
        }
        
        if (ContainerFactory.CONTAINER_CONTENT_TYPE.equals(contentType))
        {
            queries.add(new StringQuery(Container.NATURE, _odfHelper.getYearId().get()));
        }
        
        return new AndQuery(queries);
        
    }
    
    private record SearchResult(Map<String, Integer> facetResults, Long totalCount) { /** */ }
}
