/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.type.DataContext;

/**
 * Abstract generator for thematics content.
 */
public abstract class AbstractThematicsGenerator extends ServiceableGenerator
{
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfRefTableHelper = (OdfReferenceTableHelper) smanager.lookup(OdfReferenceTableHelper.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _catalogsManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
        _thematicsHelper = (ThematicsHelper) smanager.lookup(ThematicsHelper.ROLE);
    }
    
    /**
     * Sax a content.
     * @param content the content
     * @throws SAXException if an error occurs
     */
    protected void _saxContent(Content content) throws SAXException
    {
        ViewItemAccessor resultItems =  _contentTypeExtensionPoint.getExtension(RulesManager.THEMATIC_CONTENT_TYPE).getView("main");
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("title", content.getTitle());
        if (content.getLanguage() != null)
        {
            attrs.addCDATAAttribute("language", content.getLanguage());
        }
        
        XMLUtils.startElement(contentHandler, "content", attrs);
        
        DataContext context = DataContext.newInstance()
                                         .withEmptyValues(false);
        content.dataToSAX(contentHandler, resultItems, context);
        
        XMLUtils.endElement(contentHandler, "content");
    }
    
    /**
     * Sax catalog with id and title.
     * @param catalogId The catalog id
     * @throws SAXException if an error occurs
     */
    protected void _saxCatalog(String catalogId) throws SAXException
    {
        if (StringUtils.isNotEmpty(catalogId))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("value", catalogId);
            XMLUtils.createElement(contentHandler, "catalog", attrs, _catalogsManager.getCatalog(catalogId).getTitle());
        }
    }
}
