/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract observer to prepare rules observer when a content is modifying
 */
public abstract class AbstractRulesStepObserver extends AbstractLogEnabled implements Observer, Contextualizable, Serviceable
{
    /** The key to get modified rule codes in request */
    private static final String __MODIFIED_RULE_CODES_KEY = "modified-rule-codes-key";
    
    /** The key to get degrees in request */
    private static final String __DEGREES_KEY = "degrees-key";
    
    /** The key to get regimes request */
    private static final String __REGIMES_KEY = "regimes-key";
    
    /** The key to get number of sessions in request */
    private static final String __NB_SESSIONS_KEY = "nb-sessions-key";
    
    /** The key to get child containers in request */
    private static final String __CHILD_CONTAINERS = "child-containers";
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The context */
    protected Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    public boolean supports(Event event)
    {
        if (!RulesManager.isRulesEnabled())
        {
            return false;
        }
        
        if (event.getId().equals(getSupportedEventId()))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return supportsContent(content);
        }
        
        return false;
    }
    
    public int getPriority()
    {
        // Low number for high priority to delete immediately derogated or additional rules
        return 100;
    }
    
    /**
     * Get the supported event id
     * @return the event id
     */
    protected abstract String getSupportedEventId();
    
    /**
     * <code>true</code> if the observer supports the content
     * @param content the content
     * @return <code>true</code> if the observer supports the content
     */
    protected abstract boolean supportsContent(Content content);
    
    /**
     * Add the old modified rule codes in the request
     * @param content the content
     * @param modifiedRuleCodes the modified rule codes to add
     */
    protected void _addOldModifiedRuleCodes(Content content, List<String> modifiedRuleCodes)
    {
        _setRequestAttribute(__MODIFIED_RULE_CODES_KEY, content, modifiedRuleCodes);
    }
    
    /**
     * Get the list of old modified rule codes
     * @param content the content
     * @return the list of old modified rule codes
     */
    protected List<String> _getOldModifiedRuleCodes(Content content)
    {
        return this.<List<String>>_getRequestAttribute(__CHILD_CONTAINERS, content, __ -> List.of());
    }
    
    /**
     * Add the old degree ids in the request
     * @param content the content
     * @param degreeIds the degree ids to add
     */
    protected void _addOldDegreeIds(Content content, List<String> degreeIds)
    {
        _setRequestAttribute(__DEGREES_KEY, content, degreeIds);
    }
    
    /**
     * Get the list of old degree ids
     * @param content the content
     * @return the list of old degree ids
     */
    protected List<String> _getOldDegreeIds(Content content)
    {
        return this.<List<String>>_getRequestAttribute(__DEGREES_KEY, content, __ -> List.of());
    }
    
    /**
     * Add the old regime ids in the request
     * @param content the content
     * @param regimeIds the regime ids to add
     */
    protected void _addOldRegimeIds(Content content, List<String> regimeIds)
    {
        _setRequestAttribute(__REGIMES_KEY, content, regimeIds);
    }
    
    /**
     * Get the list of old regime ids
     * @param content the content
     * @return the list of old regime ids
     */
    protected List<String> _getOldRegimeIds(Content content)
    {
        return this.<List<String>>_getRequestAttribute(__REGIMES_KEY, content, __ -> List.of());
    }
    
    /**
     * Add the old number of sessions in the request
     * @param content the content
     * @param nbSessions the number of sessions to add
     */
    protected void _addOldNbSessions(Content content, String nbSessions)
    {
        _setRequestAttribute(__NB_SESSIONS_KEY, content, nbSessions);
    }
    
    /**
     * Get the old number of sessions
     * @param content the content
     * @return the old number of sessions
     */
    protected String _getOldNumberOfSessions(Content content)
    {
        return this.<String>_getRequestAttribute(__NB_SESSIONS_KEY, content, __ -> null);
    }
    
    /**
     * Add the old child containers in the request
     * @param content the content
     * @param containers the containers to add
     */
    protected void _addOldChildContainers(Content content, Set<Container> containers)
    {
        _setRequestAttribute(__CHILD_CONTAINERS, content, containers);
    }
    
    /**
     * Get the old child containers
     * @param content the content
     * @return the old child containers
     */
    protected Set<Container> _getOldChildContainers(Content content)
    {
        return this.<Set<Container>>_getRequestAttribute(__CHILD_CONTAINERS, content, __ -> Set.of());
    }
    
    /**
     * Get the child container of type "année"
     * @param programPart the program part
     * @return the child container
     */
    protected Set<Container> _getChildContainerOfTypeYear(TraversableProgramPart programPart)
    {
        Set<Container> containers = new HashSet<>();
        for (ProgramItem child : programPart.getProgramPartChildren())
        {
            if (_odfHelper.isContainerOfTypeYear((Content) child))
            {
                containers.add((Container) child);
            }
            
            if (child instanceof TraversableProgramPart childProgramPart)
            {
                containers.addAll(_getChildContainerOfTypeYear(childProgramPart));
            }
        }
        
        return containers;
    }

    private <T> T _getRequestAttribute(String attributeName, Content content, Function<Content, T> defaultValue)
    {
        return this.<T>_getRequestAttribute(ContextHelper.getRequest(_context), attributeName)
            .computeIfAbsent(content, defaultValue);
    }
    
    @SuppressWarnings("unchecked")
    private <T> Map<Content, T> _getRequestAttribute(Request request, String attributeName)
    {
        return Optional.of(request)
            .map(r -> (Map<Content, T>) r.getAttribute(attributeName))
            .orElseGet(HashMap::new);
    }
    
    private <T> void _setRequestAttribute(String attributeName, Content content, T value)
    {
        Request request = ContextHelper.getRequest(_context);
        Map<Content, T> attribute = _getRequestAttribute(request, attributeName);
        attribute.put(content, value);
        request.setAttribute(attributeName, attribute);
    }
}
