/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract class for components that handle ODF workflow step
 *
 */
public abstract class AbstractWorkflowHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The suffixe for author of workflow action */
    protected static final String __AUTHOR_SUFFIX = "_author";
    
    /** The suffixe for date of workflow action */
    protected static final String __DATE_SUFFIX = "_date";
    
    /** The suffixe for comment of workflow action */
    protected static final String __COMMENT_SUFFIX = "_comment";
    
    /** The odf helper */
    protected ODFHelper _odfHelper;
    
    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    /**
     * Get the data of workflow step
     * @param workflowHolder the workflow data holder
     * @param prefixAttributeName the prefix of attribute names holding workflow step
     * @param stepId the step id
     * @return a record with the date, author and comment of the workflow action
     */
    protected ODFWorkflowStep getWorkflowStep(ModelAwareDataHolder workflowHolder, String prefixAttributeName, String stepId)
    {
        if (workflowHolder.hasValue(prefixAttributeName + __DATE_SUFFIX))
        {
            UserIdentity author = workflowHolder.getValue(prefixAttributeName + __AUTHOR_SUFFIX);
            LocalDate date = workflowHolder.getValue(prefixAttributeName + __DATE_SUFFIX);
            String comment = workflowHolder.getValue(prefixAttributeName + __COMMENT_SUFFIX, false, "");
            
            return new ODFWorkflowStep(stepId, date, author, comment);
        }
        return null;
    }
    
    /**
     * Get the date of a workflwo step
     * @param workflowHolder the workflow data holder
     * @param prefixAttributeName the prefix of attribute names holding workflow step
     * @return the date or null if not found
     */
    protected LocalDate getWorkflowStepDate(ModelAwareDataHolder workflowHolder, String prefixAttributeName)
    {
        return workflowHolder.getValue(prefixAttributeName + __DATE_SUFFIX, false, null);
    }
    
    /**
     * Set the data of a workflow step
     * @param content the content
     * @param workflowHolder the workflow data holder
     * @param prefixAttributeName the prefix of attribute names holding workflow step
     * @param date the date of the workflow action
     * @param user the author of the workflow action
     * @param comment the comment of workflow the action
     */
    protected void setWorkflowStep(ModifiableDefaultContent content, ModifiableModelAwareDataHolder workflowHolder, String prefixAttributeName, LocalDate date, UserIdentity user, String comment)
    {
        workflowHolder.setValue(prefixAttributeName + __DATE_SUFFIX, date);
        workflowHolder.setValue(prefixAttributeName + __AUTHOR_SUFFIX, user);
        workflowHolder.setValue(prefixAttributeName + __COMMENT_SUFFIX, comment);
    }
    
    /**
     * Remove a workflow step
     * @param content The content
     * @param dataHolder the workflow data holder
     * @param prefixAttributeName the prefix of attribute names holding the workflow step
     */
    protected void removeWorkflowStep(ModifiableDefaultContent content, ModifiableModelAwareDataHolder dataHolder, String prefixAttributeName)
    {
        dataHolder.removeValue(prefixAttributeName + __DATE_SUFFIX);
        dataHolder.removeValue(prefixAttributeName + __AUTHOR_SUFFIX);
        dataHolder.removeValue(prefixAttributeName + __COMMENT_SUFFIX);
    }
    
    /**
     * Get the parent programs for a workflow status point of view. The parent step holder if present is take into account
     * @param programItem the program item
     * @return the parent program taking into account the step holder if present
     */
    protected Set<Program> _getParentPrograms(ProgramItem programItem)
    {
        if (programItem instanceof Program program)
        {
            return Set.of(program);
        }
                
        Pair<StepHolderStatus, Container> stepHolder = _pilotageHelper.getStepHolder(programItem);
        switch (stepHolder.getLeft())
        {
            case SINGLE:
                // The course has a unique step holder, returns the parent' programs of this step holder
                return _odfHelper.getParentPrograms(stepHolder.getRight());
            case MULTIPLE: // The course has several parent containers of type "year": cannot determine the unique step holder
            case NONE: // The course has no step holder (ie. none parent container with "year" type)
            case NO_YEAR: // There is no container's nature of type "year", no step holder can be defined
            default:
                return _odfHelper.getParentPrograms(programItem);
        }
    }
    
    /**
     * Get the parent programs for a workflow status point of view. The parent step holder if present is taken into account
     * @param coursePart the course part
     * @return the parent program taking into account the step holder if present
     */
    protected Set<Program> _getParentPrograms(CoursePart coursePart)
    {
        Set<Program> programs = new HashSet<>();
        for (Course course : coursePart.getCourses())
        {
            programs.addAll(_getParentPrograms(course));
        }
        return programs;
    }
    
    /**
     * Get the parent containers of type 'year' for a workflow status point of view. If a unique step holder can be determined it will be returned.
     * @param programItem the program item
     * @return all parent containers of type 'year' or the step holder if it can be uniquely determined
     */
    protected Set<Container> _getParentYearContainers(ProgramItem programItem)
    {
        Pair<StepHolderStatus, Container> stepHolder = _pilotageHelper.getStepHolder(programItem);
        switch (stepHolder.getLeft())
        {
            case SINGLE:
                // The program item has a unique step holder, returns this step holder
                return Set.of(stepHolder.getRight());
            case MULTIPLE:
                // The program item has several parent containers of type "year"
                // Get the parent 'year' with higher rules status
                return _pilotageHelper.getParentYears(programItem);
            case NONE: // The course has no step holder (ie. none parent container with "year" type)
            case NO_YEAR: // There is no container's nature of type "year", no step holder can be defined
            default:
                return Set.of();
        }
    }
    
    /**
     * Get the parent containers of type 'year' for a workflow status point of view. If a unique step holder can be determined it will be returned.
     * @param coursePart the course part
     * @return all parent containers of type 'year' or the step holder if it can be uniquely determined
     */
    protected Set<Container> _getParentYearContainers(CoursePart coursePart)
    {
        Set<Container> containers = new HashSet<>();
        for (Course course : coursePart.getCourses())
        {
            containers.addAll(_getParentYearContainers(course));
        }
        return containers;
    }
    
    /**
     * Save the content if needed, add a version (checkpoint) and move the Live label if the last version was validated.
     * @param content The content to save
     * @return <code>true</code> if the content has changed
     */
    protected boolean saveContentAndNotify(ModifiableDefaultContent content)
    {
        boolean hasChanges = saveContent(content);
        if (hasChanges)
        {
            _notifyWorkflowModification(content);
        }
        return hasChanges;
    }
    
    /**
     * Save the content if needed, add a version (checkpoint) and move the Live label if the last version was validated.
     * @param content The content to save
     * @return <code>true</code> if the content has changed
     */
    protected boolean saveContent(ModifiableDefaultContent content)
    {
        if (content.needsSave())
        {
            boolean currentVersionIsLive = ArrayUtils.contains(content.getLabels(), CmsConstants.LIVE_LABEL);
            
            content.saveChanges();
            content.checkpoint();
            
            // Move the Live label if the last version was validated.
            if (currentVersionIsLive)
            {
                content.addLabel(CmsConstants.LIVE_LABEL, true);
            }
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Send a notification with the content modified event.
     * @param content The content to notify on
     */
    protected void _notifyWorkflowModification(Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_MODIFIED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Record representing a ODF workflow step
     * @param id the step id
     * @param date the date
     * @param author the author
     * @param comment the comment
     */
    public record ODFWorkflowStep (String id, LocalDate date, UserIdentity author, String comment) { /* Empty */ }
}
