/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.time.LocalDate;
import java.util.Comparator;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.core.cache.Cache;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;

/**
 * Helper for ODF pilotage status
 */
public class PilotageStatusHelper extends AbstractWorkflowHelper implements Initializable
{
    /** The component role. */
    public static final String ROLE = PilotageStatusHelper.class.getName();
    
    /** The super right for mention validation state */
    public static final String MENTION_VALIDATION_SUPER_RIGHT_ID = "ODF_Pilotage_Mention_Validated_Super_Rights";
    
    /** The super right for orgunit validation state */
    public static final String ORGUNIT_VALIDATION_SUPER_RIGHT_ID = "ODF_Pilotage_OrgUnit_Validated_Super_Rights";
    
    /** The super right for orgunit validation state */
    public static final String CFVU_VALIDATION_SUPER_RIGHT_ID = "ODF_Pilotage_CFVU_Validated_Super_Rights";
    
    private static final String __PARENT_PROGRAMS_WITH_HIGHER_STATUS_CACHE_ID = PilotageStatusHelper.class.getName() + "$parentPrograms";
    
    /** The attribute name for the pilotage composite */
    private static final String __PILOTAGE_COMPOSITE = "pilotage";
    
    /** The attribute name for the pilotage status */
    private static final String __PILOTAGE_STATUS = "pilotage_status";
    
    /** The prefix of attributes names holding mention validation step */
    private static final String __MENTION_VALIDATION_PREFIX = "mention_validation";
    
    /** The prefix of attributes names holding mention validation step */
    private static final String __ORGUNIT_VALIDATION_PREFIX = "orgunit_validation";
    
    /** The prefix of attributes names holding CFVU validation step */
    private static final String __CFVU_VALIDATION_PREFIX = "cfvu_validation";
    
    /**
     * Enumeration for the pilotage status
     */
    public enum PilotageStatus
    {
        /** State 0 : No status */
        NONE,
        /** State 1 : Mention validated */
        MENTION_VALIDATED,
        /** State 2 : OrgUnit validated */
        ORGUNIT_VALIDATED,
        /** State 3 : CFVU validated */
        CFVU_VALIDATED
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__PARENT_PROGRAMS_WITH_HIGHER_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_PROGRAM_WITH_HIGHER_PILOTAGE_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_PROGRAM_WITH_HIGHER_PILOTAGE_STATUS_DESCRIPTION"),
                true);
    }
    
    private Cache<String, String> _getProgramParentsCache()
    {
        return _cacheManager.get(__PARENT_PROGRAMS_WITH_HIGHER_STATUS_CACHE_ID);
    }
    
    /**
     * Return true if the current user has the edit super right depend on the pilotage status of the program
     * @param program the program
     * @return true if the user has the super right
     */
    public boolean hasEditSuperRight(Program program)
    {
        PilotageStatus pilotageStatus = getPilotageStatus(program);
        
        UserIdentity user = _currentUserProvider.getUser();
        switch (pilotageStatus)
        {
            case NONE:
                return true;
            case MENTION_VALIDATED:
                return _rightManager.hasRight(user, MENTION_VALIDATION_SUPER_RIGHT_ID, program) == RightResult.RIGHT_ALLOW;
            case ORGUNIT_VALIDATED:
                return _rightManager.hasRight(user, ORGUNIT_VALIDATION_SUPER_RIGHT_ID, program) == RightResult.RIGHT_ALLOW;
            case CFVU_VALIDATED:
                return _rightManager.hasRight(user, CFVU_VALIDATION_SUPER_RIGHT_ID, program) == RightResult.RIGHT_ALLOW;
            default:
                return false;
        }
    }
    
    /**
     * Get parent program or it self from program item with the higher pilotage status.
     * Program with pilotage status NONE are ignored, so can be null if there are no program with active pilotage status.
     * @param coursePart the course part
     * @return the program parent
     */
    public Program getParentProgramWithHigherPilotageStatus(CoursePart coursePart)
    {
        Cache<String, String> cache = _getProgramParentsCache();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Program parentProgram = _computeParentProgramWithHigherPilotageStatus(coursePart);
            cache.put(coursePart.getId(), parentProgram != null ? parentProgram.getId() : null);
            return parentProgram;
        }
    }
    
    private Program _computeParentProgramWithHigherPilotageStatus(CoursePart coursePart)
    {
        return _getParentPrograms(coursePart).stream()
                .filter(p -> !PilotageStatus.NONE.equals(getPilotageStatus(p)))
                .sorted(new PilotageStatusComparator().reversed())
                .findFirst()
                .orElse(null);
    }
    
    /**
     * Get parent program or it self from program item with the higher pilotage status.
     * Program with pilotage status NONE are ignored, so can be null if there are no program with active pilotage status.
     * @param programItem the program item
     * @return the program parent
     */
    public Program getParentProgramWithHigherPilotageStatus(ProgramItem programItem)
    {
        Cache<String, String> cache = _getProgramParentsCache();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Program parentProgram = _computeParentProgramWithHigherPilotageStatus(programItem);
            cache.put(programItem.getId(), parentProgram != null ? parentProgram.getId() : null);
            return parentProgram;
        }
    }
    
    private Program _computeParentProgramWithHigherPilotageStatus(ProgramItem programItem)
    {
        Set<Program> parentPrograms = _getParentPrograms(programItem);
        return parentPrograms.stream()
            .filter(p -> !PilotageStatus.NONE.equals(getPilotageStatus(p)))
            .sorted(new PilotageStatusComparator().reversed())
            .findFirst()
            .orElse(null);
    }
    
    private final class PilotageStatusComparator implements Comparator<Program>
    {
        public int compare(Program p1, Program p2)
        {
            PilotageStatus pilotageStatus1 = getPilotageStatus(p1);
            PilotageStatus pilotageStatus2 = getPilotageStatus(p2);
            
            switch (pilotageStatus1)
            {
                case NONE:
                    return PilotageStatus.NONE.equals(pilotageStatus2) ? 0 : -1;
                case CFVU_VALIDATED:
                    return PilotageStatus.CFVU_VALIDATED.equals(pilotageStatus2) ? 0 : 1;
                case ORGUNIT_VALIDATED:
                    if (PilotageStatus.NONE.equals(pilotageStatus2) || PilotageStatus.MENTION_VALIDATED.equals(pilotageStatus2))
                    {
                        return 1;
                    }
                    else
                    {
                        return PilotageStatus.ORGUNIT_VALIDATED.equals(pilotageStatus2) ? 0 : -1;
                    }
                case MENTION_VALIDATED:
                    if (PilotageStatus.NONE.equals(pilotageStatus2))
                    {
                        return 1;
                    }
                    else
                    {
                        return PilotageStatus.MENTION_VALIDATED.equals(pilotageStatus2) ? 0 : -1;
                    }
                default:
                    return -2;
            }
        }
    }
    
    /**
     * Get the pilotage status of the content
     * @param content the content
     * @return the pilotage status
     */
    public PilotageStatus getPilotageStatus(Content content)
    {
        String pilotageStatusDataPath = __PILOTAGE_COMPOSITE + ModelItem.ITEM_PATH_SEPARATOR + __PILOTAGE_STATUS;
        String status = content.getValue(pilotageStatusDataPath, false, PilotageStatus.NONE.name());
        return PilotageStatus.valueOf(status);
    }
    
    /**
     * Set the validation attribute (date, login, comment) to the content
     * @param content the content
     * @param validationDate the validation date
     * @param user the user
     * @param comment the comment
     * @param status the pilotage status
     */
    public void setValidationAttribute(ModifiableDefaultContent content, LocalDate validationDate, UserIdentity user, String comment, PilotageStatus status)
    {
        switch (status)
        {
            case NONE :
                // Do nothing
                break;
            case MENTION_VALIDATED :
                setMentionValidationAttribute(content, validationDate, user, comment);
                break;
            case ORGUNIT_VALIDATED :
                setOrgUnitValidationAttribute(content, validationDate, user, comment);
                break;
            case CFVU_VALIDATED :
                setCFVUValidationAttribute(content, validationDate, user, comment);
                break;
            default :
                getLogger().error("{} is an unknown pilotage status", status);
        }
        
        saveContentAndNotify(content);
    }
    
    /**
     * Remove the validation attribute from the content
     * @param content the content
     * @param status the pilotage status
     */
    public void removePilotageStatus(ModifiableDefaultContent content, PilotageStatus status)
    {
        switch (status)
        {
            case NONE :
                // Do nothing
                break;
            case MENTION_VALIDATED :
                removeMentionValidationAttribute(content);
                break;
            case ORGUNIT_VALIDATED :
                removeOrgUnitValidationAttribute(content);
                break;
            case CFVU_VALIDATED :
                removeCFVUValidationAttribute(content);
                break;
            default :
                getLogger().error("{} is an unknown pilotage status", status);
        }
        
        saveContentAndNotify(content);
    }
    
    /**
     * Set the validation attribute for 'mention validated' state
     * @param content the content
     * @param validationDate the validation date
     * @param user the user
     * @param comment the comment
     */
    public void setMentionValidationAttribute(ModifiableDefaultContent content, LocalDate validationDate, UserIdentity user, String comment)
    {
        ModifiableModelAwareComposite composite = content.getComposite(__PILOTAGE_COMPOSITE, true);
        composite.setValue(__PILOTAGE_STATUS, PilotageStatus.MENTION_VALIDATED.name());
        
        setWorkflowStep(content, composite, __MENTION_VALIDATION_PREFIX, validationDate, user, comment);
    }
    
    
    /**
     * Remove validation attribute for 'mention validated' state
     * @param content the content
     */
    public void removeMentionValidationAttribute(ModifiableDefaultContent content)
    {
        // Remove the first step
        removePilotageWorkflow(content);
    }
    
    /**
     * Set the validation attribute for 'orgunit validated' state
     * @param content the content
     * @param validationDate the validation date
     * @param user the user
     * @param comment the comment
     */
    public void setOrgUnitValidationAttribute(ModifiableDefaultContent content, LocalDate validationDate, UserIdentity user, String comment)
    {
        ModifiableModelAwareComposite composite = content.getComposite(__PILOTAGE_COMPOSITE, true);
        composite.setValue(__PILOTAGE_STATUS, PilotageStatus.ORGUNIT_VALIDATED.name());

        setWorkflowStep(content, composite, __ORGUNIT_VALIDATION_PREFIX, validationDate, user, comment);
    }
    
    /**
     * Remove validation attribute for 'orgunit validated' state
     * @param content the content
     */
    public void removeOrgUnitValidationAttribute(ModifiableDefaultContent content)
    {
        ModifiableModelAwareComposite compositeMetadata = content.getComposite(__PILOTAGE_COMPOSITE, true);
        compositeMetadata.setValue(__PILOTAGE_STATUS, PilotageStatus.MENTION_VALIDATED.name());

        removeWorkflowStep(content, compositeMetadata, __ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Set the validation attribute for 'CFVU validated' state
     * @param content the content
     * @param validationDate the validation date
     * @param user the login
     * @param comment the comment
     */
    public void setCFVUValidationAttribute(ModifiableDefaultContent content, LocalDate validationDate, UserIdentity user, String comment)
    {
        ModifiableModelAwareComposite composite = content.getComposite(__PILOTAGE_COMPOSITE, true);
        composite.setValue(__PILOTAGE_STATUS, PilotageStatus.CFVU_VALIDATED.name());

        setWorkflowStep(content, composite, __CFVU_VALIDATION_PREFIX, validationDate, user, comment);
    }
    
    /**
     * Remove validation attribute for 'CFVU validated' state
     * @param content the content
     */
    public void removeCFVUValidationAttribute(ModifiableDefaultContent content)
    {
        ModifiableModelAwareComposite composite = content.getComposite(__PILOTAGE_COMPOSITE, true);
        composite.setValue(__PILOTAGE_STATUS, PilotageStatus.ORGUNIT_VALIDATED.name());

        removeWorkflowStep(content, composite, __CFVU_VALIDATION_PREFIX);
    }
    
    /**
     * Get the date of mention validation
     * @param content the content
     * @return the date of validation or null if not found
     */
    public LocalDate getMentionValidationDate(Content content)
    {
        if (content.hasValue(__PILOTAGE_COMPOSITE))
        {
            ModelAwareDataHolder pilotageWorkflow = content.getComposite(__PILOTAGE_COMPOSITE);
            return getWorkflowStepDate(pilotageWorkflow, __MENTION_VALIDATION_PREFIX);
        }
        
        return null;
    }
    
    /**
     * Get the date of orgunit validation
     * @param content the content
     * @return the date of validation or null if not found
     */
    public LocalDate getOrgUnitValidationDate(Content content)
    {
        if (content.hasValue(__PILOTAGE_COMPOSITE))
        {
            ModelAwareDataHolder pilotageWorkflow = content.getComposite(__PILOTAGE_COMPOSITE);
            return getWorkflowStepDate(pilotageWorkflow, __ORGUNIT_VALIDATION_PREFIX);
        }
        
        return null;
    }
    
    /**
     * Get the date of CFVU validation
     * @param content the content
     * @return the date of validation or null if not found
     */
    public LocalDate getCFVUValidationDate(Content content)
    {
        if (content.hasValue(__PILOTAGE_COMPOSITE))
        {
            ModelAwareDataHolder pilotageWorkflow = content.getComposite(__PILOTAGE_COMPOSITE);
            return getWorkflowStepDate(pilotageWorkflow, __CFVU_VALIDATION_PREFIX);
        }
        
        return null;
    }
    
    /**
     * Get the workflow step for mention validation
     * @param content the content
     * @return the workflow step or null if content is not mention validation
     */
    public ODFWorkflowStep getMentionValidationStep(Content content)
    {
        if (content.hasValue(__PILOTAGE_COMPOSITE))
        {
            ModelAwareDataHolder pilotageWorkflow = content.getComposite(__PILOTAGE_COMPOSITE);
            return getWorkflowStep(pilotageWorkflow, __MENTION_VALIDATION_PREFIX, PilotageStatus.MENTION_VALIDATED.name());
        }
        
        return null;
    }
    
    /**
     * Get the workflow step for orgunit validation
     * @param content the content
     * @return the workflow step or null if content is not mention validation
     */
    public ODFWorkflowStep getOrgUnitValidationStep(Content content)
    {
        if (content.hasValue(__PILOTAGE_COMPOSITE))
        {
            ModelAwareDataHolder pilotageWorkflow = content.getComposite(__PILOTAGE_COMPOSITE);
            return getWorkflowStep(pilotageWorkflow, __ORGUNIT_VALIDATION_PREFIX, PilotageStatus.ORGUNIT_VALIDATED.name());
        }
        
        return null;
    }
    
    /**
     * Get the workflow step for CFVU validation
     * @param content the content
     * @return the workflow step or null if content is not mention validation
     */
    public ODFWorkflowStep getCFVUValidationStep(Content content)
    {
        if (content.hasValue(__PILOTAGE_COMPOSITE))
        {
            ModelAwareDataHolder pilotageWorkflow = content.getComposite(__PILOTAGE_COMPOSITE);
            return getWorkflowStep(pilotageWorkflow, __CFVU_VALIDATION_PREFIX, PilotageStatus.CFVU_VALIDATED.name());
        }
        
        return null;
    }
    
    /**
     * Remove the pilotage workflow composite.
     * @param content The content to clean
     */
    public void removePilotageWorkflow(ModifiableDefaultContent content)
    {
        content.removeValue(__PILOTAGE_COMPOSITE);
    }
}
