/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.consistency;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * Object representing the result of the analysis.
 */
public class ConsistencyAnalysisResult
{
    /** Name of the data to put the indentation level in a line*/ 
    public static final String INDENTATION_LEVEL_DATA_NAME = "indentationLevel";
    
    /** Status of the result */
    private ConsistencyAnalysisStatus _status;
    
    /** The text visible in the resume */
    private I18nizableText _statusText;
    
    /** The introduction text */
    private I18nizableText _introText;
    
    /** Columns description */
    private Map<String, I18nizableText> _columns;
    
    /** The names of the columns to indent regarding the indentation level */
    private List<String> _indentableColumns;
    
    /** List of lines */
    private List<Map<String, Object>> _lines;
    
    /**
     * Constructor.
     */
    public ConsistencyAnalysisResult()
    {
        _status = ConsistencyAnalysisStatus.NEUTRAL;
        _statusText = new I18nizableText(StringUtils.EMPTY);
        _introText = new I18nizableText(StringUtils.EMPTY);
        _columns = new LinkedHashMap<>();
        _indentableColumns = new ArrayList<>();
        _lines = new ArrayList<>();
    }
    
    /**
     * Get the status.
     * @return The status
     */
    public ConsistencyAnalysisStatus getStatus()
    {
        return _status;
    }
    
    /**
     * Set the status.
     * @param status The status to set
     */
    public void setStatus(ConsistencyAnalysisStatus status)
    {
        _status = status;
    }
    
    /**
     * Get the status text for the resume.
     * @return The status text
     */
    public I18nizableText getStatusText()
    {
        return _statusText;
    }
    
    /**
     * Set the status text for the resume.
     * @param statusText The status text
     */
    public void setStatusText(I18nizableText statusText)
    {
        _statusText = statusText;
    }
    
    /**
     * Get the intro text.
     * @return The intro text
     */
    public I18nizableText getIntroText()
    {
        return _introText;
    }
    
    /**
     * Set the intro text.
     * @param introText The intro text
     */
    public void setIntroText(I18nizableText introText)
    {
        _introText = introText;
    }
    
    /**
     * Get the columns.
     * @return The columns
     */
    public Map<String, I18nizableText> getColumns()
    {
        return _columns;
    }
    
    /**
     * Add a column.
     * @param name The name of the column
     * @param label The label of the column
     */
    public void addColumn(String name, I18nizableText label)
    {
        addColumn(name, label, false);
    }
    
    /**
     * Add a column.
     * @param name The name of the column
     * @param label The label of the column
     * @param isIndentable <code>true</code> if the column is indentable, <code>false</code> otherwise
     */
    public void addColumn(String name, I18nizableText label, boolean isIndentable)
    {
        _columns.put(name, label);
        if (isIndentable)
        {
            _indentableColumns.add(name);
        }
    }
    
    /**
     * Retrieves the list of the indentable columns
     * @return the list of the indentable columns
     */
    public List<String> getIndentableColumns()
    {
        return _indentableColumns;
    }
    
    /**
     * Get the lines.
     * @return The list of lines
     */
    public List<Map<String, Object>> getLines()
    {
        return _lines;
    }
    
    /**
     * Add a line.
     * @param line The line
     */
    public void addLine(Map<String, Object> line)
    {
        _lines.add(line);
    }
    
    /**
     * Add all the lines.
     * @param lines The list of lines to add
     */
    public void addLines(List<Map<String, Object>> lines)
    {
        _lines.addAll(lines);
    }
}
