/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.workflow;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.parameter.ValidationResult;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow condition to check if all children of the given program item tree are valids
 */
public class ValidateProgramItemTreeCondition extends AbstractContentWorkflowComponent implements EnhancedCondition
{
    /** Key in transient vars to check the program item tree or not. Default to true */
    public static final String CHECK_TREE_KEY = "checkTree";
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public boolean passesCondition(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        boolean checkTree = (boolean) transientVars.getOrDefault(CHECK_TREE_KEY, true);
        if (checkTree)
        {
            Content content = getContent(transientVars);
            if (content instanceof ProgramItem programItem)
            {
                return _validateProgramItem(programItem, getConditionFailures(transientVars));
            }
        }
        return true;
    }
    
    private boolean _validateProgramItem(ProgramItem programItem, List<ConditionFailure> conditionFailures) 
    {
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            ValidationResult validationResult = _contentHelper.validateContent((Content) child);
            if (validationResult.hasErrors())
            {
                if (conditionFailures != null)
                {
                    String errorMsg = validationResult.getErrors().stream().map(error -> _i18nUtils.translate(error, "en")).collect(Collectors.joining(", "));
                    conditionFailures.add(new ConditionFailure(
                            errorMsg,
                            ValidateProgramItemTreeCondition.class.getName()
                            ));
                }
                return false;
            }
            else 
            {
                boolean isValid = _validateProgramItem(child, conditionFailures);
                if (!isValid)
                {
                    return false;
                }
            }
        }
        
        return true;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_VALIDATE_PROGRAM_ITEM_TREE_CONDITION_LABEL");
    }
}
