/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.clientsideelement.SmartContentMenu;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Menu representing the MCC status of a content
 */
public class MCCWorkflowMenu extends SmartContentMenu
{
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;

    /** The ODF helper */
    protected ODFHelper _odfHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        if (!scripts.isEmpty())
        {
            String mccStatusName = (String) scripts.get(0).getParameters().get("mcc-status");
            if (StringUtils.isNotEmpty(mccStatusName))
            {
                MCCWorkflowStep mccStatus = MCCWorkflowStep.valueOf(mccStatusName);
                if (MCCWorkflowStep.RULES_VALIDATED.equals(mccStatus))
                {
                    if (!RulesManager.isRulesEnabled())
                    {
                        return List.of();
                    }
                }
            }
        }
        return scripts;
    }
    
    /**
     * Get status of given containers from MCC status perspective
     * @param containerIds The container ids
     * @param mccStatus The MCC status
     * @return the status
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatus(List<String> containerIds, String mccStatus)
    {
        Map<String, Object> results = super.getStatus(containerIds);
        
        results.put("valid-contents", new ArrayList<>());
        results.put("active-contents", new ArrayList<>());
        results.put("noyear-contents", new ArrayList<>());
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> activeContents = (List<Map<String, Object>>) results.get("active-contents");
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> validContents = (List<Map<String, Object>>) results.get("valid-contents");
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> noYearContents = (List<Map<String, Object>>) results.get("noyear-contents");
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        
        for (Map<String, Object> allRightContent : allrightContents)
        {
            String containerId = (String) allRightContent.get("id");
            Container container = _resolver.resolveById(containerId);
            
            _getStatus(container, MCCWorkflowStep.valueOf(mccStatus), activeContents, validContents, noYearContents);
        }
        
        List<Object> activeContentIds = activeContents.stream()
                .map(p -> p.get("id"))
                .toList();
        
        results.put("allright-contents", allrightContents.stream().filter(p -> activeContentIds.contains(p.get("id"))).toList());
        
        return results;
    }
    
    @Override
    protected boolean _hasReadAccess(Content content)
    {
        return _rightManager.currentUserHasRight("ODF_Pilotage_See_MCCHistory_Rights", content) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Get status for a container
     * @param container the container
     * @param mccStatus the MCC status
     * @param activeContents The list of contents those current MCC status matches 
     * @param validContents The list of contents those current MCC status go past the step
     * @param noYearContents The list of no-year containers
     */
    protected void _getStatus(Container container, MCCWorkflowStep mccStatus, List<Map<String, Object>> activeContents, List<Map<String, Object>> validContents, List<Map<String, Object>> noYearContents)
    {
        Map<String, Object> defaultContentParams = getContentDefaultParameters(container);
        
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            // The container is not a year
            Map<String, Object> noYearContentParams = new HashMap<>(defaultContentParams);
            noYearContentParams.put("description", _getNoYearDescription(container));
            noYearContents.add(noYearContentParams);
            return;
        }
        
        if (_mccWorkflowHelper.getCurrentSteps(container).contains(mccStatus))
        {
            // The container is currently in this step
            Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
            activeContentParams.put("description", _getActiveDescription(container));
            activeContents.add(activeContentParams);
        }
        else if (_mccWorkflowHelper.isStepValid(container, mccStatus))
        {
            // The container has already passed this step
            Map<String, Object> validContentParams = new HashMap<>(defaultContentParams);
            validContentParams.put("description", _getValidDescription(container));
            validContents.add(validContentParams);
        }
    }
    
    /**
     * Get i18n description when the MCC workflow status is active
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getActiveDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("active-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get i18n description when the MCC workflow status match the current step of the content
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getValidDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("valid-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get i18n description when the container is not of type year
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getNoYearDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("noyear-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
}
