/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl.tree;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.ametys.cms.content.compare.ContentComparatorResult;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;

/**
 * Structure representing a tree of {@link ProgramItem} to be used during the
 * processing of the MCC reports.
 */
public class ProgramItemTree
{
    /**
     * Change type
     */
    public static enum ChangeType
    {
        /**
         * The metadata is added in the new content
         */
        ADDED,
        /**
         * The metadata is removed in the new content
         */
        REMOVED,
        /**
         * The metadata is modified in the old content
         */
        MODIFIED_OLD,
        /**
         * The metadata is modified in the new content
         */
        MODIFIED_NEW
    }
    
    private ProgramItemTree _parent;
    private ProgramItem _current;
    private List<ProgramItemTree> _children = new ArrayList<>();
    private ContentComparatorResult _change;
    private EducationalPath _path;

    private ProgramItemTree(ProgramItemTree parent, ProgramItem current, ContentComparatorResult change)
    {
        _parent = parent;
        _current = current;
        _change = change;
    }
    
    private ProgramItemTree(ProgramItemTree parent, ProgramItem current)
    {
        this(parent, current, null);
    }
    
    /**
     * The constructor
     * @param current The {@link ProgramItem} the tree is build on
     */
    public ProgramItemTree(ProgramItem current)
    {
        this(null, current);
    }
    
    /**
     * The constructor
     * @param current The {@link ProgramItem} the tree is build on
     * @param change changes for this item (if applicable)
     */
    public ProgramItemTree(ProgramItem current, ContentComparatorResult change)
    {
        this(null, current, change);
    }

    /**
     * Declare a child for the current object.
     * @param child child for this node
     * @return The corresponding {@link ProgramItemTree} for the child
     */
    public ProgramItemTree addChild(ProgramItem child)
    {
        return addChild(child, null);
    }
    
    /**
     * Declare a child for the current object.
     * 
     * @param child child for this node
     * @param change changes for this child (if applicable)
     * @return The corresponding {@link ProgramItemTree} for the child
     */
    public ProgramItemTree addChild(ProgramItem child, ContentComparatorResult change)
    {
        ProgramItemTree childTree = _createChild(child, change);
        addChild(childTree);
        return childTree;
    }

    /**
     * Add a new child tree
     * @param childTree The child tree to add
     */
    protected void addChild(ProgramItemTree childTree)
    {
        _children.add(childTree);
    }
    
    /**
     * Create a child for this node, without adding it to the tree (usefull for tests before adding, addChild to add)
     * @param child The child {@link ProgramItem}
     * @param change Changes for this child (if applicable)
     * @return A child tree
     */
    private ProgramItemTree _createChild(ProgramItem child, ContentComparatorResult change)
    {
        return new ProgramItemTree(this, child, change);
    }

    /**
     * Get the parent ametys object
     * @return {@link ProgramItemTree} or null
     */
    public ProgramItemTree getParent()
    {
        return _parent;
    }

    /**
     * Get the current ametys object (root level of the tree).
     * @return {@link ProgramItem}
     */
    public ProgramItem getCurrent()
    {
        return _current;
    }

    /**
     * Get the current ametys object (root level of the tree).
     * @return {@link List} of {@link ProgramItemTree}
     */
    public List<ProgramItemTree> getChildren()
    {
        return _children;
    }

    /**
     * Get the change informations for this node
     * @return {@link ContentComparatorResult} or null if not applicable
     */
    public ContentComparatorResult getChange()
    {
        return _change;
    }
    
    /**
     * Get the path in the tree.
     * orgunits are excluded.
     * @return the path as a {@link String}
     */
    public EducationalPath getPath()
    {
        // Compute the path if it does not exist
        // Only manage program items
        if (_path == null)
        {
            return Optional.ofNullable(_parent)
                .map(ProgramItemTree::getPath)
                .map(parentPath -> EducationalPath.of(parentPath, _current))
                .orElseGet(() -> EducationalPath.of(_current));
        }
        
        return _path;
    }
}

