<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2024 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                              xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                              xmlns:cms="http://ametys.org/cms/3.0"
                              xmlns:exsl="http://exslt.org/common"
                              xmlns:docbook="http://docbook.org/ns/docbook"
                              xmlns:odf="org.ametys.plugins.odfweb.xslt.OdfXSLTHelper"
                              xmlns:double="java.lang.Double"
                              xmlns:fo="http://www.w3.org/1999/XSL/Format"
                              xmlns:stringutils="org.apache.commons.lang.StringUtils"
                              exclude-result-prefixes="exsl cms docbook odf double stringutils">
    
    <xsl:import href="odf-view:odf-pilotage://stylesheets/pilotage/report/params/mcc.xsl"/>
    <xsl:import href="odf-view:odf-pilotage://stylesheets/pilotage/utils/common/mcc2pdf.xsl"/>
    <xsl:import href="odf-view:odf-pilotage://stylesheets/pilotage/report/mcc/legende2pdf.xsl"/>
    
    <xsl:param name="mainOrientation" select="'landscape'"/>
    
    <xsl:variable name="mainContent" select="/report/container"/>
    <xsl:variable name="mccWorkflowRoot" select="/report"/>
    
    <xsl:attribute-set name="body">
        <xsl:attribute name="font-size">6pt</xsl:attribute>
        <xsl:attribute name="wrap-option">wrap</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="h1">
        <xsl:attribute name="font-size">24pt</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="h2">
        <xsl:attribute name="font-size">18pt</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="h3">
        <xsl:attribute name="font-size">13.5pt</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="maquette">
        <xsl:attribute name="text-align">center</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
        <xsl:attribute name="color">#000080</xsl:attribute>
        <xsl:attribute name="background-color">#EEEEFF</xsl:attribute>
        <xsl:attribute name="border">0.5pt #000080 solid</xsl:attribute>
        <xsl:attribute name="display-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="evaluation">
        <xsl:attribute name="text-align">center</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
        <xsl:attribute name="color">white</xsl:attribute>
        <xsl:attribute name="background-color">#000080</xsl:attribute>
        <xsl:attribute name="border">0.5pt white solid</xsl:attribute>
        <xsl:attribute name="display-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="session1">
        <xsl:attribute name="text-align">center</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
        <xsl:attribute name="color">white</xsl:attribute>
        <xsl:attribute name="background-color">#0000A0</xsl:attribute>
        <xsl:attribute name="border">0.5pt white solid</xsl:attribute>
        <xsl:attribute name="display-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="session2">
        <xsl:attribute name="text-align">center</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
        <xsl:attribute name="color">white</xsl:attribute>
        <xsl:attribute name="background-color">#0020C2</xsl:attribute>
        <xsl:attribute name="border">0.5pt white solid</xsl:attribute>
        <xsl:attribute name="display-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="row">
        <xsl:attribute name="border-left">0.5pt #cccccc solid</xsl:attribute>
        <xsl:attribute name="border-bottom">0.5pt #cccccc solid</xsl:attribute>
        <xsl:attribute name="border-right">0.5pt #cccccc solid</xsl:attribute>
        <xsl:attribute name="display-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="courselist" use-attribute-sets="row">
        <xsl:attribute name="font-style">italic</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="elp" use-attribute-sets="row"/>
    
    <xsl:attribute-set name="programpart">
        <xsl:attribute name="font-weight">bold</xsl:attribute>
        <xsl:attribute name="color">#000080</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="new-header">
        <xsl:attribute name="border-top">1pt dashed lightgrey</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="inside-cell">
        <xsl:attribute name="border-right">0.5pt #cccccc solid</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="centered">
        <xsl:attribute name="text-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="legend-title">
        <xsl:attribute name="color">#000080</xsl:attribute>
        <xsl:attribute name="font-weight">bold</xsl:attribute>
        <xsl:attribute name="padding-bottom">3pt</xsl:attribute>
        <xsl:attribute name="font-size">8pt</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="skills-title" use-attribute-sets="legend-title"/>
    
    <xsl:attribute-set name="legend">
        <xsl:attribute name="display-align">center</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="codeAmetys">
        <xsl:attribute name="color">#777777</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="size-8">
        <xsl:attribute name="font-size">8pt</xsl:attribute>
    </xsl:attribute-set>
        
    <xsl:attribute-set name="lang">
        <xsl:attribute name="font-weight">normal</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:attribute-set name="remarques">
        <xsl:attribute name="font-weight">normal</xsl:attribute>
        <xsl:attribute name="color">#777777</xsl:attribute>
    </xsl:attribute-set>
    
    <xsl:template name="property-title">
        <xsl:value-of select="/report/title" />
    </xsl:template>
    
    <xsl:template name="footer">
        <xsl:value-of select="title" />
        <xsl:text> - </xsl:text>
        <xsl:apply-templates select="program|subProgram|container" mode="catalog"/>
    </xsl:template>
    
    <xsl:template name="sequences">
        <xsl:call-template name="first-page-sequence"/>
        
        <xsl:if test="//course">
            <xsl:call-template name="main-sequence"/>
            
            <!-- Detailled skills table -->
            <xsl:if test="/report/skills/macro">
                <xsl:call-template name="skills-sequence"/>
            </xsl:if>
            
            <!-- Legend -->
            <xsl:call-template name="legend-sequence"/>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="sequence-id">
        <xsl:attribute name="id">main-sequence</xsl:attribute>
    </xsl:template>
    
    <xsl:template name="first-page-sequence">
        <fo:page-sequence master-reference="{$mainOrientation}-a4" force-page-count="no-force" font-size="10pt" line-height="1.5" id="first-page-sequence">
            <xsl:call-template name="xsl-region-before"/>
            <xsl:call-template name="xsl-region-after"/>
            <fo:flow flow-name="xsl-region-body">
                <fo:block-container xsl:use-attribute-sets="body">
                    <xsl:apply-templates select="program|subProgram|container" mode="title" />
                </fo:block-container>
            </fo:flow>
        </fo:page-sequence>
    </xsl:template>
    
    <xsl:template name="legend-sequence">
        <fo:page-sequence master-reference="{$mainOrientation}-a4" force-page-count="no-force" font-size="10pt" line-height="1.5" id="last-sequence">
            <xsl:call-template name="xsl-region-before"/>
            <xsl:call-template name="xsl-region-after"/>
            <fo:flow flow-name="xsl-region-body">
                <xsl:call-template name="legende" />
            </fo:flow>
        </fo:page-sequence>
    </xsl:template>
    
    <xsl:template name="skills-sequence">
        <!-- A3 landscape format -->
        <fo:page-sequence master-reference="landscape-a3" force-page-count="no-force" font-size="10pt" line-height="1.5" id="skills-sequence">
            <xsl:call-template name="xsl-region-before">
                <xsl:with-param name="format" select="'a3'"/>
                <xsl:with-param name="orientation" select="'landscape'"/>
            </xsl:call-template>
            <xsl:call-template name="xsl-region-after"/>
            <fo:flow flow-name="xsl-region-body">
                <xsl:call-template name="skills-body"/>
            </fo:flow>
        </fo:page-sequence>
    </xsl:template>
    
    <xsl:template name="skills-body">
        <fo:block-container id="skills1">
            <fo:block xsl:use-attribute-sets="skills-title">
                <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_SKILLS"/>
                <xsl:text> - </xsl:text>
                <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_1"/>
            </fo:block>
            <xsl:call-template name="skills-table">
                <xsl:with-param name="isFirstSession" select="true()"/>
            </xsl:call-template>
        </fo:block-container>
        
        <xsl:if test="//course[@session2 = 'available']">
            <fo:block-container id="skills2" page-break-before="always">
                <fo:block xsl:use-attribute-sets="skills-title">
                    <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_SKILLS"/>
                    <xsl:text> - </xsl:text>
                    <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_2"/>
                </fo:block>
                <xsl:call-template name="skills-table">
                    <xsl:with-param name="isFirstSession" select="false()"/>
                </xsl:call-template>
            </fo:block-container>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="skills-table">
        <xsl:param name="isFirstSession"/>
        
        <fo:table xsl:use-attribute-sets="body">
            <xsl:call-template name="skills-table-columns"/>
            <fo:table-header>
                <xsl:call-template name="skills-table-header"/>
            </fo:table-header>
            <fo:table-body>
                <xsl:apply-templates select="program|subProgram|container" mode="skills">
                    <xsl:with-param name="isFirstSession" select="$isFirstSession"/>
                </xsl:apply-templates>
            </fo:table-body>
        </fo:table>
    </xsl:template>
    
    <xsl:template name="skills-table-columns">
        <fo:table-column column-width="5cm"/>
        <fo:table-column column-width="5mm"/>
        <fo:table-column column-width="2cm"/>
        <fo:table-column column-width="5mm"/>
        <fo:table-column column-width="2cm"/>
        <xsl:for-each select="/report/skills/macro/micro">
            <fo:table-column/>
        </xsl:for-each>
    </xsl:template>
    
    <xsl:template name="skills-table-header">
        <!-- First row to group titles -->
        <fo:table-row>
            <fo:table-cell xsl:use-attribute-sets="maquette" number-columns-spanned="{$nbSkillsColumnsCommon}">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MAQUETTE"/></fo:block>
            </fo:table-cell>
            <fo:table-cell xsl:use-attribute-sets="evaluation" number-columns-spanned="{$nbSkillsColumns}">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_SKILLS"/></fo:block>
            </fo:table-cell>
        </fo:table-row>
        
        <!-- Second row with general attributes and macro skills -->
        <fo:table-row>
            <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_LABEL"/></fo:block>
            </fo:table-cell>
            <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_COEF"/></fo:block>
            </fo:table-cell>
            <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_MODALITE"/></fo:block>
            </fo:table-cell>
            <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NB_NOTES"/></fo:block>
            </fo:table-cell>
            <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2">
                <fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_PONDERATION"/></fo:block>
            </fo:table-cell>
            <xsl:for-each select="/report/skills/macro">
                <fo:table-cell xsl:use-attribute-sets="session1" number-columns-spanned="{count(micro)}">
                    <fo:block>
                        <xsl:if test="code != ''">
                            <xsl:value-of select="code"/>
                            <xsl:text> </xsl:text>
                        </xsl:if>
                        <xsl:value-of select="title"/>
                    </fo:block>
                </fo:table-cell>
            </xsl:for-each>
        </fo:table-row>
        
        <!-- Third row with micro skills -->
        <fo:table-row>
            <xsl:for-each select="/report/skills/macro/micro">
                <fo:table-cell xsl:use-attribute-sets="session2">
                    <fo:block>
                        <xsl:if test="code != ''">
                            <xsl:value-of select="code"/>
                            <xsl:text> </xsl:text>
                        </xsl:if>
                        <xsl:value-of select="title"/>
                    </fo:block>
                </fo:table-cell>
            </xsl:for-each>
        </fo:table-row>
    </xsl:template>
    
    <xsl:template match="program|subProgram|container" mode="skills">
        <xsl:param name="depth" select="0"/>
        <xsl:param name="isFirstSession"/>
        
        <xsl:if test="$isFirstSession or .//course[@session2 = 'available']">
            <!-- Display the program item title -->
            <fo:table-row xsl:use-attribute-sets="programpart">
                <fo:table-cell number-columns-spanned="{$nbAllSkillsColumns}">
                    <fo:block>
                        <!-- Identation. Print 4 spaces per level -->
                        <xsl:call-template name="recursive-print">
                            <xsl:with-param name="times" select="$depth" />
                        </xsl:call-template>
                        <xsl:value-of select="title"/>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
            
            <!-- Explore children -->
            <xsl:apply-templates select="subProgram|container|courseList" mode="skills">
                <xsl:with-param name="depth" select="$depth + 1"/>
                <xsl:with-param name="isFirstSession" select="$isFirstSession"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="courseList" mode="skills">
        <xsl:param name="depth"/>
        <xsl:param name="isFirstSession"/>
        
        <xsl:if test="$isFirstSession or .//course[@session2 = 'available']">
            <!-- Display the course list type if it is not mandatory -->
            <xsl:variable name="courseListChoice" select="choiceType/@value"/>
            <xsl:choose>
                <xsl:when test="$courseListChoice = 'CHOICE'">
                    <fo:table-row xsl:use-attribute-sets="courselist">
                        <fo:table-cell number-columns-spanned="{$nbAllSkillsColumns}">
                            <fo:block>
                                <xsl:call-template name="recursive-print">
                                    <xsl:with-param name="times" select="$depth" />
                                </xsl:call-template>
                                <i18n:translate>
                                    <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COURSELIST_CHOICE"/>
                                    <i18n:param><xsl:value-of select="min" /></i18n:param>
                                </i18n:translate>
                            </fo:block>
                        </fo:table-cell>
                    </fo:table-row>
                </xsl:when>
                <xsl:when test="$courseListChoice = 'OPTIONAL'">
                    <fo:table-row xsl:use-attribute-sets="courselist">
                        <fo:table-cell number-columns-spanned="{$nbAllSkillsColumns}">
                            <fo:block>
                                <xsl:call-template name="recursive-print">
                                    <xsl:with-param name="times" select="$depth" />
                                </xsl:call-template>
                                <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COURSELIST_OPTIONAL"/>
                            </fo:block>
                        </fo:table-cell>
                    </fo:table-row>
                </xsl:when>
            </xsl:choose>
            
            <!-- Explore children -->
            <xsl:apply-templates select="course" mode="skills">
                <xsl:with-param name="depth" select="$depth + ($courseListChoice != 'MANDATORY')"/>
                <xsl:with-param name="isFirstSession" select="$isFirstSession"/>
            </xsl:apply-templates>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="course" mode="skills">
        <xsl:param name="depth"/>
        <xsl:param name="isFirstSession"/>
        
        <xsl:if test="$isFirstSession or @session2 = 'available'">
            <xsl:variable name="sessionName">
                <xsl:choose>
                    <xsl:when test="$isFirstSession">mccSession1</xsl:when>
                    <xsl:otherwise>mccSession2</xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <xsl:variable name="mccSessionEntries" select="mcc/*[local-name() = $sessionName]/entry"/>
            
            <xsl:variable name="currentCourseType" select="courseType/code"/>
            <xsl:variable name="nbNotes" select="cms:getTotalNotesRowspan($mccSessionEntries, foo, false())"/>
            
            <!-- Details of the course -->
            <fo:table-row>
                <xsl:call-template name="course-attributes">
                    <xsl:with-param name="type" select="$currentCourseType"/>
                    <xsl:with-param name="isFirst" select="false()"/>
                </xsl:call-template>
                <fo:table-cell xsl:use-attribute-sets="elp" number-rows-spanned="{$nbNotes}">
                    <fo:block>
                        <!-- Identation. Print 4 spaces per level -->
                        <xsl:call-template name="recursive-print">
                            <xsl:with-param name="times" select="$depth" />
                        </xsl:call-template>
                        <xsl:value-of select="title"/>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell xsl:use-attribute-sets="elp" number-rows-spanned="{$nbNotes}">
                    <fo:block>
                        <xsl:call-template name="display-double">
                            <xsl:with-param name="value" select="odf:getValueForPath(@id, 'mccCoefficientByEducationalPath/coefficient', @path, 'mccCoefficient')" />
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                
                <!-- Details of the first mccSession1 entry, if there is not, empty cells are displayed -->
                <xsl:choose>
                    <xsl:when test="$mccSessionEntries">
                        <xsl:apply-templates select="$mccSessionEntries[1]" mode="skills-session"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <fo:table-cell xsl:use-attribute-sets="elp"><fo:block/></fo:table-cell>
                        <fo:table-cell xsl:use-attribute-sets="elp"><fo:block/></fo:table-cell>
                        <fo:table-cell xsl:use-attribute-sets="elp"><fo:block/></fo:table-cell>
                        <xsl:variable name="currentCourse" select="."/>
                        <xsl:for-each select="/report/skills/macro/micro">
                            <xsl:variable name="currentSkillId" select="@id"/>
                            <xsl:variable name="enableSkill" select="$currentCourse/acquiredMicroSkills/entry/microSkills[@id = $currentSkillId]"/>
                            <fo:table-cell xsl:use-attribute-sets="elp">
                                <xsl:if test="$enableSkill">
                                    <xsl:attribute name="background-color">#F0F8FF</xsl:attribute>
                                </xsl:if>
                                <fo:block/>
                             </fo:table-cell>
                        </xsl:for-each>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:table-row>
            
            <!-- Details of the other mccSession1 entries, except the first one (already displayed in the previous row) -->
            <!-- But for the first entry, we displayed the note except the first one too (already displayed in the previous row) -->
            <xsl:for-each select="$mccSessionEntries">
                <!-- Details of the mccSession1 entry -->
                <xsl:if test="position() != 1">
                    <fo:table-row>
                        <xsl:call-template name="course-attributes">
                            <xsl:with-param name="type" select="$currentCourseType"/>
                            <xsl:with-param name="isFirst" select="false()"/>
                        </xsl:call-template>
                        <xsl:apply-templates select="." mode="skills-session"/>
                    </fo:table-row>
                </xsl:if>
                
                <!-- Details of notes of the mccSession entry, except the first one (already displayed in the previous row) -->
                <xsl:for-each select="notes/entry[position() > 1]">
                    <fo:table-row>
                        <xsl:call-template name="course-attributes">
                            <xsl:with-param name="type" select="$currentCourseType"/>
                            <xsl:with-param name="isFirst" select="false()"/>
                        </xsl:call-template>
                        <xsl:apply-templates select="." mode="skills-session-notes"/>
                    </fo:table-row>
                </xsl:for-each>
            </xsl:for-each>
        </xsl:if>
        
        <!-- Explore children -->
        <xsl:apply-templates select="courseList" mode="skills">
            <xsl:with-param name="depth" select="$depth + 1"/>
            <xsl:with-param name="isFirstSession" select="$isFirstSession"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="entry" mode="skills-session">
        <xsl:variable name="nbNotesSessionEntry" select="count(notes/entry)"/>
        <fo:table-cell xsl:use-attribute-sets="elp" number-rows-spanned="{$nbNotesSessionEntry}">
            <fo:block>
                <xsl:call-template name="displayCodeOrShortLabel">
                    <xsl:with-param name="contentId" select="modalite/@id"/>
                </xsl:call-template>
                <xsl:if test="label != ''">
                    <xsl:text> - </xsl:text>
                    <xsl:value-of select="label"/>
                </xsl:if>
            </fo:block>
        </fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="elp" number-rows-spanned="{$nbNotesSessionEntry}">
            <fo:block>
                <xsl:choose>
                    <xsl:when test="notes/entry">
                        <xsl:value-of select="$nbNotesSessionEntry"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:call-template name="default-nb-notes"/>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:block>
        </fo:table-cell>
        
        <xsl:choose>
            <xsl:when test="notes/entry">
                <xsl:apply-templates select="notes/entry[1]" mode="skills-session-notes"/>
            </xsl:when>
            <xsl:otherwise>
                <fo:table-cell xsl:use-attribute-sets="elp"><fo:block/></fo:table-cell>
                <xsl:variable name="currentCourse" select="ancestor::course[1]"/>
                <xsl:for-each select="/report/skills/macro/micro">
                    <xsl:variable name="currentSkillId" select="@id"/>
                    <xsl:variable name="enableSkill" select="$currentCourse/acquiredMicroSkills/entry/microSkills[@id = $currentSkillId]"/>
                    <fo:table-cell xsl:use-attribute-sets="elp">
                        <xsl:if test="$enableSkill">
                            <xsl:attribute name="background-color">#F0F8FF</xsl:attribute>
                        </xsl:if>
                        <fo:block/>
                     </fo:table-cell>
                </xsl:for-each>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template match="entry" mode="skills-session-notes">
        <fo:table-cell xsl:use-attribute-sets="elp">
            <fo:block>
                <xsl:choose>
                    <xsl:when test="ponderation != ''">
                        <xsl:call-template name="display-double">
                            <xsl:with-param name="value" select="ponderation"/>
                            <xsl:with-param name="showZero" select="true()" />
                            <xsl:with-param name="format" select="'#,##'"/>
                        </xsl:call-template>
                        <xsl:text>%</xsl:text>
                        <xsl:if test="label != ''">
                            <xsl:text> - </xsl:text>
                            <xsl:value-of select="label"/>
                        </xsl:if>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:call-template name="default-ponderation"/>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:block>
        </fo:table-cell>
        
        <xsl:variable name="currentNote" select="."/>
        <xsl:variable name="currentCourse" select="ancestor::course[1]"/>
        <xsl:for-each select="/report/skills/macro/micro">
            <xsl:variable name="currentSkillId" select="@id"/>
            <xsl:variable name="enableSkill" select="$currentCourse/acquiredMicroSkills/entry/microSkills[@id = $currentSkillId]"/>
            <fo:table-cell xsl:use-attribute-sets="elp">
                <xsl:choose>
                    <xsl:when test="$enableSkill">
                        <xsl:attribute name="background-color">#F0F8FF</xsl:attribute>
                        <fo:block>
                            <xsl:variable name="matchingSkillEntry" select="$currentNote/skills/entry[skill/@id = $currentSkillId][not(path) or stringutils:endsWith(path, $currentCourse/@path)]"/>
                            <xsl:if test="$matchingSkillEntry">
                                <xsl:call-template name="display-double">
                                    <xsl:with-param name="value" select="$matchingSkillEntry/ponderation"/>
                                    <xsl:with-param name="showZero" select="true()" />
                                    <xsl:with-param name="format" select="'#,##'"/>
                                </xsl:call-template>
                                <xsl:text>%</xsl:text>
                            </xsl:if>
                        </fo:block>
                    </xsl:when>
                    <xsl:otherwise>
                        <fo:block/>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:table-cell>
        </xsl:for-each>
    </xsl:template>
    
    <xsl:template name="menu-items">
        <fo:bookmark-tree>
            <xsl:for-each select="//*[local-name() = 'program' or local-name() = 'subProgram' or local-name() = 'container'][courseList/course]">
                <xsl:choose>
                    <xsl:when test="local-name(..) = 'report'"><xsl:apply-templates select="courseList" mode="bookmark"/></xsl:when>
                    <xsl:otherwise><xsl:apply-templates select="." mode="bookmark"/></xsl:otherwise>
                </xsl:choose>
            </xsl:for-each>
            <xsl:if test="/report/skills/macro">
                <fo:bookmark internal-destination="skills1">
                    <fo:bookmark-title>
                        <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_SKILLS"/>
                        <xsl:text> - </xsl:text>
                        <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_1"/>
                    </fo:bookmark-title>
                </fo:bookmark>
                <xsl:if test="//course[@session2 = 'available']">
                    <fo:bookmark internal-destination="skills2">
                        <fo:bookmark-title>
                            <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_SKILLS"/>
                            <xsl:text> - </xsl:text>
                            <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_2"/>
                        </fo:bookmark-title>
                    </fo:bookmark>
                </xsl:if>
            </xsl:if>
            <xsl:if test="//course">
                <fo:bookmark internal-destination="legende">
                    <fo:bookmark-title><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_LEGENDE"/></fo:bookmark-title>
                </fo:bookmark>
            </xsl:if>
        </fo:bookmark-tree>
    </xsl:template>
    
    <xsl:template match="*" mode="bookmark">
        <fo:bookmark internal-destination="mcc-table-{generate-id()}">
            <fo:bookmark-title>
                <xsl:value-of select="title"/>
                <xsl:apply-templates select="." mode="code-parenthesis"/>
            </fo:bookmark-title>
        </fo:bookmark>
    </xsl:template>
    
    <xsl:template name="body">
        <fo:block-container xsl:use-attribute-sets="body">
            <xsl:apply-templates select="program|subProgram|container" mode="document-content" />
        </fo:block-container>
    </xsl:template>
    
    <xsl:template match="program|subProgram|container" mode="document-content">
        <xsl:apply-templates select="subProgram|container|courseList"/>
    </xsl:template>
    
    <xsl:template match="program|subProgram|container" mode="title">
        <fo:block xsl:use-attribute-sets="h1"><xsl:value-of select="title"/></fo:block>
        <xsl:call-template name="catalog-title"/>
        <xsl:if test="/report/path">
            <fo:block xsl:use-attribute-sets="h3">
                <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_HEADER_INCLUDES"/>
                <fo:list-block>
                    <xsl:for-each select="/report/path">
                        <fo:list-item>
                            <fo:list-item-label><fo:block>&#x2022;</fo:block></fo:list-item-label>
                            <fo:list-item-body start-indent="body-start()-5mm">
                                <fo:block><xsl:value-of select="."/></fo:block>
                            </fo:list-item-body>
                        </fo:list-item>
                    </xsl:for-each>
                </fo:list-block>
            </fo:block>
        </xsl:if>
        <fo:table xsl:use-attribute-sets="size-8">
            <fo:table-column column-width="50%"/>
            <fo:table-column column-width="50%"/>
            <fo:table-body>
                <xsl:apply-templates select="." mode="section-header-line"/>
                <fo:table-row xsl:use-attribute-sets="new-header"><fo:table-cell number-columns-spanned="2"><fo:block>&#160;</fo:block></fo:table-cell></fo:table-row>
            </fo:table-body>
        </fo:table>
    </xsl:template>
    
    <xsl:template name="catalog-title">
        <fo:block xsl:use-attribute-sets="h2"><xsl:apply-templates select="." mode="catalog"/></fo:block>
    </xsl:template>
    
    <xsl:template match="program|subProgram|container" mode="catalog">
        <xsl:value-of select="/report/catalog"/>
    </xsl:template>
    
    <xsl:template name="section-header">
        <fo:table xsl:use-attribute-sets="size-8">
            <fo:table-column column-width="50%"/>
            <fo:table-column column-width="50%"/>
            <fo:table-body>
                <xsl:apply-templates select="ancestor-or-self::*[parent::program or parent::subProgram or parent::container][local-name() = 'subProgram' or local-name() = 'container']" mode="section-header-line"/>
                <fo:table-row xsl:use-attribute-sets="new-header"><fo:table-cell number-columns-spanned="2"><fo:block>&#160;</fo:block></fo:table-cell></fo:table-row>
            </fo:table-body>
        </fo:table>
    </xsl:template>
    
    <xsl:template match="program" mode="section-header-line">
        <fo:table-row xsl:use-attribute-sets="new-header">
            <fo:table-cell>
                <fo:block>
                    <fo:inline font-weight="bold"><xsl:value-of select="title"/></fo:inline>
                    <xsl:apply-templates select="." mode="code-parenthesis"/>
                </fo:block>
            </fo:table-cell>
            <fo:table-cell>
                <fo:block>
                    <xsl:call-template name="label-value">
                        <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_ECTS'"/>
                        <xsl:with-param name="value" select="ects/@title"/>
                    </xsl:call-template>
                </fo:block>
            </fo:table-cell>
        </fo:table-row>
        <xsl:if test="degree/@id or mention/@id">
            <fo:table-row>
                <fo:table-cell>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_DEGREE'"/>
                            <xsl:with-param name="value" select="degree/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_MENTION'"/>
                            <xsl:with-param name="value" select="mention/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
        <xsl:if test="domain/@id or speciality != ''">
            <fo:table-row>
                <fo:table-cell>
                    <xsl:variable name="domains">
                        <xsl:for-each select="domain">
                            <xsl:value-of select="@title"/>
                            <xsl:if test="position() != last()"><xsl:text>, </xsl:text></xsl:if>
                        </xsl:for-each>
                    </xsl:variable>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_DOMAIN'"/>
                            <xsl:with-param name="value" select="$domains"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_SPECIALITY'"/>
                            <xsl:with-param name="value" select="speciality"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
        <xsl:if test="cms:isNotRichTextEmpty(knowledgeCheck)">
            <fo:table-row>
                <fo:table-cell number-columns-spanned="2">
                    <fo:block>
                        <xsl:apply-templates select="knowledgeCheck"/>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="subProgram" mode="section-header-line">
        <xsl:variable name="padding" select="count(ancestor::*[parent::program or parent::subProgram or parent::container][local-name() = 'subProgram' or local-name() = 'container'])*4" />
        
        <fo:table-row xsl:use-attribute-sets="new-header">
            <fo:table-cell padding-left="{$padding}mm">
                <fo:block>
                    <fo:inline font-weight="bold"><xsl:value-of select="title"/></fo:inline>
                    <xsl:apply-templates select="." mode="code-parenthesis"/>
                </fo:block>
            </fo:table-cell>
            <fo:table-cell>
                <fo:block>
                    <xsl:call-template name="label-value">
                        <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_ECTS'"/>
                        <xsl:with-param name="value" select="ects/@title"/>
                    </xsl:call-template>
                </fo:block>
            </fo:table-cell>
        </fo:table-row>
        <xsl:if test="cms:isNotRichTextEmpty(knowledgeCheck)">
            <fo:table-row>
                <fo:table-cell number-columns-spanned="2">
                    <fo:block>
                        <xsl:apply-templates select="knowledgeCheck"/>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="container" mode="section-header-line">
        <xsl:variable name="padding" select="count(ancestor::*[parent::program or parent::subProgram or parent::container][local-name() = 'subProgram' or local-name() = 'container'])*4" />
        
        <fo:table-row xsl:use-attribute-sets="new-header">
            <fo:table-cell padding-left="{$padding}mm">
                <fo:block>
                    <fo:inline font-weight="bold"><xsl:value-of select="title"/></fo:inline>
                    <xsl:apply-templates select="." mode="code-parenthesis"/>
                </fo:block>
            </fo:table-cell>
            <fo:table-cell>
                <fo:block>
                    <xsl:call-template name="label-value">
                        <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_ECTS'"/>
                        <xsl:with-param name="value">
                            <xsl:call-template name="display-double">
                                <xsl:with-param name="value" select="ects" />
                                <xsl:with-param name="showZero" select="false()" />
                            </xsl:call-template>
                        </xsl:with-param>
                    </xsl:call-template>
                </fo:block>
            </fo:table-cell>
        </fo:table-row>
        <xsl:if test="nature/@id or period/@id">
            <fo:table-row>
                <fo:table-cell padding-left="{$padding}mm">
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_NATURE'"/>
                            <xsl:with-param name="value" select="nature/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_PERIOD'"/>
                            <xsl:with-param name="value" select="period/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
        <xsl:if test="mccCompensable != ''">
            <fo:table-row>
                <fo:table-cell padding-left="{$padding}mm">
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_COMPENSABLE'"/>
                            <xsl:with-param name="value" select="mccCompensable"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell><fo:block/></fo:table-cell>
            </fo:table-row>
        </xsl:if>
        <xsl:if test="justifiedAbsence/@id != '' or unjustifiedAbsence/@id != ''">
            <fo:table-row>
                <fo:table-cell padding-left="{$padding}mm">
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_ABS_JUSTIF'"/>
                            <xsl:with-param name="value" select="justifiedAbsence/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_ABS_INJUSTIF'"/>
                            <xsl:with-param name="value" select="unjustifiedAbsence/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
        <xsl:if test="mccRegime/@id or mccNbSessions != ''">
            <fo:table-row>
                <fo:table-cell padding-left="{$padding}mm">
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_REGIME'"/>
                            <xsl:with-param name="value" select="mccRegime/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell>
                    <fo:block>
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey" select="'PLUGINS_ODF_PILOTAGE_MCC_HEADER_NB_SESSIONS'"/>
                            <xsl:with-param name="value" select="mccNbSessions"/>
                        </xsl:call-template>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
        </xsl:if>
        <xsl:if test="mcc_workflow/mcc_orgunit_validation_date != '' or mcc_workflow/cfvu_mcc_validation_date != ''">
            <fo:table-row>
                <fo:table-cell padding-left="{$padding}mm">
                    <fo:block>
                        <fo:inline font-weight="bold"><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_HEADER_VALIDATION_DATE_ORGUNIT"/></fo:inline>
                        <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_LABEL_VALUE_SEPARATOR"/>
                        <i18n:date src-pattern="yyyy-MM-dd" pattern="long"><xsl:value-of select="mcc_workflow/mcc_orgunit_validation_date"/></i18n:date>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell>
                    <fo:block>
                        <xsl:if test="mcc_workflow/cfvu_mcc_validation_date != ''">
                            <fo:inline font-weight="bold"><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_HEADER_VALIDATION_DATE_CFVU"/></fo:inline>
                            <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_LABEL_VALUE_SEPARATOR"/>
                            <i18n:date src-pattern="yyyy-MM-dd" pattern="long"><xsl:value-of select="mcc_workflow/cfvu_mcc_validation_date"/></i18n:date>
                        </xsl:if>
                    </fo:block>
                </fo:table-cell>
            </fo:table-row>
            <xsl:if test="mcc_workflow/mcc_orgunit_validation_comment != '' or mcc_workflow/cfvu_mcc_validation_comment != ''">
                <fo:table-row>
                    <fo:table-cell padding-left="{$padding}mm">
                        <fo:block>
                            <xsl:if test="mcc_workflow/mcc_orgunit_validation_comment != ''">
                                <xsl:call-template name="text.nl2block">  
                                    <xsl:with-param name="input" select="mcc_workflow/mcc_orgunit_validation_comment"/>
                                </xsl:call-template>
                            </xsl:if>
                        </fo:block>
                    </fo:table-cell>
                    <fo:table-cell>
                        <fo:block>
                            <xsl:if test="mcc_workflow/cfvu_mcc_validation_comment != ''">
                                <xsl:call-template name="text.nl2block">  
                                    <xsl:with-param name="input" select="mcc_workflow/cfvu_mcc_validation_comment"/>
                                </xsl:call-template>
                            </xsl:if>
                        </fo:block>
                    </fo:table-cell>
                </fo:table-row>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="table-columns">
        <xsl:param name="displayMcc2"/>
        
        <xsl:call-template name="table-columns-maquette"/>
        <xsl:call-template name="table-columns-evaluation"/>
        <xsl:call-template name="table-columns-session1"/>
        <xsl:if test="$displayMcc2">
            <xsl:call-template name="table-columns-session2"/>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="table-columns-maquette">
        <fo:table-column column-width="2%"/>
        <fo:table-column column-width="5%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="15%"/>
        <fo:table-column column-width="2%"/>
        <fo:table-column column-width="2%"/>
        <fo:table-column column-width="2%"/>
        <fo:table-column column-width="2%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="2%"/>
        <fo:table-column column-width="2%"/>
    </xsl:template>
    
    <xsl:template name="table-columns-evaluation"/>
    
    <xsl:template name="table-columns-session1">
        <fo:table-column column-width="8%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="2%"/>
    </xsl:template>
    
    <xsl:template name="table-columns-session2">
        <fo:table-column column-width="8%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
        <fo:table-column column-width="3%"/>
    </xsl:template>
    
    <xsl:template name="table-header">
        <xsl:param name="displayMcc2"/>
        
        <fo:table-header>
            <fo:table-row>
                <fo:table-cell xsl:use-attribute-sets="maquette" number-columns-spanned="{$nbColumnsMaquette}"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MAQUETTE"/></fo:block></fo:table-cell>
                <fo:table-cell xsl:use-attribute-sets="evaluation" number-columns-spanned="{$nbColumnsElpEval + $nbColumnsMccSession1 + $displayMcc2 * $nbColumnsMccSession2}"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_EVALUATION"/></fo:block></fo:table-cell>
            </fo:table-row>
            <fo:table-row>
                <xsl:call-template name="table-header-maquette"/>
                <xsl:call-template name="table-header-evaluation-elp"/>
                <fo:table-cell xsl:use-attribute-sets="session1" number-columns-spanned="{$nbColumnsMccSession1}"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_1"/></fo:block></fo:table-cell>
                <xsl:if test="$displayMcc2">
                    <fo:table-cell xsl:use-attribute-sets="session2" number-columns-spanned="{$nbColumnsMccSession2}"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_2"/></fo:block></fo:table-cell>
                </xsl:if>
            </fo:table-row>
            <fo:table-row>
                <xsl:call-template name="table-header-session1"/>
                <xsl:if test="$displayMcc2">
                    <xsl:call-template name="table-header-session2"/>
                </xsl:if>
            </fo:table-row>
        </fo:table-header>
    </xsl:template>
    
    <xsl:template name="table-header-maquette">
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_EVALUATED"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_CODE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_NATURE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_LABEL"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_SHARED"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_ECTS"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_COEF"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_NOTE_ELIM"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_REGIME"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="maquette" number-rows-spanned="2" number-columns-spanned="2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_VOLUME_HORAIRE"/></fo:block></fo:table-cell>
    </xsl:template>
    
    <xsl:template name="table-header-evaluation-elp"/>
    
    <xsl:template name="table-header-session1">
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_LABEL"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_MODALITE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NATURE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_DUREE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_COEF"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NOTE_ELIM"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NB_NOTES"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_PONDERATION"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session1"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_REPORT_SESSION_2"/></fo:block></fo:table-cell>
    </xsl:template>
    
    <xsl:template name="table-header-session2">
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_LABEL"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_MODALITE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NATURE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_DUREE"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_COEF"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NOTE_ELIM"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_NB_NOTES"/></fo:block></fo:table-cell>
        <fo:table-cell xsl:use-attribute-sets="session2"><fo:block><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_PONDERATION"/></fo:block></fo:table-cell>
    </xsl:template>
    
    <xsl:template match="program|subProgram|container">
        <xsl:if test="courseList/course">
            <xsl:call-template name="mcc-table">
                <xsl:with-param name="courseLists" select="courseList[course]"/>
            </xsl:call-template>
        </xsl:if>
        <xsl:apply-templates select="subProgram|container" />
    </xsl:template>
    
    <xsl:template match="courseList">
        <xsl:if test="course">
            <xsl:call-template name="mcc-table">
                <xsl:with-param name="courseLists" select="."/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="mcc-table">
        <xsl:param name="courseLists"/>
        
        <xsl:if test="$courseLists/course">
            <xsl:variable name="displayMcc2" select="count($courseLists//course[@session2 = 'available']) > 0"/>
            
            <fo:block-container page-break-before="always" id="mcc-table-{generate-id()}">
                <xsl:call-template name="section-header"/>
                <fo:table>
                    <xsl:call-template name="table-columns">
                        <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                    </xsl:call-template>
                    <xsl:call-template name="table-header">
                        <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                    </xsl:call-template>
                    <fo:table-body>
                        <xsl:apply-templates select="$courseLists" mode="table-content">
                            <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                        </xsl:apply-templates>
                    </fo:table-body>
                </fo:table>
                <xsl:if test="cms:isNotRichTextEmpty(controles)">
                    <fo:block>
                        <xsl:apply-templates select="controles" />
                    </fo:block>
                </xsl:if>
            </fo:block-container>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="courseList" mode="table-content">
        <xsl:param name="depth" select="'0'" />
        <xsl:param name="displayMcc2" />
        
        <xsl:variable name="courseListChoice" select="choiceType/@value"/>
        <xsl:choose>
            <xsl:when test="$courseListChoice = 'CHOICE'">
                <fo:table-row xsl:use-attribute-sets="courselist">
                    <fo:table-cell xsl:use-attribute-sets="inside-cell centered" number-columns-spanned="{$titleColumnPosition - 1}"><fo:block/></fo:table-cell>
                    <fo:table-cell number-columns-spanned="{$nbColumns - $titleColumnPosition + 1 - $nbColumnsMccSession2 * not($displayMcc2)}">
                        <fo:block>
                            <xsl:call-template name="recursive-print">
                                <xsl:with-param name="times" select="$depth" />
                            </xsl:call-template>
                            <i18n:translate>
                                <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COURSELIST_CHOICE"/>
                                <i18n:param><xsl:value-of select="min" /></i18n:param>
                            </i18n:translate>
                        </fo:block>
                    </fo:table-cell>
                </fo:table-row>
            </xsl:when>
            <xsl:when test="$courseListChoice = 'OPTIONAL'">
                <fo:table-row xsl:use-attribute-sets="courselist">
                    <fo:table-cell xsl:use-attribute-sets="inside-cell centered" number-columns-spanned="{$titleColumnPosition - 1}"><fo:block/></fo:table-cell>
                    <fo:table-cell number-columns-spanned="{$nbColumns - $titleColumnPosition + 1 - $nbColumnsMccSession2 * not($displayMcc2)}">
                        <fo:block>
                            <xsl:call-template name="recursive-print">
                                <xsl:with-param name="times" select="$depth" />
                            </xsl:call-template>
                            <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_COURSELIST_OPTIONAL"/>
                        </fo:block>
                    </fo:table-cell>
                </fo:table-row>
            </xsl:when>
        </xsl:choose>
        
        <xsl:apply-templates select="course">
            <xsl:with-param name="depth" select="$depth + ($courseListChoice != 'MANDATORY')"/>
            <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="course">
        <xsl:param name="depth" select="'0'" />
        <xsl:param name="displayMcc2" select="true()"/>
        
        <xsl:variable name="currentCourse" select="."/>
        <xsl:variable name="courseParts">
            <xsl:for-each select="exsl:node-set($naturesEnseignement)/*">
                <xsl:call-template name="enseignement-cells">
                    <xsl:with-param name="code">
                        <xsl:call-template name="displayCodeOrShortLabel">
                            <xsl:with-param name="contentId" select="@id"/>
                        </xsl:call-template>
                    </xsl:with-param>
                    <xsl:with-param name="volumeHoraire" select="$currentCourse/courseParts[nature/@id = current()/@id]/nbHours"/>
                    <xsl:with-param name="mcc1" select="$currentCourse/mcc/mccSession1/entry[natureEnseignement/@id = current()/@id]"/>
                    <xsl:with-param name="mcc2" select="$currentCourse[@session2 = 'available']/mcc/mccSession2/entry[natureEnseignement/@id = current()/@id]"/>
                    <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                </xsl:call-template>
            </xsl:for-each>
            
            <!-- Ajouter les lignes sans nature d'enseignement -->
            <xsl:if test="$currentCourse/mcc/*/entry[not(natureEnseignement/@id)]">
                <xsl:call-template name="enseignement-cells">
                    <xsl:with-param name="mcc1" select="$currentCourse/mcc/mccSession1/entry[not(natureEnseignement/@id)]"/>
                    <xsl:with-param name="mcc2" select="$currentCourse[@session2 = 'available']/mcc/mccSession2/entry[not(natureEnseignement/@id)]"/>
                    <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                </xsl:call-template>
            </xsl:if>
        </xsl:variable>
        
        <xsl:choose>
            <xsl:when test="exsl:node-set($courseParts)/row">
                <xsl:for-each select="exsl:node-set($courseParts)/row">
                    <xsl:choose>
                        <xsl:when test="position() = 1">
                            <fo:table-row xsl:use-attribute-sets="elp">
                                <xsl:call-template name="course-attributes">
                                    <xsl:with-param name="type" select="$currentCourse/courseType/code"/>
                                </xsl:call-template>
                                
                                <xsl:apply-templates select="$currentCourse" mode="cells">
                                    <xsl:with-param name="depth" select="$depth"/>
                                    <!-- Sum all computed rows from course parts and evaluations to get the total course rowspan -->
                                    <xsl:with-param name="courseRowspan" select="count(exsl:node-set($courseParts)/row)"/>
                                </xsl:apply-templates>
                                
                                <xsl:copy-of select="*"/>
                            </fo:table-row>
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:table-row xsl:use-attribute-sets="elp">
                                <xsl:call-template name="course-attributes">
                                    <xsl:with-param name="type" select="$currentCourse/courseType/code"/>
                                    <xsl:with-param name="isFirst" select="false()"/>
                                </xsl:call-template>
                                
                                <xsl:copy-of select="*"/>
                            </fo:table-row>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:for-each>
            </xsl:when>
            <xsl:otherwise>
                <fo:table-row xsl:use-attribute-sets="elp">
                    <xsl:call-template name="course-attributes">
                        <xsl:with-param name="type" select="$currentCourse/courseType/code"/>
                    </xsl:call-template>
                    
                    <xsl:apply-templates select="." mode="cells">
                        <xsl:with-param name="depth" select="$depth"/>
                    </xsl:apply-templates>
                    
                    <xsl:call-template name="mcc-empty-cells">
                        <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                    </xsl:call-template>
                </fo:table-row>
            </xsl:otherwise>
        </xsl:choose>
        
        <xsl:apply-templates select="courseList[course]" mode="table-content">
            <xsl:with-param name="depth" select="$depth + 1"/>
            <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template name="enseignement-cells">
        <xsl:param name="code"/>
        <xsl:param name="volumeHoraire" select="foo/*"/>
        <xsl:param name="mcc1"/>
        <xsl:param name="mcc2"/>
        <xsl:param name="displayMcc2"/>
        
        <xsl:variable name="mccLines" select="double:max(1, double:max(count($mcc1), count($mcc2) * $displayMcc2))"/>
        
        <xsl:if test="count($volumeHoraire) > 0 or $mccLines > 0">
            <xsl:variable name="natureEnseignementRowspan" select="cms:getTotalNotesRowspan($mcc1, $mcc2, $displayMcc2)"/>
            
            <row>
                <fo:table-cell number-rows-spanned="{$natureEnseignementRowspan}">
                    <fo:block>
                        <xsl:value-of select="$code"/>
                    </fo:block>
                </fo:table-cell>
                <fo:table-cell xsl:use-attribute-sets="inside-cell" number-rows-spanned="{$natureEnseignementRowspan}">
                    <fo:block>
                        <xsl:if test="$code or count($volumeHoraire) > 0">
                            <xsl:call-template name="display-double">
                                <xsl:with-param name="value" select="sum($volumeHoraire)"/>
                            </xsl:call-template>
                        </xsl:if>
                    </fo:block>
                </fo:table-cell>
                <xsl:call-template name="mcc">
                    <xsl:with-param name="mcc1" select="$mcc1[1]"/>
                    <xsl:with-param name="mcc2" select="$mcc2[1]"/>
                    <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                </xsl:call-template>
            </row>
            
            <!-- Other notes on same MCC session line -->
            <xsl:call-template name="mcc-recursive-row-notes">
                <xsl:with-param name="currentPosition" select="2"/>
                <xsl:with-param name="mcc1" select="$mcc1[1]"/>
                <xsl:with-param name="mcc2" select="$mcc2[1]"/>
                <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
            </xsl:call-template>
            
            <!-- Other MCC session lines -->
            <xsl:call-template name="mcc-recursive-row">
                <xsl:with-param name="currentPosition" select="2"/>
                <xsl:with-param name="maxPosition" select="$mccLines"/>
                <xsl:with-param name="mcc1" select="$mcc1"/>
                <xsl:with-param name="mcc2" select="$mcc2"/>
                <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="mcc-recursive-row-notes">
        <xsl:param name="currentPosition"/>
        <xsl:param name="mcc1"/>
        <xsl:param name="mcc2"/>
        <xsl:param name="displayMcc2"/>
        <xsl:param name="maxPosition" select="double:max(count($mcc1/notes/entry), count($mcc2/notes/entry) * $displayMcc2)"/>
        
        <xsl:if test="$currentPosition &lt;= $maxPosition">
            <row>
                <!-- session 1 -->
                <xsl:apply-templates select="$mcc1" mode="cell-session-ponderation">
                    <xsl:with-param name="position" select="$currentPosition"/>
                    <xsl:with-param name="isLast" select="false()"/>
                </xsl:apply-templates>
                
                <!-- session 2 -->
                <xsl:if test="$displayMcc2">
                    <xsl:apply-templates select="$mcc2" mode="cell-session-ponderation">
                        <xsl:with-param name="position" select="$currentPosition"/>
                    </xsl:apply-templates>
                </xsl:if>
            </row>
            
            <!-- Recursive call -->
            <xsl:call-template name="mcc-recursive-row-notes">
                <xsl:with-param name="currentPosition" select="$currentPosition + 1"/>
                <xsl:with-param name="mcc1" select="$mcc1"/>
                <xsl:with-param name="mcc2" select="$mcc2"/>
                <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                <xsl:with-param name="maxPosition" select="$maxPosition"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="mcc-recursive-row">
        <xsl:param name="currentPosition"/>
        <xsl:param name="maxPosition"/>
        <xsl:param name="mcc1"/>
        <xsl:param name="mcc2"/>
        <xsl:param name="displayMcc2"/>
        
        <xsl:if test="$currentPosition &lt;= $maxPosition">
            <row>
                <xsl:call-template name="mcc">
                    <xsl:with-param name="mcc1" select="$mcc1[$currentPosition]"/>
                    <xsl:with-param name="mcc2" select="$mcc2[$currentPosition]"/>
                    <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
                </xsl:call-template>
            </row>
            
            <!-- Other notes on same MCC session line -->
            <xsl:call-template name="mcc-recursive-row-notes">
                <xsl:with-param name="currentPosition" select="2"/>
                <xsl:with-param name="mcc1" select="$mcc1[$currentPosition]"/>
                <xsl:with-param name="mcc2" select="$mcc2[$currentPosition]"/>
                <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
            </xsl:call-template>
            
            <!-- Recursive call -->
            <xsl:call-template name="mcc-recursive-row">
                <xsl:with-param name="currentPosition" select="$currentPosition + 1"/>
                <xsl:with-param name="maxPosition" select="$maxPosition"/>
                <xsl:with-param name="mcc1" select="$mcc1"/>
                <xsl:with-param name="mcc2" select="$mcc2"/>
                <xsl:with-param name="displayMcc2" select="$displayMcc2"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>

    <xsl:template name="mcc">
        <xsl:param name="mcc1"/>
        <xsl:param name="mcc2"/>
        <xsl:param name="displayMcc2"/>
        
        <!-- Rowspan is the max number of notes between the entry of session1 and session2, it there is no notes, it is set to 1 -->
        <xsl:variable name="sessionRowspan" select="double:max(1, double:max(count($mcc1/notes/entry), count($mcc2/notes/entry) * $displayMcc2))"/>
        
        <!-- session 1 -->
        <xsl:call-template name="mcc-session-cells-1">
            <xsl:with-param name="entry" select="$mcc1" />
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan" />
        </xsl:call-template>
        
        <!-- session 2 -->
        <xsl:if test="$displayMcc2">
            <xsl:call-template name="mcc-session-cells-2">
                <xsl:with-param name="entry" select="$mcc2" />
                <xsl:with-param name="sessionRowspan" select="$sessionRowspan" />
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="mcc-session-cells-1">
        <xsl:param name="entry" />
        <xsl:param name="sessionRowspan" />
        
        <xsl:choose>
            <xsl:when test="$entry">
                <xsl:apply-templates select="$entry" mode="mcc-session-cells-non-empty-1">
                    <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
                </xsl:apply-templates>
            </xsl:when>
            <xsl:otherwise>
                <xsl:call-template name="mcc-session-empty-cells-1">
                    <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="mcc-session-cells-2">
        <xsl:param name="entry" />
        <xsl:param name="sessionRowspan" />
        
        <xsl:choose>
            <xsl:when test="$entry">
                <xsl:apply-templates select="$entry" mode="mcc-session-cells-non-empty-2">
                    <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
                </xsl:apply-templates>
            </xsl:when>
            <xsl:otherwise>
                <xsl:call-template name="mcc-session-empty-cells-2">
                    <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template match="entry" mode="style"/>
    
    <xsl:template match="entry" mode="mcc-session-cells-non-empty-1">
        <xsl:param name="sessionRowspan" select="1"/>
        
        <xsl:apply-templates select="." mode="cell-session-label">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-modalite">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-nature">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-duree">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-coef">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-note">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-nb-notes">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-ponderation">
            <xsl:with-param name="isLast" select="false()"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-report">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="entry" mode="mcc-session-cells-non-empty-2">
        <xsl:param name="sessionRowspan" select="1"/>
        
        <xsl:apply-templates select="." mode="cell-session-label">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-modalite">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-nature">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-duree">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-coef">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-note">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-nb-notes">
            <xsl:with-param name="sessionRowspan" select="$sessionRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-session-ponderation"/>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-label">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:if test="label != ''">
                    <fo:block font-weight="bold"><xsl:value-of select="label"/></fo:block>
                </xsl:if>
                <xsl:if test="foreignLanguage/@id">
                    <fo:block xsl:use-attribute-sets="lang">
                        <xsl:apply-templates select="." mode="style"/>
                        
                        <xsl:call-template name="label-value">
                            <xsl:with-param name="i18nLabelKey">PLUGINS_ODF_PILOTAGE_MCC_COL_MCC_SESSION_LANG</xsl:with-param>
                            <xsl:with-param name="value" select="foreignLanguage/@title"/>
                        </xsl:call-template>
                    </fo:block>
                </xsl:if>
                <xsl:if test="remarques != ''">
                    <fo:block xsl:use-attribute-sets="remarques">
                        <xsl:apply-templates select="." mode="style"/>
                        
                        <xsl:call-template name="text.nl2block">  
                            <xsl:with-param name="input" select="remarques"/>
                        </xsl:call-template>
                    </fo:block>
                </xsl:if>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-modalite">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:if test="nombre != ''">
                    <xsl:value-of select="nombre" />
                    <xsl:if test="modalite/@id">
                        <xsl:text> </xsl:text>
                    </xsl:if>
                </xsl:if>
                <xsl:call-template name="displayCodeOrShortLabel">
                    <xsl:with-param name="contentId" select="modalite/@id"/>
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-nature">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:call-template name="displayCodeOrShortLabel">
                    <xsl:with-param name="contentId" select="nature/@id"/>
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-duree">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:if test="duree != ''"><xsl:value-of select="odf:minute2hour(duree)" /></xsl:if>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-coef">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:if test="coefficient != ''">
                    <xsl:choose>
                        <xsl:when test="contains(coefficient, '%') or contains(coefficient, '/')">
                            <xsl:value-of select="coefficient"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:call-template name="display-double">
                                <xsl:with-param name="value" select="coefficient" />
                                <xsl:with-param name="showZero" select="true()" />
                            </xsl:call-template>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:if>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-note">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:call-template name="display-double">
                    <xsl:with-param name="value" select="noteEliminatoire" />
                    <xsl:with-param name="showZero" select="true()" />
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-nb-notes">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:choose>
                    <xsl:when test="notes/entry">
                        <xsl:value-of select="count(notes/entry)"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:call-template name="default-nb-notes"/>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-ponderation">
        <xsl:param name="position" select="1"/>
        <xsl:param name="isLast" select="true()"/>
        
        <xsl:variable name="value" select="notes/entry[$position]/ponderation"/>
        
        <fo:table-cell xsl:use-attribute-sets="centered">
            <xsl:apply-templates select="." mode="style"/>
            <xsl:if test="$isLast">
                <xsl:attribute name="border-right">0.5pt #cccccc solid</xsl:attribute>
            </xsl:if>
            
            <fo:block>
                <xsl:choose>
                    <xsl:when test="$value">
                        <xsl:call-template name="display-double">
                            <xsl:with-param name="value" select="$value"/>
                            <xsl:with-param name="showZero" select="true()" />
                            <xsl:with-param name="format" select="'#,##'"/>
                        </xsl:call-template>
                        <xsl:text>%</xsl:text>
                    </xsl:when>
                    <xsl:when test="$position = 1">
                        <xsl:call-template name="default-ponderation"/>
                    </xsl:when>
                </xsl:choose>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="entry" mode="cell-session-report">
        <xsl:param name="sessionRowspan"/>
        
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-rows-spanned="{$sessionRowspan}">
            <xsl:apply-templates select="." mode="style"/>
            
            <fo:block>
                <xsl:if test="reportSession2 = 'true'">X</xsl:if>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cells">
        <xsl:param name="depth" select="'0'" />
        <xsl:param name="courseRowspan" select="1"/>
        
        <xsl:apply-templates select="." mode="cell-course-evaluated">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-code">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-nature">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-label">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
            <xsl:with-param name="depth" select="$depth"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-shared">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-ects">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-coef">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-note">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
        <xsl:apply-templates select="." mode="cell-course-regime">
            <xsl:with-param name="courseRowspan" select="$courseRowspan"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-evaluated">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell centered" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:if test="not(isEvaluated) or isEvaluated = 'true'">X</xsl:if>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-code">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell codeAmetys" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:apply-templates select="." mode="code"/>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-nature">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:value-of select="courseType/code" />
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-label">
        <xsl:param name="courseRowspan"/>
        <xsl:param name="depth"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <!-- Identation. Print 4 spaces per level -->
                <xsl:call-template name="recursive-print">
                    <xsl:with-param name="times" select="$depth" />
                </xsl:call-template>
                <xsl:value-of select="title" />
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-shared">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell centered" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:if test="isShared = 'true'">X</xsl:if>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-ects">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell centered" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:call-template name="display-double">
                    <xsl:with-param name="value" select="odf:getEcts(@id, @path)" />
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-coef">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell centered" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:call-template name="display-double">
                    <xsl:with-param name="value" select="odf:getValueForPath(@id, 'mccCoefficientByEducationalPath/coefficient', @path, 'mccCoefficient')" />
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-regime">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:call-template name="displayCodeOrShortLabel">
                    <xsl:with-param name="contentId" select="mccRegime/@id"/>
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template match="course" mode="cell-course-note">
        <xsl:param name="courseRowspan"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-rows-spanned="{$courseRowspan}">
            <fo:block>
                <xsl:call-template name="display-double">
                    <xsl:with-param name="value" select="mccNoteEliminatoire" />
                    <xsl:with-param name="showZero" select="true()" />
                </xsl:call-template>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <xsl:template name="mcc-empty-cells">
        <xsl:param name="displayMcc2" />
        
        <xsl:call-template name="volume-horaire-empty-cells" />
        <xsl:call-template name="mcc-session-empty-cells-1" />
        <xsl:if test="$displayMcc2">
            <xsl:call-template name="mcc-session-empty-cells-2" />
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="volume-horaire-empty-cells">
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-columns-spanned="2"><fo:block/></fo:table-cell>
    </xsl:template>
    
    <xsl:template name="mcc-session-empty-cells-1">
        <xsl:param name="sessionRowspan" select="1"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-columns-spanned="{$nbColumnsMccSession1}" number-rows-spanned="{$sessionRowspan}"><fo:block/></fo:table-cell>
    </xsl:template>
    
    <xsl:template name="mcc-session-empty-cells-2">
        <xsl:param name="sessionRowspan" select="1"/>
        <fo:table-cell xsl:use-attribute-sets="inside-cell" number-columns-spanned="{$nbColumnsMccSession2}" number-rows-spanned="{$sessionRowspan}"><fo:block/></fo:table-cell>
    </xsl:template>
    
    <xsl:template match="knowledgeCheck|controles">
        <xsl:apply-templates select="docbook:article">
            <xsl:with-param name="level" select="3"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <xsl:template name="course-attributes">
        <xsl:param name="type"/>
        <xsl:param name="isFirst" select="true()"/>
        
        <xsl:if test="$type='UE'">
            <xsl:attribute name="font-weight">bold</xsl:attribute>
            <xsl:if test="$isFirst">
                <xsl:attribute name="border-top">0.5pt black solid</xsl:attribute>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="label-value">
        <xsl:param name="i18nLabelKey"/>
        <xsl:param name="value"/>
        
        <xsl:if test="$value != ''">
            <fo:inline font-weight="bold"><i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="{$i18nLabelKey}"/></fo:inline>
            <i18n:text i18n:catalogue="plugin.odf-pilotage" i18n:key="PLUGINS_ODF_PILOTAGE_MCC_LABEL_VALUE_SEPARATOR"/>
            <xsl:value-of select="$value"/>
        </xsl:if>
    </xsl:template>
</xsl:stylesheet>
