/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.clientsideelement.content.SmartContentClientSideElementHelper;
import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.core.ui.ClientSideElement;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for {@link ClientSideElement} that should be aware of attributes' restrictions
 *
 */
public class PilotageClientSideElementHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = PilotageClientSideElementHelper.class.getName();
    
    private SmartContentClientSideElementHelper _smartHelper;
    private AmetysObjectResolver _resolver;

    public void service(ServiceManager manager) throws ServiceException
    {
        _smartHelper = (SmartContentClientSideElementHelper) manager.lookup(SmartContentClientSideElementHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the status of content from the point of view of restrictions on attributes
     * @param contentIds the contents' ids
     * @param modelPath the path of attribute
     * @param results the result for client side
     */
    @SuppressWarnings("unchecked")
    public void getRestrictionStatus(List<String> contentIds, String modelPath, Map<String, Object> results)
    {
        List<Map<String, Object>> invalidContents = new ArrayList<>();
        
        for (String contentId : contentIds)
        {
            Content content = _resolver.resolveById(contentId);
            if (content instanceof ProgramItem programItem)
            {
                ModelItem modelItem = ModelHelper.getModelItem(modelPath, ((Content) programItem).getModel());
                if (modelItem instanceof RestrictedModelItem rModelItem && !rModelItem.canWrite(programItem))
                {
                    Map<String, Object> contentParams = _smartHelper.getContentDefaultParameters((Content) programItem);
                    invalidContents.add(contentParams);
                }
            }
        }
        
        results.put("invalidrestrictionstatus-contents", invalidContents);
    }
}
