/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.observation.Event;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * Abstract observer to delete rule when observing a thematic content
 */
public abstract class AbstractThematicRulesObserver extends AbstractRulesStepObserver
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** The rules manager */
    protected RulesManager _rulesManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _rulesManager = (RulesManager) manager.lookup(RulesManager.ROLE);
    }
    
    @Override
    protected boolean supportsContent(Content content)
    {
        return _contentTypesHelper.isInstanceOf(content, RulesManager.THEMATIC_CONTENT_TYPE);
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        ModifiableContent content = (ModifiableContent) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        String catalog = content.getValue("catalog");
        String thematicCode = content.getValue(RulesManager.THEMATIC_CODE);
        List< ? extends ModifiableIndexableRepeaterEntry> ruleEntries = Optional.ofNullable(content)
                .map(c -> c.<ModifiableIndexableRepeater>getValue(RulesManager.THEMATIC_RULES))
                .map(ModifiableIndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .toList();
        
        Map<Container, Set<Integer>> rulePositionsToDelete = getRulePositionsToDelete(content, catalog, thematicCode, ruleEntries);
        for (Container container : rulePositionsToDelete.keySet())
        {
            _rulesManager.deleteRules(container, rulePositionsToDelete.get(container));
        }
    }
    
    /**
     * Get the position of rule to delete by container
     * @param content the thematic content
     * @param catalog the catalog
     * @param thematicCode the thematic code
     * @param ruleEntries the rule entries
     * @return the rule to delete by container
     * @throws RepositoryException if an error occurred
     */
    protected abstract Map<Container, Set<Integer>> getRulePositionsToDelete(ModifiableContent content, String catalog, String thematicCode, List< ? extends ModifiableIndexableRepeaterEntry> ruleEntries) throws RepositoryException;
    
    /**
     * Get all the rule positions linked to the thematic or with the given codes
     * @param session the session
     * @param catalog the catalog
     * @param thematicCode the thematic code
     * @param ruleCodes the rule codes
     * @param withAdditional <code>true</code> to has additional rules
     * @return the map of rule positions for each container
     * @throws RepositoryException if an error occurred
     */
    protected Map<Container, Set<Integer>> _getContainerRulePositions(Session session, String catalog, String thematicCode, List<String> ruleCodes, boolean withAdditional) throws RepositoryException
    {
        StringBuilder queryString = new StringBuilder("//element(*, ametys:container)");
        // Add condition on catalog
        queryString.append("[@ametys:catalog = '").append(catalog).append("']/ametys:rules/*[");
        // Add condition on thematic code
        queryString.append("@ametys:thematicCode = '").append(thematicCode).append("'");
        // Add condition on rules code from the thematic
        for (String code : ruleCodes)
        {
            queryString.append(" or @ametys:code = '").append(code).append("'");
        }
        queryString.append("]");
        
        return _getContainers(session, queryString.toString())
            .stream()
            .collect(
                Collectors.toMap(
                    Function.identity(),
                    c -> _getRulePositions(
                        c,
                        e -> _matchRule(e, ruleCodes, withAdditional)
                    )
                )
            );
    }
    
    private boolean _matchRule(ModifiableIndexableRepeaterEntry entry, List<String> ruleCodes, boolean withAdditional)
    {
        if (entry.hasValue(RulesManager.RULE_THEMATIC_CODE))
        {
            return withAdditional;
        }
        return ruleCodes.contains(entry.getValue(RulesManager.RULE_CODE));
    }
    
    /**
     * Get the container rule positions
     * @param container the container
     * @param filter the filter
     * @return the set of rule positions
     */
    private Set<Integer> _getRulePositions(Container container, Predicate<ModifiableIndexableRepeaterEntry> filter)
    {
        return Optional.ofNullable(container)
                .map(c -> c.<ModifiableIndexableRepeater>getValue(RulesManager.CONTAINER_RULES))
                .map(ModifiableIndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .filter(filter)
                .map(ModifiableIndexableRepeaterEntry::getPosition)
                .collect(Collectors.toSet());
    }
    
    /**
     * Get containers from the query
     * @param session the session
     * @param queryAsString the query as string
     * @return the list of container
     * @throws RepositoryException if an error occurred
     */
    private Set<Container> _getContainers(Session session, String queryAsString) throws RepositoryException
    {
        Set<Container> containers = new HashSet<>();
        
        QueryManager queryManager = session.getWorkspace().getQueryManager();
        @SuppressWarnings("deprecation")
        Query query = queryManager.createQuery(queryAsString, Query.XPATH);
        NodeIterator nodeIterator = query.execute().getNodes();
        while (nodeIterator.hasNext())
        {
            Node node = nodeIterator.nextNode();
            containers.add(_getParentContainer(node));
        }
        
        return containers;
    }
    
    private Container _getParentContainer(Node node)
    {
        try
        {
            Node parent = node.getParent();
            while (parent != null)
            {
                if (parent.getPrimaryNodeType().isNodeType(ContainerFactory.CONTAINER_NODETYPE))
                {
                    return _resolver.resolve(parent, false);
                }
                parent = parent.getParent();
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred getting the parent container", e);
        }
        
        throw new AmetysRepositoryException("We should find a parent container from node : " + node);
    }
}
