/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.actions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.odfpilotage.helper.AbstractWorkflowHelper.ODFWorkflowStep;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Abstract action to get the workflow history step of a {@link ProgramItem}
 *
 */
public abstract class AbstractProgramItemHistoryAction extends ServiceableAction
{
    /** The user helper */
    protected UserHelper _userHelper;

    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content helper */
    protected ContentHelper _contentHelper;

    /** The right manager */
    protected RightManager _rightManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);

        String id = request.getParameter("contentId");
        ProgramItem programItem = _resolver.resolveById(id);

        if (canSeeHistory(programItem))
        {
            ProgramItem parentProgramItem = getParentProgramItemForHistory(programItem);
            
            if (parentProgramItem != null)
            {
                List<ODFWorkflowStep> workflowSteps = getWorkflowSteps(parentProgramItem);
                
                List<Map<String, Object>> workflowHistory = workflowSteps.stream()
                        .map(this::workflowStepToJson)
                        .collect(Collectors.toList());
                
                result.put("workflow", workflowHistory);
                
                result.put("title", _contentHelper.getTitle((Content) parentProgramItem));
                
                result.put("isParentProgram", !parentProgramItem.getId().equals(programItem.getId()));
            }
            else
            {
                result.put("workflow", List.of());
            }
        }
        else
        {
            result.put("noright", true);
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Determines if user can access to history
     * @param programItem the program item
     * @return true if user is allowed to see history
     */
    protected abstract boolean canSeeHistory(ProgramItem programItem);
    
    /**
     * Get the parent program item from which the history will be computed
     * @param programItem The program item
     * @return the parent program
     */
    protected abstract ProgramItem getParentProgramItemForHistory(ProgramItem programItem);

    
    /**
     * Get the workflow steps
     * @param programItem the program item
     * @return the workflow steps
     */
    protected abstract List<ODFWorkflowStep> getWorkflowSteps(ProgramItem programItem);
    
    /**
     * Get the workflow step to JSON format
     * @param step teh workflow step
     * @return the workflow step as JSON
     */
    protected Map<String, Object> workflowStepToJson(ODFWorkflowStep step)
    {
        Map<String, Object> step2json = new HashMap<>();
        step2json.put("id", step.id());

        UserIdentity author = step.author();
        step2json.put("author", _userHelper.user2json(author));
        step2json.put("comment", step.comment());
        step2json.put("date", DateUtils.localDateToString(step.date()));
        
        step2json.put("title", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_WORKFLOW_" + step.id().toUpperCase() + "_ACTION_LABEL"));
        step2json.put("actionLabel", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_WORKFLOW_" + step.id().toUpperCase() + "_ACTION_DESCRIPTION"));
        
        return step2json;
    }
}
