/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.cost.entity;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.ametys.runtime.model.ModelItem;

/**
 * Class representing {@link Effectives}
 */
public class Effectives
{
    /** The overridden effective */
    // Store to Double because of weight
    private Optional<Double> _overriddenEffective;
    
    /** The entered effective */
    // Store to Double because of weight
    private Optional<Double> _enteredEffective;
    
    /** The estimated effective */
    private Optional<Double> _estimatedEffective;

    /** The estimated effective */
    private Map<String, Double> _estimatedEffectiveByPath;
    
    /** The computed effective (from steps, ignoring intermediate overrides) */
    private Optional<Double> _computedEffective;
    
    /** The computed effectives by step */
    private Map<String, Double> _computedEffectiveByStep;
    
    /** The local effective by complete path */
    private Map<String, Double> _localEffectiveByPath;
    
    /** The number of occurrences for mutualized elements */
    private Integer _nbOccurrences;
    
    /** The consistency status toward the parent's effectives */
    private boolean _isInconsistentWithParent;
   
    /**
     * The constructor
     */
    public Effectives()
    {
        _overriddenEffective = Optional.empty();
        _enteredEffective = Optional.empty();
        _estimatedEffective = Optional.empty();
        _estimatedEffectiveByPath = new HashMap<>();
        _computedEffective = Optional.empty();
        _computedEffectiveByStep = new HashMap<>();
        _localEffectiveByPath = new HashMap<>();
        _nbOccurrences = 1;
        _isInconsistentWithParent = false;
    }
    
    /**
     * Apply a weight to all effectives by step of a program item but not save it.
     * @param weight the weight
     * @return A clone of the effectives but with a weight
     */
    public Effectives cloneWithWeight(double weight)
    {
        Effectives clone = new Effectives();
        clone.setOverriddenEffective(_overriddenEffective.map(eff -> eff * weight));
        clone.setEnteredEffective(_enteredEffective.map(eff -> eff * weight));
        for (Map.Entry<String, Double> entry : _estimatedEffectiveByPath.entrySet())
        {
            clone.addEstimatedEffective(entry.getKey(), entry.getValue() * weight);
        }
        for (Map.Entry<String, Double> entry : _computedEffectiveByStep.entrySet())
        {
            clone.addComputedEffective(entry.getKey(), entry.getValue() * weight);
        }
        return clone;
    }
    
    /**
     * Set the number of the occurrences of the current item.
     * @param nbOccurrences The number of occurrences
     */
    public void setNbOccurrences(Integer nbOccurrences)
    {
        // Prevent from dividing by 0, should not happen
        _nbOccurrences = nbOccurrences == 0 ? 1 : nbOccurrences;
    }
    
    /**
     * Get the number of the occurrences of the current item.
     * @return The number of occurrences
     */
    public Integer getNbOccurrences()
    {
        return _nbOccurrences;
    }
    
    /**
     * Set the overridden effectives by the cost computation tool
     * @param overriddenEffective An {@link Optional} of the overridden effective
     */
    public void setOverriddenEffective(Optional<Double> overriddenEffective)
    {
        _overriddenEffective = overriddenEffective;
    }
    
    /**
     * Set the entered effectives in the 'numberOfStudentsEstimated' attribute
     * @param enteredEffective An {@link Optional} of the entered effective
     */
    public void setEnteredEffective(Optional<Double> enteredEffective)
    {
        _enteredEffective = enteredEffective;
    }

    /**
     * Add an estimated effective to the existing one.
     * @param itemPath The item path
     * @param estimatedEffective The estimated effective
     */
    public void addEstimatedEffective(String itemPath, Double estimatedEffective)
    {
        _estimatedEffective = Optional.of(Double.sum(_estimatedEffective.orElse(0d), estimatedEffective));
        _estimatedEffectiveByPath.put(itemPath, estimatedEffective + _estimatedEffectiveByPath.getOrDefault(itemPath, 0d));
    }

    /**
     * Add a computed effective to the existing one.
     * @param step The step id
     * @param computedEffective The computed effective
     */
    public void addComputedEffective(String step, Double computedEffective)
    {
        _computedEffective = Optional.of(Double.sum(_computedEffective.orElse(0d), computedEffective));
        _computedEffectiveByStep.put(step, computedEffective + _computedEffectiveByStep.getOrDefault(step, 0d));
    }

    /**
     * Add a local effective to the existing one.
     * @param itemPath The item path
     * @param localEffective The local effective
     */
    public void addLocalEffective(String itemPath, Double localEffective)
    {
        _localEffectiveByPath.put(itemPath, localEffective);
    }
    
    /**
     * Add the effectives to the current effectives
     * @param effectivesToAdd the parent effectives
     * @param nameInPath the name to add in the path
     */
    public void add(Effectives effectivesToAdd, String nameInPath)
    {
        // Update estimated effectives if written effective is not defined
        if (_overriddenEffective.isEmpty() && _enteredEffective.isEmpty())
        {
            // Report overridden or estimated parent effectives to current estimated effectives
            Optional<Double> parentLocalEffectives = effectivesToAdd.getOverriddenEffective()
                    .or(effectivesToAdd::getEnteredEffective)
                    .map(eff -> eff / effectivesToAdd.getNbOccurrences());
            if (parentLocalEffectives.isPresent())
            {
                addEstimatedEffective(nameInPath, parentLocalEffectives.get());
            }
            else
            {
                for (Map.Entry<String, Double> entry : effectivesToAdd.getEstimatedEffectiveByPath().entrySet())
                {
                    addEstimatedEffective(entry.getKey() + ModelItem.ITEM_PATH_SEPARATOR + nameInPath, entry.getValue());
                }
            }
        }
        
        // Computed effectives
        for (Map.Entry<String, Double> entry : effectivesToAdd.getComputedEffectiveByStep().entrySet())
        {
            addComputedEffective(entry.getKey(), entry.getValue());
        }
    }
    
    /**
     * Sum effectives to the current effectives
     * @param effectivesToAdd the child effectives
     */
    public void sum(Effectives effectivesToAdd)
    {
        // Update estimated effectives if written effective is not defined
        for (Map.Entry<String, Double> entry : effectivesToAdd.getEstimatedEffectiveByPath().entrySet())
        {
            addEstimatedEffective(entry.getKey(), entry.getValue() / effectivesToAdd.getNbOccurrences());
        }
        
        // Computed effectives
        for (Map.Entry<String, Double> entry : effectivesToAdd.getComputedEffectiveByStep().entrySet())
        {
            addComputedEffective(entry.getKey(), entry.getValue() / effectivesToAdd.getNbOccurrences());
        }
    }

    /**
     * Get the entered effectives in the 'numberOfStudentsEstimated' attribute
     * @return an {@link Optional} of the entered effective, {@link Optional#empty()} if not defined
     */
    public Optional<Double> getEnteredEffective()
    {
        return _enteredEffective;
    }

    /**
     * Get the overridden effectives by the cost computation tool
     * @return an {@link Optional} of the overridden effective, {@link Optional#empty()} if not defined
     */
    public Optional<Double> getOverriddenEffective()
    {
        return _overriddenEffective;
    }
    
    /**
     * Get the global estimated effective
     * @return the estimated effective
     */
    public Optional<Double> getEstimatedEffective()
    {
        return _estimatedEffective;
    }
    
    /**
     * Get the global estimated effectives by path
     * @return the estimated effectives by path
     */
    public Map<String, Double> getEstimatedEffectiveByPath()
    {
        return _estimatedEffectiveByPath;
    }
    
    /**
     * Get the inconsistency status
     * @return true if effectives in inconsistent with parent's
     */
    public boolean isInconsistentWithParent()
    {
        return _isInconsistentWithParent;
    }

    /**
     * Set the inconsistency status
     * @param isInconsistentWithParent true if effective is inconsistent
     */
    public void setInconsistentWithParent(boolean isInconsistentWithParent)
    {
        this._isInconsistentWithParent = isInconsistentWithParent;
    }

    /**
     * Get the global computed effective
     * @return the computed effective
     */
    public Optional<Double> getComputedEffective()
    {
        return _computedEffective;
    }
    
    /**
     * Get the global estimated effectives by step
     * @return the computed effectives by step
     */
    public Map<String, Double> getComputedEffectiveByStep()
    {
        return _computedEffectiveByStep;
    }
    
    /**
     * Get the local effectives of a program item in a specific path
     * @param programItemPath the complete path
     * @return the local effectives
     */
    public Double getLocalEffective(String programItemPath)
    {
        return _localEffectiveByPath.get(programItemPath);
    }
    
    /**
     * Get the local effectives of a program item by path
     * @return the local effectives
     */
    public Map<String, Double> getLocalEffectiveByPath()
    {
        return _localEffectiveByPath;
    }
}
