/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.plugins.odfpilotage.cost.CostComputationComponent;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.plugins.odfpilotage.cost.entity.EqTD;
import org.ametys.plugins.odfpilotage.cost.entity.ProgramItemData;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;

/**
 * Pilotage report for "potentiel enseignant".
 */
public class PotentielEnseignantReport extends AbstractReport
{
    /** CalculerEffectifComponent */
    protected CostComputationComponent _costComputationComponent;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _costComputationComponent = (CostComputationComponent) manager.lookup(CostComputationComponent.ROLE);
    }
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        return "potentielenseignant";
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_XLS;
    }

    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_XLS);
    }

    @Override
    protected void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
        Map<String, Double> nbHoursSumByDiscipline = _sumNbHoursByDiscipline(catalog, lang, orgUnit);
        
        try
        {
            handler.startDocument();
        
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            XMLUtils.startElement(handler, "report", attrs);
            
            // Sax data
            _refTableHelper.saxItems(handler, "odf-enumeration.DisciplineEnseignement");
            _saxPotentielEnseignant(handler, orgUnit);
            _saxNbHoursSum(handler, nbHoursSumByDiscipline);
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Potentiel enseignant' report for orgunit '{}'", orgUnit.getUAICode(), e);
        }
    }
    
    /**
     * Sax the final lines : number of hours sum by discipline.
     * @param handler The handler
     * @throws SAXException if an error occurs
     */
    private void _saxNbHoursSum(ContentHandler handler, Map<String, Double> nbHoursSumByDiscipline) throws SAXException
    {
        XMLUtils.startElement(handler, "sumsByDiscipline");
        for (String discipline : nbHoursSumByDiscipline.keySet())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("discipline", discipline);
            XMLUtils.createElement(handler, "sum", attrs, ReportHelper.FORMAT_2_DIGITS.format(nbHoursSumByDiscipline.get(discipline)));
        }
        XMLUtils.endElement(handler, "sumsByDiscipline");
    }
    
    /**
     * Sax the potentials for the given org unit.
     * @param handler The handler
     * @param orgUnit The org unit
     * @throws SAXException if an error occurs
     */
    private void _saxPotentielEnseignant(ContentHandler handler, OrgUnit orgUnit) throws SAXException
    {
        ModelAwareRepeater repeater = orgUnit.getRepeater("potentielsEnseignant");
        XMLUtils.startElement(handler, "potentiels");
        if (repeater != null)
        {
            for (ModelAwareRepeaterEntry entry : repeater.getEntries())
            {
                XMLUtils.startElement(handler, "entry");
                
                // Discipline
                _saxOptional(
                    handler,
                    "discipline",
                    Optional.of("discipline")
                        .map(entry::<ContentValue>getValue)
                        .map(ContentValue::getContentId)
                );
                
                // Potentiel
                _saxOptional(
                    handler,
                    "potentiel",
                    Optional.of("potentiel")
                        .map(entry::<Long>getValue)
                        .map(String::valueOf)
                );
                
                XMLUtils.endElement(handler, "entry");
            }
        }
        XMLUtils.endElement(handler, "potentiels");
    }
    
    /**
     * Sax if optional is present.
     * @param handler The handler
     * @param tagName The tag name
     * @param value The optional value to sax
     * @throws SAXException if an error occurs
     */
    private void _saxOptional(ContentHandler handler, String tagName, Optional<String> value) throws SAXException
    {
        if (value.isPresent())
        {
            XMLUtils.createElement(handler, tagName, value.get());
        }
    }
    
    /**
     * Sum hours by discipline for the course parts of the given programs which belongs to the given org unit.
     * @param catalog The catalog
     * @param lang The language
     * @param orgUnit The org unit
     * @return a {@link Map} of each displine with its sum of hours in the orgunit
     */
    private Map<String, Double> _sumNbHoursByDiscipline(String catalog, String lang, OrgUnit orgUnit)
    {
        CostComputationData costData = _costComputationComponent.computeCostsOnOrgUnit(orgUnit, catalog, lang, false);
        
        Set<CoursePart> courseParts = new LinkedHashSet<>();
        costData.getComputedPrograms().forEach(program -> courseParts.addAll(_getCoursePartsForProgramItem(program)));
        
        Map<String, Double> nbHoursSumByDiscipline = new HashMap<>();
        
        for (CoursePart coursePart : courseParts)
        {
            // Si le coursePart est porté par la composante (étapes -> étape porteuse -> composantes -> filles de la composante principale ?)
            if (_isHoldByOrgUnit(coursePart, orgUnit))
            {
                Double eqTD = Optional.of(coursePart)
                        .map(CoursePart::getId)
                        .map(costData::get)
                        .map(ProgramItemData::getEqTD)
                        .map(EqTD::getGlobalEqTD)
                        .orElse(0D);

                nbHoursSumByDiscipline.compute(
                    _getDisciplineEnseignement(coursePart), // Can be empty but non blocker (already logged)
                    (k, v) -> v == null ? eqTD : v + eqTD
                );
            }
        }
        
        return nbHoursSumByDiscipline;
    }
    
    private boolean _isHoldByOrgUnit(CoursePart coursePart, OrgUnit orgUnit)
    {
        Course courseHolder = coursePart.getCourseHolder();
        if (courseHolder == null)
        {
            getLogger().error("Les heures d'enseignement '{}' ({}) n'ont pas d'ELP porteur.", coursePart.getTitle(), coursePart.getCode());
            return false;
        }
        
        Set<OrgUnit> stepOrgUnits = _retrieveOrgUnits(courseHolder);
        for (OrgUnit stepOrgUnit : stepOrgUnits)
        {
            if (!_matchOrgUnit(stepOrgUnit, orgUnit))
            {
                return false;
            }
        }
        
        return !stepOrgUnits.isEmpty();
    }
    

    private Set<OrgUnit> _retrieveOrgUnits(ProgramItem programItem)
    {
        Set<OrgUnit> directOrgUnits = _getDirectOrgUnits(programItem);
        return directOrgUnits.isEmpty()
                ? _retrieveOrgUnitsFromParents(programItem)
                : directOrgUnits;
    }
    
    private Set<OrgUnit> _retrieveOrgUnitsFromParents(ProgramItem programItem)
    {
        return _odfHelper.getParentProgramItems(programItem)
                .stream()
                .map(this::_retrieveOrgUnits)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }
    
    private Set<OrgUnit> _getDirectOrgUnits(ProgramItem programItem)
    {
        List<String> orgUnitIds = programItem.getOrgUnits();
        return _resolveAll(orgUnitIds);
    }

    private Set<OrgUnit> _resolveAll(Collection<String> orgUnitIds)
    {
        return orgUnitIds
                .stream()
                .map(this::_resolve)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }
    
    private OrgUnit _resolve(String id)
    {
        try
        {
            return _resolver.resolveById(id);
        }
        catch (AmetysRepositoryException e)
        {
            // Silently fail
            return null;
        }
    }
    
    private boolean _matchOrgUnit(OrgUnit orgUnit, OrgUnit orgUnitToMatch)
    {
        if (orgUnit.equals(orgUnitToMatch))
        {
            return true;
        }
        
        return Optional.of(orgUnit)
            .map(OrgUnit::getParentOrgUnit)
            .map(parent -> _matchOrgUnit(parent, orgUnitToMatch))
            .orElse(false);
    }
    
    private String _getDisciplineEnseignement(CoursePart coursePart)
    {
        String disciplineId = _getDisciplineEnseignementAttribute(coursePart)
            .or(() -> _getDisciplineEnseignementAttribute(coursePart.getCourseHolder()))
            .orElse(StringUtils.EMPTY);
        
        if (StringUtils.isEmpty(disciplineId))
        {
            getLogger().warn("Les heures d'enseignement '{}' ({}) n'ont pas de discipline associée.", coursePart.getTitle(), coursePart.getCode());
        }
        
        return disciplineId;
    }

    private Optional<String> _getDisciplineEnseignementAttribute(Content content)
    {
        return Optional.ofNullable(content)
            .map(c -> c.<ContentValue>getValue("disciplineEnseignement"))
            .map(ContentValue::getContentId);
    }
    
    /**
     * Get the {@link CoursePart} of the given program item. Only get course parts when the {@link Course} doesn't have any children.
     * @param programItem The program item
     * @return A {@link List} of {@link CoursePart}
     */
    private List<CoursePart> _getCoursePartsForProgramItem(ProgramItem programItem)
    {
        // ELP de dernier niveau
        if (programItem instanceof Course course && course.getCourseLists().isEmpty())
        {
            return course.getCourseParts();
        }
        
        // Sinon on explore la suite de l'arborescence...
        List<CoursePart> courseParts = new ArrayList<>();
        _odfHelper.getChildProgramItems(programItem).forEach(child -> courseParts.addAll(_getCoursePartsForProgramItem(child)));
        return courseParts;
    }
}
