/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.time.LocalDate;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper;

/**
 * Class to generate "règlement des études" extract as DOC.
 */
public class ReglementEtudesExtract extends AbstractExtract
{
    /** The pilotage status helper */
    protected PilotageStatusHelper _pilotageStatusHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _pilotageStatusHelper = (PilotageStatusHelper) manager.lookup(PilotageStatusHelper.ROLE);
    }
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        return "reglementetudes";
    }
    
    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        // Check the type of the program item (currently, by default, only programs are accepted)
        ProgramItem programItem = _resolver.resolveById(programItemId);
        if (!(programItem instanceof Program program))
        {
            throw new UnsupportedOperationException("The report '" + getType(reportParameters, true) + "' can be launch only on programs.");
        }
        
        try
        {
            handler.startDocument();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            XMLUtils.startElement(handler, "report", attrs);
            
            _saxReport(handler, program, _odfHelper.getChildSubPrograms(program));
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Règlements des études' extract for program '{}' ({})", program.getTitle(), program.getCode(), e);
        }
    }
    
    private void _saxReport(ContentHandler handler, Program program, Set<SubProgram> subPrograms) throws SAXException
    {
        _saxProgram(handler, program);
        
        for (SubProgram subProgram : subPrograms)
        {
            _saxSubProgram(handler, subProgram);
        }
    }
    
    private void _saxProgram(ContentHandler handler, Program program) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("title", program.getTitle());
        
        LocalDate date = _pilotageStatusHelper.getCFVUValidationDate(program);
        if (date != null)
        {
            attr.addCDATAAttribute("CFVUDate", DateUtils.localDateToString(date));
        }
        
        XMLUtils.startElement(handler, "program", attr);
        
        // Catalog
        XMLUtils.createElement(handler, "catalog", program.getCatalog());
        
        // Degree
        String degreeId = program.getDegree();
        if (StringUtils.isNotEmpty(degreeId))
        {
            XMLUtils.createElement(handler, "degree", _refTableHelper.getItemLabel(degreeId, program.getLanguage()));
        }

        // Mention
        String mentionId = program.getMention();
        if (StringUtils.isNotEmpty(mentionId))
        {
            XMLUtils.createElement(handler, "mention", _refTableHelper.getItemLabel(mentionId, program.getLanguage()));
        }
        
        // Domain
        String[] domains = program.getDomain();
        for (String domain : domains)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("code", _refTableHelper.getItemCode(domain));
            XMLUtils.createElement(handler, "domain", attrs, _refTableHelper.getItemLabel(domain, program.getLanguage()));
        }
        
        // Règlement des études
        program.dataToSAX(handler, "reglementEtudes");
        
        XMLUtils.endElement(handler, "program");
    }
    
    private void _saxSubProgram(ContentHandler handler, SubProgram subprogram) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("title", subprogram.getTitle());
        XMLUtils.startElement(handler, "subprogram", attr);
        
        subprogram.dataToSAX(handler, "reglementEtudes");
        
        XMLUtils.endElement(handler, "subprogram");
    }
}
