/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.plugins.odfpilotage.rule.RulesManager;

/**
 * Observer to add already modified rule in transient variables of modified thematic
 */
public class UpdateThematicRulesStep1Observer extends AbstractRulesStepObserver
{
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_MODIFYING;
    }
    
    @Override
    protected boolean supportsContent(Content content)
    {
        return _contentTypesHelper.isInstanceOf(content, RulesManager.THEMATIC_CONTENT_TYPE);
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        List<String> degreeIds = Stream.of(content.getValue(RulesManager.THEMATIC_DEGREE, false, new ContentValue[0]))
                .map(ContentValue::getContentId)
                .toList();
        _addOldDegreeIds(content, degreeIds);
        
        List<String> regimeIds = Stream.of(content.getValue(RulesManager.THEMATIC_REGIME, false, new ContentValue[0]))
                .map(ContentValue::getContentId)
                .toList();
        _addOldRegimeIds(content, regimeIds);
        
        String nbSessions = content.getValue(RulesManager.THEMATIC_NB_SESSIONS);
        _addOldNbSessions(content, nbSessions);
        
        List<String> modifiedRuleCodes = Optional.ofNullable(content)
                .map(c -> c.<ModifiableIndexableRepeater>getValue(RulesManager.THEMATIC_RULES))
                .map(ModifiableIndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .filter(e -> "MODIFIED".equals(e.getValue(RulesManager.RULE_STATUS)))
                .map(e -> e.<String>getValue(RulesManager.RULE_CODE))
                .toList();
        _addOldModifiedRuleCodes(content, modifiedRuleCodes);
    }
}
