/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Pilotage status actions
 * @private
 */
Ext.define('Ametys.plugins.odf.pilotage.actions.PilotageStatusButtonActions', {
	singleton: true,
	
	/**
     * @private
     * @property {String} _status The pilotage status of the button
     */
	
	/**
     * @private
     * @property {String} _contentId The content id to handle
     */
	
	/**
	 * Action function to be called by the controller.
	 * Will change the pilotage status.
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 * @param {Boolean} state the state of the button
	 */
	act: function (controller, state)
	{
		this._status = controller.status;
		this._contentId = controller.getContentIds()[0]; // no multiselection
		
		controller.serverCall (
				'getSharedChildName', 
				[this._contentId], 
				Ext.bind(this._actCB, this, [controller, state], 1), 
				{ refreshing: true }
		);
	},
	
	/**
	 * Callback function called after #_act is processed.
	 * @param {String[]} childNames The children
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 * @param {Boolean} state the state of the button
	 * @private
	 */
	_actCB: function (childNames, controller, state)
	{
		if (childNames.length > 0)
		{
			var msg = "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_MODIFY_WORKFLOW_WARNING_MSG1}}";
			msg += "<ul>";
			for (var i in childNames)
			{
				msg += "<li>" + childNames[i] + "</li>";
			}
			msg += "</ul>";
			msg += "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_MODIFY_WORKFLOW_WARNING_MSG2}}<br/>";

			if (state)
			{
				msg += "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_MODIFY_WORKFLOW_WARNING_MSG3}}";
				Ametys.Msg.confirm(controller.getInitialConfig("dialog-title"),
						msg,
						Ext.bind(this._openCommentBox, this, [controller, state], 1),
						this
				);
			}
			else
			{
				msg += controller.getInitialConfig("dialog-back-msg");
				Ametys.Msg.confirm("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_GO_BACK_CONFIRM_TITLE}}", 
						msg,
						Ext.bind(this._removePilotageStatus, this, [controller], 1),
						this
				);
			}
		}
		else
		{
			this._openCommentBox("yes", controller, state);
		}
	},
	
	/**
	 * Callback function called after #_actCB is processed.
	 * @param {String} answer The answer of the confirmation box
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 * @param {Boolean} state the state of the button
	 * @private
	 */
	_openCommentBox: function (answer, controller, state)
	{
		if (answer == "yes")
		{
			if (state)
			{
				controller.serverCall (
						'getValidationDate', 
						[this._contentId], 
						Ext.bind(this._openCommentBoxCB, this, [controller], 1), 
						{ refreshing: true }
				);
			}
			else
			{
				Ametys.Msg.confirm("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_GO_BACK_CONFIRM_TITLE}}", 
						controller.getInitialConfig("dialog-back-msg"),
						Ext.bind(this._removePilotageStatus, this, [controller], 1),
						this
				);
			}
		}
	},
	
	/**
	 * Callback function called after #_openCommentBox is processed.
	 * @param {Object} date The JSON result 
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 * @private
	 */
	_openCommentBoxCB: function (date, controller)
	{
        this._commentBox = Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
            dialogTitle: controller.getInitialConfig("dialog-title"),
            dialogIconCls: controller.getInitialConfig("dialog-glyph"),
            dateValue: new Date(),
            minDate: new Date(date),
            minDateText: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_GO_NEXT_BOX_MIN_DATE_ERROR}}",
            maxDate: new Date(),
            maxDateText: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_WORKFLOW_BUTTON_GO_NEXT_BOX_MAX_DATE_ERROR}}",
            validateCallback: Ext.bind(this._setPilotageStatus, this, [controller], 2),
            autoCloseDialog: false
        })
	},
	
	/**
	 * @private 
	 * Set the pilotage status
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 */
	_setPilotageStatus: function(date, comment, controller)
	{
		controller.serverCall (
				'setPilotageStatus', 
				[this._contentId, this._status, date, comment], 
				Ext.bind(this._updatePilotageStatusCB, this), 
				{ refreshing: true, errorMessage: true }
		);
	},
	
	/**
	 * @private 
	 * Remove the pilotage status
	 * @param {String} answer The answer of the confirmation box
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 */
	_removePilotageStatus: function(answer, controller)
	{
		if (answer == "yes")
		{
			controller.serverCall (
					'removePilotageStatus', 
					[this._contentId, this._status], 
					Ext.bind(this._updatePilotageStatusCB, this), 
					{refreshing: true, errorMessage: true}
			);
		}
	},
	
	/**
	 * Callback function called after #_removePilotageStatus or #_setPilotageStatus is processed.
	 * @param {Object} response The JSON result 
	 * @private
	 */
	_updatePilotageStatusCB : function(response)
	{
		if (this._commentBox)
		{
			this._commentBox.close();
		}

		Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            parameters: {major: true},
            targets: {
                id: Ametys.message.MessageTarget.CONTENT,
                parameters: {
                    ids: [this._contentId]
                }
            }
        });
	}
});
