/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.clientsideelement;

import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.i18n.I18nUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.tree.ODFTreeToolClientSideElement;

/**
 * Override the TreeTooClientSideElement to retrieve informations on enseignement natures and nb of hours
 */
public class CostModelingClientSideElement extends ODFTreeToolClientSideElement implements Contextualizable
{
    /** The Avalon role name */
    public static final String ROLE = CostModelingClientSideElement.class.getName();
    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    /** The context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _refTableHelper = (OdfReferenceTableHelper) smanager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    protected void _lazyConfigure()
    {
        super._lazyConfigure();
        _script.getParameters().put("enseignement-natures", getEnseignementNatures());
    }
    
    /**
     * Retrieve and get all teaching nature in a JSON map
     * @return a JSON map containing the grid structure for all teaching nature
     */
    public Map<String, Map<String, Object>> getEnseignementNatures()
    {
        Map objectModel = ContextHelper.getObjectModel(_context);
        Locale locale = I18nUtils.findLocale(objectModel, "locale", null, Locale.getDefault(), true);
        String lang = locale.getLanguage();
        
        return _refTableHelper.getItems(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE)
            .stream()
            .sequential()
            .sorted(Comparator
                .comparingLong(this::_orderFromCategory)
                .thenComparingLong(OdfReferenceTableEntry::getOrder)
                .thenComparing(OdfReferenceTableEntry::getCode)
            )
            .collect(
                Collectors.toMap(
                    OdfReferenceTableEntry::getCode,
                    entry -> Map.of(
                        "id", entry.getId(),
                        "label", entry.getLabel(lang),
                        "archived", entry.isArchived()
                    ),
                    (a, b) -> a,
                    LinkedHashMap::new
                )
            );
    }
    
    private Long _orderFromCategory(OdfReferenceTableEntry referenceTableEntry)
    {
        return Optional.of(referenceTableEntry)
                .map(OdfReferenceTableEntry::getContent)
                .map(c -> c.<ContentValue>getValue("category"))
                .flatMap(ContentValue::getContentIfExists)
                .map(c -> c.getValue("order", false, Long.MAX_VALUE))
                .orElse(Long.MAX_VALUE);
    }
}
