/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.cost.CostComputationComponent;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.plugins.odfpilotage.cost.entity.Effectives;
import org.ametys.plugins.odfpilotage.cost.entity.EqTD;
import org.ametys.plugins.odfpilotage.cost.entity.ProgramItemData;
import org.ametys.plugins.odfpilotage.cost.entity.VolumesOfHours;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ModelItem;

/**
 * Pilotage report for cost modeling synthesis
 */
public class SyntheseReport extends AbstractReport
{
    /** CalculerEffectifComponent */
    protected CostComputationComponent _costComputationComponent;
    
    /** The I18N utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _costComputationComponent = (CostComputationComponent) manager.lookup(CostComputationComponent.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        return "synthese";
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_XLS;
    }

    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_XLS);
    }

    @Override
    protected void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
        CostComputationData costData = _costComputationComponent.computeCostsOnOrgUnit(orgUnit, catalog, lang, false);
        
        try
        {
            handler.startDocument();
        
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            XMLUtils.startElement(handler, "report", attrs);

            _reportHelper.saxNaturesEnseignement(handler, getLogger());
            _writeLines(handler, costData, orgUnit);
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Synthese' report for orgunit '{}'", orgUnit.getTitle(), e);
        }
    }
    
    /**
     * Write lines content of the report
     * @param handler the handler
     * @param costData informations about the capacity
     * @param orgUnit the orgUnit
     * @throws SAXException to handle XMLUtils exceptions
     */
    private void _writeLines(ContentHandler handler, CostComputationData costData, OrgUnit orgUnit) throws SAXException
    {
        XMLUtils.startElement(handler, "lines");
        
        Double orgUnitEffective = 0D;
        Map<Program, Set<Container>> stepsByProgram = _getStepsByProgram(costData);
        for (Program program : stepsByProgram.keySet())
        {
            Double programEffective = 0D;
            Set<Container> steps = stepsByProgram.get(program);
            for (Container step : steps)
            {
                programEffective += _writeStepLine(handler, costData, program, step);
            }
            
            _writeProgramLine(handler, costData, program, programEffective);
            orgUnitEffective += programEffective;
        }
        
        _writeOrgUnitLine(handler, costData, orgUnit, orgUnitEffective);
        
        XMLUtils.endElement(handler, "lines");
    }
    
    private Map<Program, Set<Container>> _getStepsByProgram(CostComputationData costData)
    {
        String yearId = _odfHelper.getYearId().orElse(null);
        Map<Program, Set<Container>> stepsByProgram = new TreeMap<>(ReportHelper.CONTENT_TITLE_COMPARATOR);
        
        // Get steps on each program
        for (Program program : costData.getComputedPrograms())
        {
            stepsByProgram.put(
                program,
                _getChildrendStepsForProgram(program, yearId)
            );
        }
        
        return stepsByProgram;
    }

    private Set<Container> _getChildrendStepsForProgram(Program program, String yearId)
    {
        // Useless function if year id is null
        if (yearId == null)
        {
            return Set.of();
        }
        
        // Get the steps recursively of the current program
        // Then group identical steps and add the number of occurrences into a map, ordering by title
        return _getChildrendStepsForProgramItem(program, yearId)
            .collect(Collectors.toCollection(() -> new LinkedHashSet<>()));
    }
    
    private Stream<Container> _getChildrendStepsForProgramItem(ProgramItem programItem, String yearId)
    {
        // Search if the current element is a container and is of type year
        if (_odfHelper.isContainerOfNature(programItem, yearId))
        {
            return Stream.of((Container) programItem);
        }
        
        // In all other cases, search in the child elements
        return _odfHelper.getChildProgramItems(programItem)
            .stream()
            .flatMap(child -> _getChildrendStepsForProgramItem(child, yearId));
    }

    private Double _writeStepLine(ContentHandler handler, CostComputationData costData, Program program, Container step) throws AmetysRepositoryException, SAXException
    {
        XMLUtils.startElement(handler, "line");
        XMLUtils.createElement(handler, "type", "container");
        XMLUtils.createElement(handler, "diplome", program.getTitle());
        XMLUtils.createElement(handler, "annee", step.getTitle());
        // Regexp = programName/(.*/)*stepName
        Double stepEffective = _writeValues(handler, costData, step, program.getName() + ModelItem.ITEM_PATH_SEPARATOR + "(.*" + ModelItem.ITEM_PATH_SEPARATOR + ")*" + step.getName(), null);
        XMLUtils.endElement(handler, "line");
        return stepEffective;
    }
    
    private void _writeProgramLine(ContentHandler handler, CostComputationData costData, Program program, Double effective) throws AmetysRepositoryException, SAXException
    {
        XMLUtils.startElement(handler, "line");
        XMLUtils.createElement(handler, "type", "program");
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        i18nParams.put("program",  new I18nizableText(program.getTitle()));
        XMLUtils.createElement(handler, "diplome", _i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_SYNTHESIS_TOTAL", i18nParams), program.getLanguage()));
        // Regexp = programName
        _writeValues(handler, costData, program, program.getName(), effective);
        XMLUtils.endElement(handler, "line");
    }
    
    private void _writeOrgUnitLine(ContentHandler handler, CostComputationData costData, OrgUnit orgUnit, Double effective) throws SAXException
    {
        XMLUtils.startElement(handler, "line");
        XMLUtils.createElement(handler, "type", "orgunit");
        XMLUtils.createElement(handler, "diplome", _i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_SYNTHESIS_GRAND_TOTAL"), orgUnit.getLanguage()));
        // Regexp = orgUnitName
        _writeValues(handler, costData, orgUnit, orgUnit.getName(), effective);
        XMLUtils.endElement(handler, "line");
    }
    
    private Double _writeValues(ContentHandler handler, CostComputationData costData, Content item, String regexPath, Double effective) throws SAXException
    {
        ProgramItemData itemData = costData.get(item.getId());
        Double localEffective = effective;
        
        if (itemData != null)
        {
            Pattern filterPattern = Pattern.compile(regexPath);
            
            // Volumes of hours
            VolumesOfHours volumesOfHours = itemData.getVolumesOfHours();
            if (volumesOfHours != null)
            {
                for (Entry<String, Double> volumeOfHours : volumesOfHours.getVolumes().entrySet())
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("id", volumeOfHours.getKey());
                    XMLUtils.createElement(handler, "volume", attrs, String.valueOf(volumeOfHours.getValue()));
                }
                XMLUtils.createElement(handler, "total", String.valueOf(volumesOfHours.getTotal()));
            }
            
            // EqTD
            EqTD eqTD = itemData.getEqTD();
            if (eqTD != null)
            {
                Double localEqTD = Optional.ofNullable(eqTD.getLocalEqTD())
                    .map(map -> _sumValuesWithFilteredName(map, filterPattern))
                    .orElse(0D);
                XMLUtils.createElement(handler, "eqtdLocal", String.valueOf(localEqTD));
                
                Double proratedEqTD = Optional.ofNullable(eqTD.getProratedEqTD())
                    .map(map -> _sumValuesWithFilteredName(map, filterPattern))
                    .orElse(0D);
                XMLUtils.createElement(handler, "eqtdProrated", String.valueOf(proratedEqTD));
            }
            
            // Effectives
            // If there is an effective in the method parameters, it is because it is the sum of all steps effectives
            // Otherwise, we are on a step and we get the sum of the local effectives in the program (a step can be shared in a program)
            if (localEffective == null)
            {
                localEffective = Optional.ofNullable(itemData.getEffectives())
                    .map(Effectives::getLocalEffectiveByPath)
                    .map(map -> _sumValuesWithFilteredName(map, filterPattern))
                    .orElse(0D);
            }
            XMLUtils.createElement(handler, "effectif", String.valueOf(Math.round(localEffective)));

            // H/E ratio
            Optional<Double> heRatio = _filterMapValuesWithFilteredName(itemData.getHeRatios(), filterPattern)
                .flatMap(Optional::stream)
                .reduce(Double::sum);
            if (heRatio.isPresent())
            {
                XMLUtils.createElement(handler, "heRatio", String.valueOf(heRatio.get()));
            }
        }
        
        return Optional.ofNullable(localEffective).orElse(0D);
    }
    
    /**
     * Filter a map on its keys with the filter pattern, then sum all values.
     * @param mapToFilterAndSum The map to filter and sum
     * @param filterPattern The filter pattern
     * @return The sum after filtering
     */
    private Double _sumValuesWithFilteredName(Map<String, Double> mapToFilterAndSum, Pattern filterPattern)
    {
        return _filterMapValuesWithFilteredName(mapToFilterAndSum, filterPattern)
            .reduce(0D, Double::sum);
    }
    
    private <T> Stream<T> _filterMapValuesWithFilteredName(Map<String, T> mapToFilter, Pattern filterPattern)
    {
        return mapToFilter
            .entrySet()
            .stream()
            .filter(entry -> filterPattern.matcher(entry.getKey()).matches())
            .map(Entry::getValue);
    }
}
