/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.property;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.model.properties.AbstractIndexableContentProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.ThematicsHelper;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;

/**
 * {@link Property} for overriden thematics on a {@link Container}
 */
public class OverridenThematicsProperty extends AbstractIndexableContentProperty<Content>
{
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _thematicsHelper = (ThematicsHelper) manager.lookup(ThematicsHelper.ROLE);
    }
    
    @Override
    protected Set< ? extends ModifiableContent> _getLinkedContents(Content content)
    {
        if (content instanceof Container container)
        {
            String catalog = container.getCatalog();
            return Optional.of(content)
                .map(c -> c.<ModelAwareRepeater>getValue(RulesManager.CONTAINER_RULES))
                .map(ModelAwareRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .map(e -> e.<String>getValue(RulesManager.RULE_CODE))
                .map(rCode -> StringUtils.substringBefore(rCode, "-"))
                .distinct()
                .map(tCode -> _thematicsHelper.getThematic(catalog, tCode))
                .filter(Objects::nonNull)
                .filter(ModifiableContent.class::isInstance)
                .map(ModifiableContent.class::cast)
                .collect(Collectors.toSet());
        }
        
        return null;
    }
    
    @Override
    public boolean isMultiple()
    {
        return true;
    }
    
    public String getContentTypeId()
    {
        return RulesManager.THEMATIC_CONTENT_TYPE;
    }
}
