 /*
*  Copyright 2025 Anyware Services
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/

// For each thematic:
//  - Verify the consistency of rules code and update it if needed
//  - Update the counter "TS[X]-ruleNumber" with the right value

// For each container:
//  - Verify the consistency of additional thematics code and update it if needed
//  - Update the counter "TA" with the right value
//  - For each additional thematic:
//      - Verify the consistency of rules code and update it if needed
//      - Update the counter "TA[X]-ruleNumber" with the right value
//  - For each additional rules of existing thematic:
//      - Verify the consistency of rules code and update it if needed
//      - Update the counter "TS[X]-ruleNumber" with the right value

const Optional = Java.type("java.util.Optional");
const List = Java.type("java.util.List");
const Integer = Java.type("java.lang.Integer");
const HashSet = Java.type("java.util.HashSet");
const StringUtils = Java.type("org.apache.commons.lang3.StringUtils");
const CODE_PATTERN = java.util.regex.Pattern.compile("^(?:.*?)([0-9]+)$");

let thematics = Repository.query(`//element(*, ametys:content)[@ametys-internal:contentType = 'odf-enumeration.Thematic']`);
thematics.forEach(
    thematic =>
    {
        logger.info(`[${thematic.getValue("catalog")}] Update thematic [${thematic.getValue("code")}] ${thematic.getTitle()} (${thematic.getId()})`);
        Content.migrate(
            thematic,
            [updateThematic],
            false,
            false,
            false,
            true
        );
    }
);

let containers = Repository.query(`//element(*, ametys:container)`);
containers.forEach(
    container =>
    {
        let hasThematics = !_getEntries(container, "thematics").isEmpty();
        let hasAdditionalRules = _getEntries(container, "rules").stream()
                                                                .anyMatch(r => StringUtils.isNotBlank(r.getValue("thematicCode")));
        
        if (hasThematics || hasAdditionalRules)
        {
            logger.info(`[${container.getValue("catalog")}] Update container [${container.getValue("code")}] ${container.getTitle()} (${container.getId()})`);
            Content.migrate(
                container,
                [updateContainer],
                false,
                false,
                false,
                true
            );
        }
    }
);

function updateThematic(content)
{
    _updateEntries(content, _getEntries(content, "rules"), content.getValue("code") + "-", content.getValue("code") + "-ruleNumber");
}

function updateContainer(content)
{
    // Update additional thematics
    let thematics = _getEntries(content, "thematics");
    _updateEntries(content, thematics, "TA", "TA");
    
    // Update additional rules of additionalThematics
    for (let thematic of thematics)
    {
        _updateEntries(content, _getEntries(thematic, "rules"), thematic.getValue("code") + "-", thematic.getValue("code") + "-ruleNumber");
    }
    
    // Update additional rules of existing thematics
    let thematicsWithAdditionalRules = _getEntries(content, "rules").stream()
                                                                    .map(e => e.getValue("thematicCode"))
                                                                    .filter(c => StringUtils.isNotBlank(c))
                                                                    .toList();
    for (let thematicCode of thematicsWithAdditionalRules)
    {
        let rules = _getEntries(content, "rules").stream()
                                                 .filter(e => thematicCode == e.getValue("thematicCode"))
                                                 .toList();
        _updateEntries(content, rules, thematicCode + "-A", thematicCode + "-ruleNumber");
    }
}

function _updateEntries(content, entries, prefix, counterName)
{
    let entriesWithExistingCode = [];
    let existingNumbers = new HashSet();
    let currentMaxNumber = 0;
    
    // Verify number consistency
    for (let entry of entries)
    {
        let code = entry.getValue("code");
        let currentNumber = _getNumber(code);
        
        if (existingNumbers.add(currentNumber))
        {
            if (currentNumber > currentMaxNumber)
            {
                currentMaxNumber = currentNumber;
            }
            
            if (code != prefix + currentNumber)
            {
                logger.info(` - Update "${code}" to ${prefix + currentNumber}`);
                entry.setValue("code", prefix + currentNumber);
            }
        }
        // The same code has been defined several times on this thematic
        else
        {
            entriesWithExistingCode.push(entry);
        }
    }
    
    // Update entries with inconsistent code
    for (let entry of entriesWithExistingCode)
    {
        currentMaxNumber++;
        let newCode = prefix + currentMaxNumber;
        logger.info(` - Entry with code ${entry.getValue("code")} appears twice, replace the code by ${newCode}`);
        entry.setValue("code", newCode);
    }
    
    // Update the entry number if needed
    let savedMaxNumber = content.getInternalDataHolder().getValue(counterName, null);
    if (savedMaxNumber == null || savedMaxNumber < currentMaxNumber)
    {
        logger.info(` - Set "${counterName}" to ${currentMaxNumber}`);
        content.getInternalDataHolder().setValue(counterName, currentMaxNumber);
    }
}

function _getEntries(dataHolder, repeaterName)
{
    return Optional.ofNullable(dataHolder.getRepeater(repeaterName))
                   .map(r => r.getEntries())
                   .orElseGet(() => List.of())
}

function _getNumber(code)
{
    let matcher = CODE_PATTERN.matcher(code);
    return matcher.matches() ? Integer.valueOf(matcher.group(1)) : null;
}
