/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.enumerators;

import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * {@link Enumerator} for the steps holder
 */
public class StepHolderEnumerator implements Enumerator<ContentValue>, Serviceable, Contextualizable, Configurable
{
    private Context _context;
    private AmetysObjectResolver _resolver;
    private PilotageHelper _pilotageHelper;
    private ODFHelper _odfHelper;
    
    /**
     * If <code>true</code> and the context is not an ELP, we get all the steps.
     */
    private boolean _usedForSearch;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration customConf = configuration.getChild("enumeration").getChild("custom-enumerator");
        _usedForSearch = customConf.getChild("used-for-search").getValueAsBoolean(false);
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }

    public I18nizableText getEntry(ContentValue value) throws Exception
    {
        try
        {
            return _getDisplayName((Container) value.getContent());
        }
        catch (UnknownAmetysObjectException e)
        {
            // Ignore.
        }
        
        return null;
    }
    
    public Map<ContentValue, I18nizableText> getEntries() throws Exception
    {
        Optional<Course> course = Optional.of(_context)
            .map(ContextHelper::getRequest)
            .map(r -> r.getAttribute(Content.class.getName()))
            .filter(Course.class::isInstance)
            .map(Course.class::cast);
            
        // course est nul si on est hors contexte de l'édition
        Stream<Container> containers = course.isPresent()
            ? _pilotageHelper.getSteps(course.get()).stream()
            : (_usedForSearch ? _getAllSteps() : Stream.empty());
        
        return containers
            .collect(Collectors.toMap(ContentValue::new, this::_getDisplayName));
    }

    private Stream<Container> _getAllSteps()
    {
        return _odfHelper.getYearId()
            .map(anneeId -> new StringExpression("nature", Operator.EQ, anneeId))
            .map(anneeExpression -> new AndExpression(new ContentTypeExpression(Operator.EQ, ContainerFactory.CONTAINER_CONTENT_TYPE), anneeExpression))
            .map(expression -> QueryHelper.getXPathQuery(null, ContainerFactory.CONTAINER_NODETYPE, expression))
            .map(_resolver::<Container>query)
            .map(AmetysObjectIterable::stream)
            .orElseGet(Stream::of);
    }
    
    private I18nizableText _getDisplayName(Container container)
    {
        StringBuilder displayName = new StringBuilder();
        
        // Display code
        displayName.append("[");
        displayName.append(container.getDisplayCode());
        displayName.append("] ");
        
        displayName.append(container.getTitle());
        
        return new I18nizableText(displayName.toString());
    }
}
