/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.property;

import java.util.List;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.model.properties.AbstractIndexableStaticProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link Property} for MCC workflow status on a {@link Container}
 */
public class ContainerMCCWorkflowStatusProperty extends AbstractIndexableStaticProperty<String, String, Container>
{
    /** Name of property storing MCC workflow status */
    public static final String PROPERTY_NAME = "mccWorkflowStatus";
    
    private MCCWorkflowHelper _mccWorkflowHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _mccWorkflowHelper = (MCCWorkflowHelper) smanager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    public Object getValue(Container container)
    {
        List<MCCWorkflowStep> currentSteps = _mccWorkflowHelper.getCurrentSteps(container);
        
        return currentSteps.stream().map(MCCWorkflowStep::name).toArray(String[]::new);
    }
    
    @Override
    public Enumerator<String> getEnumerator()
    {
        StaticEnumerator<String> enumerator = new StaticEnumerator<>();
        
        Stream.of(MCCWorkflowStep.values()).forEach(step -> enumerator.add(step.label(), step.name()));
        
        return enumerator;
    }

    @Override
    public boolean isMultiple()
    {
        return true;
    }
    
    @Override
    protected String getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
}
