/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.container;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.rule.AbstractThematicsGenerator;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.type.DataContext;

/**
 * Generator to sax rules for a given container.
 */
public class ContainerRulesGenerator extends AbstractThematicsGenerator
{
    /** The rules manager */
    protected RulesManager _rulesManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _rulesManager = (RulesManager) smanager.lookup(RulesManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        contentHandler.startDocument();
        
        if (RulesManager.isRulesEnabled())
        {
            String containerId = request.getParameter("programItem");
            if (StringUtils.isNotBlank(containerId))
            {
                Container container = _resolver.resolveById(containerId);
                if (_rulesManager.hasHandleRulesRight(container))
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("canEditRule", String.valueOf(_rulesManager.isRulesModificationAvailable(container)));
                    XMLUtils.startElement(contentHandler, "thematics", attrs);

                    _saxContainer(request, container);
                
                    XMLUtils.startElement(contentHandler, "general");
                    for (Content thematic : _thematicsHelper.getCompatibleThematics(container))
                    {
                        _saxContent(thematic);
                    }
                    XMLUtils.endElement(contentHandler, "general");
                    
                    XMLUtils.startElement(contentHandler, "container-rules");
                    _saxContainerRules(container);
                    XMLUtils.endElement(contentHandler, "container-rules");
                    
                    XMLUtils.startElement(contentHandler, "specific");
                    _saxSpecificRules(container);
                    XMLUtils.endElement(contentHandler, "specific");

                    XMLUtils.endElement(contentHandler, "thematics");
                }
            }
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * Sax the container information with current thematic and current rule
     * @param request the request
     * @param container the container
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxContainer(Request request, Container container) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", container.getId());

        String thematicId = request.getParameter("thematicId");
        if (StringUtils.isNotBlank(thematicId))
        {
            attrs.addCDATAAttribute("thematicId", thematicId);
            String ruleId = request.getParameter("ruleId");
            if (StringUtils.isNotBlank(ruleId))
            {
                attrs.addCDATAAttribute("ruleId", ruleId);
            }
        }
        
        XMLUtils.startElement(contentHandler, "container", attrs);
        View view = View.of(
            container.getModel(),
            "title",
            "catalog",
            PilotageHelper.CONTAINER_MCC_NUMBER_OF_SESSIONS,
            PilotageHelper.CONTAINER_MCC_REGIME
        );
        DataContext context = RepositoryDataContext.newInstance().withObject(container);
        container.dataToSAX(contentHandler, view, context);
        
        XMLUtils.endElement(contentHandler, "container");
    }
    
    private void _saxContainerRules(Container content) throws SAXException
    {
        ViewItemAccessor resultItems =  _contentTypeExtensionPoint.getExtension(ContainerFactory.CONTAINER_CONTENT_TYPE).getView("thematics");
        
        DataContext context = RepositoryDataContext.newInstance().withObject(content);
        content.dataToSAX(contentHandler, resultItems, context);
    }
    
    private void _saxSpecificRules(Container content) throws SAXException
    {
        ViewItemAccessor resultItems =  _contentTypeExtensionPoint.getExtension(ContainerFactory.CONTAINER_CONTENT_TYPE).getView(RulesManager.SPECIFIC_RULE_VIEW);
        if (resultItems != null)
        {
            DataContext context = RepositoryDataContext.newInstance().withObject(content);
            content.dataToSAX(contentHandler, resultItems, context);
        }
    }
}
