/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.schedulable.helper;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.quartz.JobDataMap;

import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.impl.AbstractMCCReport;
import org.ametys.plugins.odfpilotage.report.impl.MCCReport;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.runtime.model.ElementDefinition;

/**
 * Helper for the program and orgunit schedulables of the MCC report.
 */
public final class MCCReportSchedulableHelper
{
    /** The key to define if we want a result by program or year */
    public static final String JOBDATAMAP_ONE_FILE_BY_PROGRAM_KEY = "oneFileByProgram";
    
    /** The key for the definitive versions */
    public static final String JOBDATAMAP_DEFINITIVE_KEY = "definitive";
    
    /** The key for the alternatives versions */
    public static final String JOBDATAMAP_RULES_VERSION_KEY = "rulesVersion";
    
    /** The key to define if we want only year belonging to the current orgUnit */
    public static final String JOBDATAMAP_FILTER_ON_CURRENT_ORGUNIT = "filterOnCurrentOrgUnit";
    
    /** The key to define the orgUnit to to get only year belonging to it */
    public static final String JOBDATAMAP_FILTER_BY_ORGUNIT = "filterByOrgUnit";
    
    private MCCReportSchedulableHelper()
    {
        // Empty constructor
    }
    
    /**
     * Get the specific additionnal parameters for MCC report.
     * @param jobDataMap The map of parameters
     * @return The additional parameters
     */
    public static Map<String, String> getReportParameters(JobDataMap jobDataMap)
    {
        Map<String, String> reportParameters = new HashMap<>();
        
        boolean isDefinitive = false;
        // Definitive is not present for MCC diff
        if (jobDataMap.containsKey(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_DEFINITIVE_KEY))
        {
            isDefinitive = jobDataMap.getBoolean(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_DEFINITIVE_KEY);
            reportParameters.put(MCCReport.PARAMETER_DEFINITIVE, Boolean.toString(isDefinitive));
        }
        
        // If we get the definitive version, force export to PDF
        if (isDefinitive)
        {
            if (jobDataMap.containsKey(Scheduler.PARAM_VALUES_PREFIX + AbstractReportSchedulable.JOBDATAMAP_OUTPUT_FORMAT_KEY))
            {
                reportParameters.put(PilotageReport.PARAMETER_OUTPUT_FORMAT, PilotageReport.OUTPUT_FORMAT_PDF);
            }
        }
        // Get the value only if not definitive
        // If rules are active, there is no oneFileByProgram parameter
        else if (jobDataMap.containsKey(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_ONE_FILE_BY_PROGRAM_KEY))
        {
            boolean oneFileByProgram = jobDataMap.getBoolean(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_ONE_FILE_BY_PROGRAM_KEY);
            reportParameters.put(AbstractMCCReport.PARAMETER_ONE_FILE_BY_PROGRAM, Boolean.toString(oneFileByProgram));
        }
        
        // If rules are not active, there is no version parameter
        if (jobDataMap.containsKey(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_RULES_VERSION_KEY))
        {
            reportParameters.put(MCCReport.PARAMETER_RULES_VERSION, jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_RULES_VERSION_KEY));
        }
        
        // Add filter on orgUnit parameter
        if (jobDataMap.containsKey(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_FILTER_ON_CURRENT_ORGUNIT))
        {
            boolean filterOnOrgUnit = jobDataMap.getBoolean(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_FILTER_ON_CURRENT_ORGUNIT);
            reportParameters.put(AbstractMCCReport.PARAMETER_FILTER_ON_CURRENT_ORGUNIT, Boolean.toString(filterOnOrgUnit));
        }
        else if (jobDataMap.containsKey(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_FILTER_BY_ORGUNIT))
        {
            String chosenOrgUnitId = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_FILTER_BY_ORGUNIT);
            if (StringUtils.isNotBlank(chosenOrgUnitId))
            {
                reportParameters.put(AbstractMCCReport.PARAMETER_FILTER_BY_ORGUNIT, chosenOrgUnitId);
            }
        }
        
        return reportParameters;
    }
    
    /**
     * Remove the schedulable parameter 'rulesVersion' if rules are disabled.
     * @param original The original parameters
     * @return the original parameters if there is no modification or parameters without rulesVersion parameter if necessary
     */
    public static Map<String, ElementDefinition> getParameters(Map<String, ElementDefinition> original)
    {
        if (RulesManager.isRulesEnabled())
        {
            // Remove oneFileByProgram parameter if rules are enabled
            if (original.containsKey(JOBDATAMAP_ONE_FILE_BY_PROGRAM_KEY))
            {
                Map<String, ElementDefinition> copy = new LinkedHashMap<>(original);
                copy.remove(MCCReportSchedulableHelper.JOBDATAMAP_ONE_FILE_BY_PROGRAM_KEY);
                return copy;
            }
        }
        else
        {
            // Remove version parameter if rules are not enabled
            if (original.containsKey(JOBDATAMAP_RULES_VERSION_KEY))
            {
                Map<String, ElementDefinition> copy = new LinkedHashMap<>(original);
                copy.remove(MCCReportSchedulableHelper.JOBDATAMAP_RULES_VERSION_KEY);
                return copy;
            }
        }
        
        return original;
    }
}
