 /*
*  Copyright 2025 Anyware Services
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/

// For each thematic:
//  - Move counter from TS-ruleNumber to TS[X]-ruleNumber

// For each container with additional thematics:
//  - Move counter from TA-ruleNumber to several counters TA[X]-ruleNumber

const Optional = Java.type("java.util.Optional");
const List = Java.type("java.util.List");
const Integer = Java.type("java.lang.Integer");

Repository.query(`//element(*, ametys:content)[@ametys-internal:contentType = 'odf-enumeration.Thematic']`).forEach(
    thematic =>
    {
        logger.info(`[${thematic.getValue("catalog")}] Update thematic [${thematic.getValue("code")}] ${thematic.getTitle()} (${thematic.getId()})`);
        Content.migrate(
            thematic,
            [updateThematic],
            false,
            false,
            false,
            true
        );
    }
);

Repository.query("//element(*, ametys:container)").forEach(
    container =>
    {
        let hasThematics = !_getEntries(container, "thematics").isEmpty();
        
        if (hasThematics)
        {
            logger.info(`[${container.getValue("catalog")}] Update container [${container.getValue("code")}] ${container.getTitle()} (${container.getId()})`);
            Content.migrate(
                container,
                [updateThematicsInContainer],
                false,
                false,
                false,
                true
            );
        }
    }
);

function updateThematic(content)
{
    _updateThematic(content, content);
    
    _removeOldCounter(content, "TS-ruleNumber");
}

function updateThematicsInContainer(content)
{
    let thematics = _getEntries(content, "thematics");
    for (let thematic of thematics)
    {
        _updateThematic(content, thematic);
    }
    
    _removeOldCounter(content, "TA-ruleNumber");
}

function _updateThematic(content, thematic)
{
    let thematicCode = thematic.getValue("code");
    let maxRuleNumber = 0;
    
    let rules = _getEntries(thematic, "rules");
    for (let rule of rules)
    {
        let ruleCode = rule.getValue("code");
        let currentRuleNumber = Integer.valueOf(ruleCode.replace(thematicCode + "-", ""));
        if (currentRuleNumber > maxRuleNumber)
        {
            maxRuleNumber = currentRuleNumber;
        }
    }

    let savedRuleNumber = content.getInternalDataHolder().getValue(thematicCode + "-ruleNumber", null);
    if (savedRuleNumber == null || savedRuleNumber < maxRuleNumber)
    {
        logger.info(` - Add "${thematicCode}-ruleNumber" with value ${maxRuleNumber}`);
        content.getInternalDataHolder().setValue(thematicCode + "-ruleNumber", maxRuleNumber);
    }
}

function _removeOldCounter(content, counterName)
{
    if (content.getInternalDataHolder().hasValueOrEmpty(counterName))
    {
        logger.info(` - Remove "${counterName}" metadata`);
        content.getInternalDataHolder().removeValue(counterName);
    }
}

function _getEntries(dataHolder, repeaterName)
{
    return Optional.ofNullable(dataHolder.getRepeater(repeaterName))
                   .map(r => r.getEntries())
                   .orElseGet(() => List.of())
}
