/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfpilotage.cost.CostComputationComponent;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.plugins.odfpilotage.cost.entity.Effectives;
import org.ametys.plugins.odfpilotage.cost.entity.EqTD;
import org.ametys.plugins.odfpilotage.cost.entity.Groups;
import org.ametys.plugins.odfpilotage.cost.entity.NormDetails;
import org.ametys.plugins.odfpilotage.cost.entity.ProgramItemData;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.model.ModelItem;

/**
 * Pilotage report for cost model
 */
public class CoutMaquettesReport extends AbstractReport
{
    /** CalculerEffectifComponent */
    protected CostComputationComponent _costComputationComponent;
    
    private int _order;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _costComputationComponent = (CostComputationComponent) manager.lookup(CostComputationComponent.ROLE);
    }
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        return "coutmaquettes";
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_XLS;
    }

    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_XLS, OUTPUT_FORMAT_CSV);
    }
    
    @Override
    protected void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
        List<Program> selectedPrograms = _odfHelper.getProgramsFromOrgUnit(orgUnit, catalog, lang);
        CostComputationData costData = _costComputationComponent.computeCostsOnPrograms(selectedPrograms, true);
        
        try
        {
            handler.startDocument();
        
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            XMLUtils.startElement(handler, "report", attrs);
            
            _generateReport(handler, costData, orgUnit, lang, catalog);
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Coût des maquettes' report for orgunit '{}'", orgUnit.getTitle(), e);
        }
    }
    
    private void _generateReport(ContentHandler handler, CostComputationData costData, OrgUnit orgUnit, String lang, String catalog) throws SAXException
    {
        Map<Program, Object> contentsTree = _getStructure(orgUnit, lang, catalog);
        if (contentsTree.size() > 0)
        {
            _order = 1;
            _saxTree(handler, contentsTree);
        }
        
        _writeColumns(handler, costData);
        _writeLines(handler, costData);
    }
    
    /**
     * Sax the information related to the courses of the tree
     * @param handler the handler
     * @param programTree the program tree to sax
     * @throws SAXException if an error occurs when SAXing
     */
    @SuppressWarnings("unchecked")
    private void _saxTree(ContentHandler handler, Map<Program, Object> programTree) throws SAXException
    {
        for (Entry<Program, Object> programEntry : programTree.entrySet())
        {
            if (programEntry.getValue() != null && programEntry.getValue() instanceof Map<?, ?>)
            {
                _saxCourseFromTree(handler, (Map<ProgramItem, Object>) programEntry.getValue(), programEntry.getKey());
            }
        }
    }

    private void _saxCourseFromTree(ContentHandler handler, Map<ProgramItem, Object> programTree, Program program) throws SAXException
    {
        _saxCourseFromTree(handler, programTree, program, null, null, null, null, null, 1, program.getName());
    }

    private void _saxCourseFromTree(ContentHandler handler, Map<ProgramItem, Object> tree, Program program, SubProgram subprogram, Container containerYear, CourseList list, Integer listPosition, Course parentCourse, int level, String hierarchy) throws SAXException
    {
        int courseListPosition = 0;
        for (Entry<ProgramItem, Object> entry : tree.entrySet())
        {
            ProgramItem child = entry.getKey();
            @SuppressWarnings("unchecked")
            Map<ProgramItem, Object> subTree = (Map<ProgramItem, Object>) entry.getValue();

            String childHierarchy = hierarchy + ModelItem.ITEM_PATH_SEPARATOR + child.getName();
            
            if (child instanceof Course childCourse)
            {
                _saxCourse(handler, program, subprogram, containerYear, list, listPosition, childCourse, parentCourse, level, subTree.isEmpty(), childHierarchy);
                _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, list, listPosition, childCourse, level + 1, childHierarchy);
            }
            else if (child instanceof Program childProgram)
            {
                _saxCourseFromTree(handler, subTree, childProgram, subprogram, containerYear, list, listPosition, parentCourse, level, childHierarchy);
            }
            else if (child instanceof Container childContainer)
            {
                String containerNature = _refTableHelper.getItemCode(childContainer.getNature());
                
                if ("annee".equals(containerNature))
                {
                    _saxCourseFromTree(handler, subTree, program, subprogram, childContainer, list, listPosition, parentCourse, level, childHierarchy);
                }
                else
                {
                    _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, list, listPosition, parentCourse, level, childHierarchy);
                }
            }
            else if (child instanceof SubProgram childSubProgram)
            {
                _saxCourseFromTree(handler, subTree, program, childSubProgram, containerYear, list, listPosition, parentCourse, level, childHierarchy);
            }
            else if (child instanceof CourseList childCourseList)
            {
                courseListPosition++;
                _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, childCourseList, courseListPosition, parentCourse, level, childHierarchy);
            }
        }
    }
    
    private void _saxCourse(ContentHandler handler, Program program, SubProgram subprogram, Container containerYear, CourseList list, Integer listPosition, Course course, Course parentCourse, int level, boolean lastLevel, String hierarchy) throws SAXException
    {
        if (course != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", course.getId());
            attrs.addCDATAAttribute("name", course.getName());
            attrs.addCDATAAttribute("path", StringUtils.join(hierarchy, "/"));
            XMLUtils.startElement(handler, "course", attrs);
            
            // Order
            XMLUtils.createElement(handler, "ordre", String.valueOf(_order));
            
            // Program
            _saxProgram(handler, program);
            
            // Subprogram
            _saxSubProgram(handler, subprogram);
            
            // Year
            _saxYear(handler, containerYear);
            
            // Parent course
            _saxParentCourse(handler, parentCourse);
            
            // Course list
            _saxCourseList(handler, list, listPosition);
            
            // A des fils
            boolean aDesFils = course.hasCourseLists();
            XMLUtils.createElement(handler, "aDesFils", aDesFils ? "X" : "");

            _saxPartage(handler,  course.getParentCourseLists().size());
            
            Container etape = _getEtapePorteuse(course, hierarchy);
            
            String porte = _getPorte(etape, containerYear);
           
            // Porté ("X" si l'ELP est porté par l'étape courante (Etape porteuse=COD_ETP), vide sinon)
            XMLUtils.createElement(handler, "porte", porte);
            
            // Niveau
            XMLUtils.createElement(handler, "niveau", "niv" + level);

            // Date de création
            XMLUtils.createElement(handler, "creationDate", DateUtils.zonedDateTimeToString(course.getCreationDate()));

            // Code Apogée
            XMLUtils.createElement(handler, "codeApogee", course.getValue("elpCode", false, StringUtils.EMPTY));
            
            // Nature de l'élément
            XMLUtils.createElement(handler, "nature", _refTableHelper.getItemCode(course.getCourseType()));

            // Libellé court
            XMLUtils.createElement(handler, "libelleCourt", course.getValue("shortLabel", false, StringUtils.EMPTY));
            
            // Libellé
            XMLUtils.createElement(handler, "libelle", course.getTitle());
            
            // Code Ametys (ELP)
            XMLUtils.createElement(handler, "elpCode", course.getDisplayCode());
            
            // Lieu
            _reportHelper.saxContentAttribute(handler, course, "campus", "campus");

            // Crédits ECTS
            XMLUtils.createElement(handler, "ects", String.valueOf(course.getEcts()));
            
            String teachingActivity = _refTableHelper.getItemCode(course.getTeachingActivity());
            String stage = teachingActivity.equals("SA") ? "X" : "";
            
            // Element stage
            XMLUtils.createElement(handler, "stage", stage);
            
            // Code semestre et type de période (pair, impair, an)
            Content period = Optional.of("period")
                    .map(course::<ContentValue>getValue)
                    .flatMap(ContentValue::getContentIfExists)
                    .orElse(null);
            if (period != null)
            {
                try
                {
                    String periodCode = Optional.of("code")
                        .map(period::<String>getValue)
                        .orElse(StringUtils.EMPTY);
                    
                    String periodTypeCode = Optional.ofNullable(period.<ContentValue>getValue("type"))
                        .flatMap(ContentValue::getContentIfExists)
                        .map(c -> c.<String>getValue("code"))
                        .orElse(StringUtils.EMPTY);

                    XMLUtils.createElement(handler, "periode", "s10".equals(periodCode) ? "s0" : periodCode);
                    XMLUtils.createElement(handler, "periodeType", periodTypeCode);
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().error("Impossible de retrouver la période : {}", period, e);
                }
            }
            
            // Composante
            _saxOrgUnit(handler, _getOrgUnit(course, hierarchy));
            
            // Code ANU
            long codeAnu = course.getValue("CodeAnu", false, 0L);
            XMLUtils.createElement(handler, "CodeAnu", codeAnu > 0 ? String.valueOf(codeAnu) : StringUtils.EMPTY);
            
            // Calcul des charges
            XMLUtils.createElement(handler, "calculCharges", aDesFils ? "" : "X");

            // Discipline
            String disciplineEnseignement = Optional.of("disciplineEnseignement")
                    .map(course::<ContentValue>getValue)
                    .flatMap(ContentValue::getContentIfExists)
                    .map(OdfReferenceTableEntry::new)
                    .map(entry -> {
                        String code = entry.getCode();
                        return (StringUtils.isNotEmpty(code) ? "[" + code + "] " : StringUtils.EMPTY) + entry.getLabel(course.getLanguage());
                    })
                    .orElse(StringUtils.EMPTY);
            
            XMLUtils.createElement(handler, "discipline", disciplineEnseignement);

            if (lastLevel)
            {
                // Etape porteuse
                _saxStepHolder(handler, etape);
                
                // Heures d'enseignement
                XMLUtils.startElement(handler, "courseParts");
                for (CoursePart coursePart : course.getCourseParts())
                {
                    AttributesImpl attr = new AttributesImpl();
                    attr.addCDATAAttribute("id", coursePart.getId());
                    XMLUtils.createElement(handler, "coursePart", attr);
                }
                XMLUtils.endElement(handler, "courseParts");
            }
            
            XMLUtils.endElement(handler, "course");
            
            _order++;
        }
    }
    
    private void _saxProgram(ContentHandler handler, Program program) throws SAXException
    {
        if (program != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", program.getId());
            attrs.addCDATAAttribute("name", program.getName());
            XMLUtils.startElement(handler, "program", attrs);

            XMLUtils.createElement(handler, "title", program.getTitle());
            XMLUtils.createElement(handler, "code", program.getDisplayCode());
            
            String orgUnit = Arrays.stream(program.<ContentValue[]>getValue("orgUnit"))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(OrgUnit.class::cast)
                .map(ou -> ou.getTitle() + " (" + ou.getDisplayCode() + ")")
                .collect(Collectors.joining(", "));
            XMLUtils.createElement(handler, "orgUnits", orgUnit);
            
            XMLUtils.endElement(handler, "program");
        }
    }
    
    private void _saxSubProgram(ContentHandler handler, SubProgram subprogram) throws SAXException
    {
        if (subprogram != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", subprogram.getId());
            attrs.addCDATAAttribute("name", subprogram.getName());
            XMLUtils.startElement(handler, "subProgram", attrs);
            
            // Parcours
            XMLUtils.createElement(handler, "title", subprogram.getTitle());
            XMLUtils.createElement(handler, "code", subprogram.getDisplayCode());
            
            XMLUtils.endElement(handler, "subProgram");
        }
    }
    
    private void _saxYear(ContentHandler handler, Container container) throws SAXException
    {
        if (container != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", container.getId());
            attrs.addCDATAAttribute("name", container.getName());
            XMLUtils.startElement(handler, "year", attrs);
            
            // Année
            XMLUtils.createElement(handler, "title", container.getTitle());
            // Code Ametys
            XMLUtils.createElement(handler, "code", container.getDisplayCode());
            // COD_ETP
            XMLUtils.createElement(handler, "COD_ETP", container.getValue("etpCode", false, StringUtils.EMPTY));
            // COD_VRS_ETP
            XMLUtils.createElement(handler, "COD_VRS_ETP", container.getValue("vrsEtpCode", false, StringUtils.EMPTY));
            // Number of students estimated
            XMLUtils.createElement(handler, "nbStudents", String.valueOf(container.getValue("numberOfStudentsEstimated", false, 0)));
            
            XMLUtils.endElement(handler, "year");
        }
    }
    
    private void _saxParentCourse(ContentHandler handler, Course parentCourse) throws SAXException
    {
        if (parentCourse != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", parentCourse.getId());
            attrs.addCDATAAttribute("name", parentCourse.getName());
            XMLUtils.startElement(handler, "parentCourse", attrs);
            
            // Code Ametys ELP père
            XMLUtils.createElement(handler, "codeELPPere", parentCourse.getDisplayCode());

            XMLUtils.endElement(handler, "parentCourse");
        }
    }
    
    private void _saxOrgUnit(ContentHandler handler, OrgUnit orgUnit) throws SAXException
    {
        if (orgUnit != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", orgUnit.getId());
            XMLUtils.startElement(handler, "orgUnit", attrs);
            
            // Code composante
            String codeComposante = Optional.ofNullable(orgUnit)
                .map(o -> o.<String>getValue("codCmp"))
                .orElse(StringUtils.EMPTY);
            XMLUtils.createElement(handler, "codeComposante", codeComposante);
            
            // Code CIP
            String codeCIP = Optional.ofNullable(orgUnit)
                .map(o -> o.<String>getValue("codCipApogee"))
                .orElse(StringUtils.EMPTY);
            XMLUtils.createElement(handler, "codeCIP", codeCIP);
            
            XMLUtils.endElement(handler, "orgUnit");
        }
    }
    
    private String _getPorte(Container etape, Container containerYear)
    {
        String porte = "";
        if (etape != null && containerYear != null)
        {
            String etpCode = containerYear.getValue("etpCode", false, StringUtils.EMPTY);
            if (StringUtils.isNotEmpty(etpCode))
            {
                porte = etpCode.equals(etape.getValue("etpCode", false, StringUtils.EMPTY)) ? "X" : "";
            }
        }
        return porte;
    }
    
    private OrgUnit _getOrgUnit(Course course, String hierarchy)
    {
        OrgUnit orgUnit = null;
        
        List<String> courseOrgUnits = course.getOrgUnits();
        if (!courseOrgUnits.isEmpty())
        {
            try
            {
                orgUnit = _resolver.resolveById(courseOrgUnits.get(0));
            }
            catch (UnknownAmetysObjectException e)
            {
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("La composante référencée par l'élément pédagogique {} ({}) n'a pas été trouvée.", _pilotageHelper.getDisplayablePath(hierarchy), course.getCode());
                }
            }
            
            if (courseOrgUnits.size() > 1 && getLogger().isWarnEnabled())
            {
                getLogger().warn("L'élément pédagogique {} ({}) référence plus d'une composante.", _pilotageHelper.getDisplayablePath(hierarchy), course.getCode());
            }
        }
        
        return orgUnit;
    }
    
    private Container _getEtapePorteuse(Course course, String hierarchy)
    {
        Pair<StepHolderStatus, Container> stepHolder = _pilotageHelper.getStepHolder(course);
        switch (stepHolder.getKey())
        {
            case NO_YEAR:
                if (getLogger().isWarnEnabled())
                {
                    getLogger().warn("Impossible de trouver une nature de conteneur 'annee' pour l'élément pédagogique {} ({}).", _pilotageHelper.getDisplayablePath(hierarchy), course.getCode());
                }
                break;
            case NONE:
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("L'élément pédagogique {} ({}) n'est rattaché à aucune année.", _pilotageHelper.getDisplayablePath(hierarchy), course.getCode());
                }
                break;
            case MULTIPLE:
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("Impossible de définir une année porteuse unique sur l'élément pédagogique {} ({}).", _pilotageHelper.getDisplayablePath(hierarchy), course.getCode());
                }
                break;
            case WRONG_YEAR:
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("L'année porteuse {} n'est pas dans l'arborescence de l'élément pédagogique {} ({})", stepHolder.getValue().getTitle(), _pilotageHelper.getDisplayablePath(hierarchy), course.getCode());
                }
                break;
            default:
                // Nothing to do
                break;
        }
        
        return stepHolder.getValue();
    }
    
    private void _saxCourseList(ContentHandler handler, CourseList list, Integer position) throws SAXException
    {
        if (list != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", list.getId());
            attrs.addCDATAAttribute("name", list.getName());
            attrs.addCDATAAttribute("displayName", "Lst" + position);
            
            // Type
            ChoiceType typeList = list.getType();
            if (typeList != null)
            {
                switch (typeList)
                {
                    case CHOICE:
                        attrs.addCDATAAttribute("type", "X");
                        // Min-Max (Ne remplir que pour le type "CHOICE" (ne mettre que le min))
                        attrs.addCDATAAttribute("minmax", _reportHelper.formatNumberToSax(list.getMinNumberOfCourses()));
                        break;
                    case MANDATORY:
                        attrs.addCDATAAttribute("type", "O");
                        break;
                    case OPTIONAL:
                        attrs.addCDATAAttribute("type", "F");
                        break;
                    default:
                        // Nothing to do
                        break;
                }
            }

            XMLUtils.createElement(handler, "courseList", attrs);
        }
    }
    
    private void _saxPartage(ContentHandler handler, long nbParents) throws SAXException
    {
        if (nbParents > 1)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("occurrences", _reportHelper.formatNumberToSax(nbParents));
            XMLUtils.createElement(handler, "partage", attrs);
        }
    }
    
    private void _saxStepHolder(ContentHandler handler, Container etape) throws SAXException
    {
        if (etape != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", etape.getId());
            XMLUtils.startElement(handler, "stepHolder", attrs);
            
            XMLUtils.createElement(handler, "code", etape.getDisplayCode());
            XMLUtils.createElement(handler, "etpCode", etape.getValue("etpCode", false, StringUtils.EMPTY));
            XMLUtils.createElement(handler, "vrsEtpCode", etape.getValue("vrsEtpCode", false, StringUtils.EMPTY));
            XMLUtils.createElement(handler, "title", etape.getTitle());
            
            XMLUtils.endElement(handler, "stepHolder");
        }
    }
    
    /**
     * Generate the data structure that will be used to create the report
     * @param rootOrgUnit the root org unit
     * @param lang the lang of programs
     * @param catalog the catalog of programs
     * @return The structure
     */
    private Map<Program, Object> _getStructure(OrgUnit rootOrgUnit, String lang, String catalog)
    {
        Map<Program, Object> programTree = new TreeMap<>(ReportHelper.CONTENT_TITLE_COMPARATOR);
        
        // On ne récupère que la composante racine ou bien les composantes, enfant direct du root org unit, et on ignore les départements.
        if (rootOrgUnit.getParentOrgUnit() == null || rootOrgUnit.getParentOrgUnit().getParentOrgUnit() == null)
        {
            // Chercher les formations concernées par la composante sélectionnée et ses enfants
            List<Program> programs = _odfHelper.getProgramsFromOrgUnit(rootOrgUnit, catalog, lang);
            for (Program program : programs)
            {
                Map<ProgramItem, Object> courses = _reportHelper.getCoursesFromContent(program);
                programTree.put(program, courses);
            }
        }
        
        return programTree;
    }
    
    /**
     * Write lines content of the report
     * @param handler the handler
     * @param costData informations about the capacity
     * @throws SAXException to handle XMLUtils exceptions
     */
    private void _writeLines(ContentHandler handler, CostComputationData costData) throws SAXException
    {
        XMLUtils.startElement(handler, "courseParts");
        
        // Write a course part by line
        for (String contentId : costData.keySet())
        {
            ProgramItemData coursePartData = costData.get(contentId);
            if (coursePartData.isCoursePart())
            {
                CoursePart coursePart = _resolver.resolveById(contentId);
                
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("id", contentId);
                attr.addCDATAAttribute("name", coursePart.getName());
                XMLUtils.startElement(handler, "coursePart", attr);
                Map<String, String> calculatedCoursePart = getValues(coursePart, coursePartData);
                for (String columnName : calculatedCoursePart.keySet())
                {
                    XMLUtils.createElement(handler, columnName, calculatedCoursePart.get(columnName));
                }
                
                // Prorated eqTD
                for (Entry<String, Double> proratedEqTD : coursePartData.getEqTD().getProratedEqTD().entrySet())
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("path", proratedEqTD.getKey());
                    XMLUtils.createElement(handler, "proratedEqTD", attrs, String.valueOf(proratedEqTD.getValue()));
                }
                
                // Ventilation des effectifs
                for (Entry<String, Double> effective : coursePartData.getEffectives().getLocalEffectiveByPath().entrySet())
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("path", effective.getKey());
                    XMLUtils.createElement(handler, "effectif", attrs, String.valueOf(effective.getValue()));
                }
                
                // EqTD portés
                for (Entry<String, Double> localEqTD : coursePartData.getEqTD().getLocalEqTD().entrySet())
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("path", localEqTD.getKey());
                    XMLUtils.createElement(handler, "localEqTD", attrs, String.valueOf(localEqTD.getValue()));
                }
                
                XMLUtils.endElement(handler, "coursePart");
            }
        }
        
        XMLUtils.endElement(handler, "courseParts");
    }
    
    private void _writeColumns(ContentHandler handler, CostComputationData costData) throws SAXException
    {
        // Columns
        XMLUtils.startElement(handler, "columns");
        for (Container yearToDisplay : _getYearsToDisplay(costData))
        {
            _writeColumn(handler, yearToDisplay);
        }
        XMLUtils.endElement(handler, "columns");
    }
    
    private void _writeColumn(ContentHandler handler, Container year) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", year.getId());
        attrs.addCDATAAttribute("name", year.getName());
        XMLUtils.createElement(handler, "column", attrs, _getDisplayCode(year));
    }
    
    private Set<Container> _getYearsToDisplay(CostComputationData costData)
    {
        return costData.entrySet()
            .stream()
            .filter(entry -> entry.getValue().isCoursePart())
            .map(Entry::getKey)
            .map(_resolver::<CoursePart>resolveById)
            .map(CoursePart::getCourses)
            .flatMap(List::stream)
            .map(_pilotageHelper::getSteps)
            .flatMap(Set::stream)
            .collect(Collectors.toCollection(() -> new LinkedHashSet<>()));
    }
    
    private String _getDisplayCode(Container container)
    {
        return _getValueOrNull(container, "anneePorteuse")
            .or(() ->
                _getValueOrNull(container, "etpCode")
                    .map(code ->
                        code
                        + _getValueOrNull(container, "vrsEtpCode")
                            .map(c -> "-" + c)
                            .orElse(StringUtils.EMPTY)
                    )
            )
            .or(() -> _getValueOrNull(container, "code"))
            .orElse(StringUtils.EMPTY);
    }
    
    private Optional<String> _getValueOrNull(Content content, String attributeName)
    {
        return Optional.of(content)
                .map(c -> c.<String>getValue(attributeName))
                .filter(StringUtils::isNotEmpty);
    }
    
    /**
     * Create a map of values to sax
     * @param coursePart the coursePart
     * @param programItemData the program item cost data
     * @return the map of values to sax
     */
    private Map<String, String> getValues(CoursePart coursePart, ProgramItemData programItemData)
    {
        Map<String, String> values = new HashMap<>();
        
        // *** Course part ***
        values.put("titre", coursePart.getTitle());
        values.put("code", coursePart.getCode());
        values.put("nature", _refTableHelper.getItemCode(coursePart.getNature()));
        values.put("volumeHoraire", ReportHelper.FORMAT_2_DIGITS.format(coursePart.getNumberOfHours()));
        
        Course courseHolder = coursePart.getCourseHolder();
        if (courseHolder == null)
        {
            getLogger().warn("Les heures d'enseignement '{}' ({}) n'ont plus d'ELP porteur.", coursePart.getTitle(), coursePart.getCode());
        }
        else
        {
            values.put("elpPorteur", courseHolder.getDisplayCode());
        }
        
        // *** Norme et + ***
        Effectives effectives = programItemData.getEffectives();
        if (effectives != null)
        {
            values.put("effectifCalcule", effectives.getComputedEffective().map(ReportHelper.FORMAT_2_DIGITS::format).orElse(StringUtils.EMPTY));
            values.put("effectifPrevisionnel", effectives.getEstimatedEffective().map(ReportHelper.FORMAT_2_DIGITS::format).orElse(StringUtils.EMPTY));
        }
        NormDetails normDetails = programItemData.getNormDetails();
        if (normDetails != null)
        {
            values.put("effectifMax", Optional.of(normDetails).map(NormDetails::getEffectiveMax).map(ReportHelper.FORMAT_2_DIGITS::format).orElse(StringUtils.EMPTY));
            values.put("effectifMinSup", Optional.of(normDetails).map(NormDetails::getEffectiveMinSup).map(ReportHelper.FORMAT_2_DIGITS::format).orElse(StringUtils.EMPTY));
            values.put("norme", normDetails.getNormLabel());
        }
        Groups groups = programItemData.getGroups();
        if (groups != null)
        {
            values.put("groupesCalcules", ReportHelper.FORMAT_2_DIGITS.format(groups.getComputedGroups()));
            Long groupsToOpen = groups.getGroupsToOpen();
            if (groupsToOpen != null)
            {
                values.put("groupesAOuvrir", ReportHelper.FORMAT_2_DIGITS.format(groupsToOpen));
            }
        }
        
        EqTD eqTD = programItemData.getEqTD();
        if (eqTD != null)
        {
            values.put("eqTDTotal", ReportHelper.FORMAT_2_DIGITS.format(eqTD.getGlobalEqTD()));
        }
        
        values.values().removeIf(Objects::isNull);
        
        return values;
    }
}
