/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.export;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.odfpilotage.rule.AbstractThematicsGenerator;
import org.ametys.plugins.odfpilotage.rule.enumerators.NbSessionsEnumerator;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * Generator to export rules.
 */
public class ThematicsExportGenerator extends AbstractThematicsGenerator implements Initializable
{
    private String _odfLang;
    
    public void initialize() throws Exception
    {
        _odfLang = Config.getInstance().getValue("odf.programs.lang");
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "export");
        if (_thematicsHelper.hasHandleThematicRight())
        {
            _saxFilters();
            _saxContents();
            _saxGroups();
        }
        XMLUtils.endElement(contentHandler, "export");
        contentHandler.endDocument();
    }
    
    private void _saxContents() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "contents");
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String catalog = request.getParameter("catalog");
        String degreeId = request.getParameter("degree");
        
        for (Content thematic : _thematicsHelper.getCompatibleThematics(catalog, degreeId))
        {
            _saxContent(thematic);
        }
        
        XMLUtils.endElement(contentHandler, "contents");
    }
    
    /**
     * Sax selected filters.
     * @throws SAXException if an error occurs
     */
    protected void _saxFilters() throws SAXException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String catalog = request.getParameter("catalog");
        String degreeId = request.getParameter("degree");
        
        XMLUtils.startElement(contentHandler, "filters");
        
        _saxCatalog(catalog);
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("value", degreeId);
        XMLUtils.createElement(contentHandler, "degree", attrs, _odfRefTableHelper.getItemLabel(degreeId, _odfLang));
        
        XMLUtils.endElement(contentHandler, "filters");
    }
    
    /**
     * Sax groups of the document.
     * @throws SAXException if an error occurs
     */
    protected void _saxGroups() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "groups");
        
        // Number of sessions
        _saxEnumerator(new NbSessionsEnumerator(), "nbSessions");
        
        // Regimes
        _odfRefTableHelper.saxItems(contentHandler, "odf-enumeration.MccRegime");
        
        XMLUtils.endElement(contentHandler, "groups");
    }
    
    /**
     * Sax enumerator entries.
     * @param enumerator the enumerator
     * @param tagName the root tag name
     * @throws SAXException if an error occurs
     */
    protected void _saxEnumerator(Enumerator<String> enumerator, String tagName) throws SAXException
    {
        try
        {
            Map<String, I18nizableText> entries = enumerator.getEntries();
            XMLUtils.startElement(contentHandler, tagName);
            for (String key : entries.keySet())
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("key", key);
                XMLUtils.startElement(contentHandler, "entry", attrs);
                entries.get(key).toSAX(contentHandler);
                XMLUtils.endElement(contentHandler, "entry");
            }
            XMLUtils.endElement(contentHandler, tagName);
        }
        catch (Exception e)
        {
            if (e instanceof SAXException se)
            {
                throw se;
            }
            
            throw new SAXException("Error while resolving enumerator entries " + enumerator.getClass().getName(), e);
        }
    }
}
