/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* Imports, components and constants */

const HashMap = Java.type("java.util.HashMap");
const List = Java.type("java.util.List");
const Optional = Java.type("java.util.Optional");

const ValidateContentCondition = Ametys.serviceManager.lookup("org.ametys.cms.workflow.ValidateContentCondition");

const CONTENT_KEY = org.ametys.cms.workflow.AbstractContentWorkflowComponent.CONTENT_KEY;
const FAIL_CONDITIONS_KEY = org.ametys.cms.workflow.AbstractContentWorkflowComponent.FAIL_CONDITIONS_KEY;

/* Teaching natures */

logger.info("Migration of teaching natures");

let contents = Repository.query("//element(*, ametys:content)[@ametys-internal:contentType = 'odf-enumeration.EnseignementNature']");
let modified = 0;

contents.forEach(
    content =>
    {
        Content.migrate( 
            content, 
            [_migrateNature], 
            true, /* old version still comptatible */ 
            null, /* no tag */ 
            false, /* verbose */
            true /* synchronize live */
        );
    }
);

logger.info(`${modified} of ${contents.getSize()} teaching natures has been updated`);

/* Norms */

logger.info("Migration of norms");

contents = Repository.query("//element(*, ametys:content)[@ametys-internal:contentType = 'odf-enumeration.Norme']");
modified = 0;

contents.forEach(
    content =>
    {
        Content.migrate( 
            content, 
            [_migrateNorm], 
            true, /* old version still comptatible */ 
            null, /* no tag */ 
            false, /* verbose */
            true /* synchronize live */
        );
    }
);

logger.info(`${modified} of ${contents.getSize()} norms has been updated`);

/* Functions */

function _migrateNature(content)
{
    let hasChanges = _applyFunctionIfInvalidValue(content, "effectifMax", _removeValue);
    hasChanges = _applyFunctionIfInvalidValue(content, "effectifMinSup", _removeValue) || hasChanges;
    
    if (hasChanges)
    {
        logger.info(`Update teaching nature ${content.getValue("code")} (${content.getId()})`);
        modified++;
    }
}

function _migrateNorm(content)
{
    let hasChanges = false;
    
    let entries = Optional.of(content)
        .map(c => c.getRepeater("groupEffectives"))
        .map(r => r.getEntries())
        .orElse(List.of());
    
    entries.forEach(
        entry =>
        {
            hasChanges =
                _applyFunctionIfInvalidValue(entry, "effectifMax", _removeRepeaterEntry) ||
                _applyFunctionIfInvalidValue(entry, "effectifMinSup", _replaceValueByOne) ||
                hasChanges;
        }
    );
    
    if (hasChanges)
    {
        logger.info(`Update norm ${content.getValue("code")} (${content.getId()})`);
        modified++;
    }
}

function _isInvalidValue(dataHolder, attributeName)
{
    return dataHolder.getValue(attributeName, false, 1) < 1;
}

function _applyFunctionIfInvalidValue(dataHolder, attributeName, functionToApply)
{
    if (_isInvalidValue(dataHolder, attributeName))
    {
        functionToApply(dataHolder, attributeName)
        return true;
    }
    return false;
}

function _removeValue(dataHolder, attributeName)
{
    dataHolder.removeValue(attributeName);
}

function _replaceValueByOne(dataHolder, attributeName)
{
    dataHolder.setValue(attributeName, 1);
}

function _removeRepeaterEntry(repeaterEntry, attributeName)
{
    repeaterEntry.getHoldingRepeater().removeEntry(repeaterEntry.getPosition());
}
