/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report;

import java.util.Map;
import java.util.Set;

import org.xml.sax.ContentHandler;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.Supporter;

/**
 * Interface for pilotage reports.
 */
public interface PilotageReport extends Supporter<AbstractReportSchedulable>
{
    /** The key for the output format */
    public static final String PARAMETER_OUTPUT_FORMAT = "outputFormat";
    /** The key for the program */
    public static final String PARAMETER_PROGRAM = "program";
    /** The key for the orgunit */
    public static final String PARAMETER_ORGUNIT = "orgunit";
    /** The key for the catalog */
    public static final String PARAMETER_CATALOG = "catalog";
    /** The key for the lang */
    public static final String PARAMETER_LANG = "lang";

    /** The XLS output format */
    public static final String OUTPUT_FORMAT_XLS = "xls";
    /** The DOC output format */
    public static final String OUTPUT_FORMAT_DOC = "doc";
    /** The CSV output format */
    public static final String OUTPUT_FORMAT_CSV = "csv";
    /** The PDF output format */
    public static final String OUTPUT_FORMAT_PDF = "pdf";
    
    /**
     * The pilotage report target.
     */
    public enum PilotageReportTarget
    {
        /** To execute a report on a program */
        PROGRAM (new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_TOOL_TARGET_PROGRAM")),
        /** To execute a report on an orgunit */
        ORGUNIT (new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_TOOL_TARGET_ORGUNIT"));

        private I18nizableText _label;

        PilotageReportTarget(I18nizableText newLabel)
        {
            this._label = newLabel;
        }

        /**
         * Get the label
         * @return The label
         */
        public I18nizableText getLabel()
        {
            return _label;
        }
    }

    /**
     * Retrieves the id of the report.
     * @return the id.
     */
    public String getId();
    
    /**
     * Retrieves the label of the report.
     * @return the label.
     */
    public I18nizableText getLabel();

    /**
     * Get the plugin name of the current report.
     * @return the plugin name.
     */
    public String getPluginName();
    
    /**
     * Get the short name of the report
     * @return The report short name
     */
    public String getType();
    
    /**
     * Get the name of the report with complements if report parameters is filled.
     * @param reportParameters The report parameters
     * @param shortName <code>true</code> to get the short type name
     * @return The report name
     */
    public String getType(Map<String, String> reportParameters, boolean shortName);
    
    /**
     * Get the list of supported output formats
     * @return A {@link Set} of supported output formats
     */
    public Set<String> getSupportedOutputFormats();
    
    /**
     * Get the default output format
     * @return the default output format
     */
    public String getDefaultOutputFormat();
    
    /**
     * Check if the output format is supported.
     * @param outputFormat The output format
     * @return <code>true</code> if the output format is supported
     */
    public boolean isSupportedFormat(String outputFormat);
    
    /**
     * Get the report content: list of files and zip name.
     * @param target The report target
     * @param reportParameters The report parameters
     * @return the report content
     */
    public PilotageReportContent getReportContent(PilotageReportTarget target, Map<String, String> reportParameters);

    /**
     * Get the report filename for a given orgunit
     * @param catalog The catalog
     * @param lang The lang
     * @param orgUnit The orgunit
     * @param reportParameters The report parameters
     * @param outputFormat The output format
     * @return the file name
     */
    public String getReportFileName(String catalog, String lang, OrgUnit orgUnit, Map<String, String> reportParameters, String outputFormat);
    
    /**
     * Get the report filename for a given program
     * @param programItem The programItem
     * @param reportParameters The report parameters
     * @param outputFormat The output format
     * @return the file name
     */
    public String getReportFileName(ProgramItem programItem, Map<String, String> reportParameters, String outputFormat);
    
    /**
     * Sax a program item on the given content handler from the program item identifier and report parameters
     * @param handler The handler
     * @param programItemId The program item identifier
     * @param reportParameters The report parameters
     */
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters);
    
    /**
     * Sax an org unit on the given content handler from the orgunit identifier and report parameters
     * @param handler The handler
     * @param orgUnitId The orgunit identifier
     * @param reportParameters The report parameters
     */
    public void saxOrgUnit(ContentHandler handler, String orgUnitId, Map<String, String> reportParameters);
    
    /**
     * Record representing the content of a pilotage report: files and ZIP name
     * @param zipName The final ZIP name
     * @param files The list of files to generate (name and pipeline to call)
     */
    public record PilotageReportContent(String zipName, Map<String, String> files) { /* empty */ }
}
