/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.io.IOException;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.cache.Cache;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder.UserInput;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowException.ExceptionType;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.workflow.ValidateProgramItemTreeCondition;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ModelItem;

import jakarta.mail.MessagingException;

/**
 * Helper for MCC workflow
 */
public class MCCWorkflowHelper extends AbstractWorkflowHelper implements Initializable
{
    /** The component role. */
    public static final String ROLE = MCCWorkflowHelper.class.getName();
    
    /** The super right for rules validation state on year */
    public static final String RULES_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_Rules_Validated_Super_Rights";
    
    /** The super right for MCC fields validation state on year */
    public static final String MCC_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_Validated_Super_Rights";
    
    /** The super right for MCC orgunit control state */
    public static final String MCC_ORGUNIT_CONTROLLED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_Orgunit_Controlled_Super_Rights";
    
    /** The super right for MCC orgunit validation state */
    public static final String MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_Orgunit_Validated_Super_Rights";
    
    /** The super right for MCC CFVU validation state */
    public static final String MCC_CFVU_CONTROLLED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_CFVU_Controlled_Super_Rights";
    
    /** The super right for MCC CFVU validation state */
    public static final String MCC_CFVU_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_CFVU_Validated_Super_Rights";
    
    /** The attribute name for the MCC validated pdf repeater */
    public static final String MCC_VALIDATED_PDF_REPEATER = "mcc-validated-pdf";
    
    /** The identifier for the MCC workflow action */
    public static final Integer MCC_WORKFLOW_ACTION_ID = 222222;
    
    /** The attribute name for the pilotage composite */
    private static final String __MCC_WORKFLOW_COMPOSITE = "mcc_workflow";
    
    /** The suffixe for status of MCC workflow step */
    private static final String __STATUS_SUFFIX = "_status";
    
    /** The prefix for rules mention validation attributes */
    private static final String __MCC_VALIDATION_PREFIX = "mcc_validation";
    
    /** The prefix for rules mention validation attributes */
    private static final String __RULES_VALIDATION_PREFIX = "rules_validation";
    
    /** The prefix for MCC orgunit pre-validation attributes */
    private static final String __MCC_ORGUNIT_CONTROL_PREFIX = "mcc_orgunit_control";
    
    /** The prefix for MCC orgunit validation attributes */
    private static final String __MCC_ORGUNIT_VALIDATION_PREFIX = "mcc_orgunit_validation";
    
    /** The prefix for MCC CFVU pre-validation attributes */
    private static final String __MCC_CFVU_CONTROL_PREFIX = "cfvu_mcc_control";
    
    /** The prefix for MCC CFVU validation attributes */
    private static final String __MCC_CFVU_VALIDATION_PREFIX = "cfvu_mcc_validation";
    
    private static final String __PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID = MCCWorkflowHelper.class.getName() + "$parentContainers";
    
    private static final String __PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID = MCCWorkflowHelper.class.getName() + "$parentContainersForRules";
    
    private static final String __PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID = MCCWorkflowHelper.class.getName() + "$parentContainersForMCC";

    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    /** The user manager */
    protected UserManager _userManager;
    /** The content helper */
    protected ContentHelper _contentHelper;
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    /** The user languages manager */
    protected UserLanguagesManager _userLanguagesManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _userLanguagesManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_CFVU_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_CFVU_STATUS_DESCRIPTION"),
                true);
        
        _cacheManager.createRequestCache(__PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_RULES_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_RULES_STATUS_DESCRIPTION"),
                true);
        
        _cacheManager.createRequestCache(__PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_STATUS_DESCRIPTION"),
                true);
    }
    
    private Cache<String, String> _getContainerParentsCache()
    {
        return _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID);
    }
    
    private Cache<String, String> _getContainerParentsCacheForRules()
    {
        return _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID);
    }
    
    private Cache<String, String> _getContainerParentsCacheForMCC()
    {
        return _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID);
    }
    
    /**
     * Invalidate all caches
     */
    public void invalidateAllCaches()
    {
        _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID).invalidateAll();
        _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID).invalidateAll();
        _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID).invalidateAll();
    }
    
    /**
     * Enumeration for steps of MCC workflow
     *
     */
    public enum MCCWorkflowStep
    {
        /** Rules are validated to mention level */
        DRAFT
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_DRAFT_STEP_LABEL");
            }
        },
        /** Rules are validated to mention level */
        RULES_VALIDATED
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATE_STEP_LABEL");
            }
        },
        /** MCC are validated to mention level */
        MCC_VALIDATED
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATE_STEP_LABEL");
            }
        },
        /** MCC have been controlled and ready for orgunit validation */
        MCC_ORGUNIT_CONTROLLED
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_ORGUNIT_CONTROL_STEP_LABEL");
            }
        },
        /** MCC are validated by orgunit */
        MCC_ORGUNIT_VALIDATED
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_ORGUNIT_VALIDATION_STEP_LABEL");
            }
        },
        /** MCC have been controlled and ready for CFVU validation */
        MCC_CFVU_CONTROLLED
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_CONTROL_STEP_LABEL");
            }
        },
        /** MCC are validated by CFVU */
        MCC_CFVU_VALIDATED
        {
            @Override
            public I18nizableText label()
            {
                return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATE_STEP_LABEL");
            }
        };
        
        /**
         * Get the MCC step label
         * @return the MCC step label
         */
        public abstract I18nizableText label();
    }
    
    /**
     * Enumeration for MCC workflow action
     */
    public enum MCCWorkflowAction
    {
        /** Validate rules to mention level */
        VALIDATE_RULES,
        /** Refuse control of rules to orgunit level */
        REFUSE_RULES_ORGUNIT_CONTROL,
        /** Validate MCC to mention level */
        VALIDATE_MCC,
        /** Refuse control of MCC to orgunit level */
        REFUSE_MCC_ORGUNIT_CONTROL,
        /** Control MCC to orgunit level */
        CONTROL_ORGUNIT,
        /** Refuse validation of MCC to orgunit level */
        REFUSE_VALIDATION_ORGUNIT,
        /** Validate MCC to orgunit level */
        VALIDATE_ORGUNIT,
        /** Refuse CFVU control of MCC */
        REFUSE_CONTROL_CFVU,
        /** Control MCC to CFVU */
        CONTROL_CFVU,
        /** Revert control of MCC to CFVU level */
        REFUSE_VALIDATION_CFVU,
        /** Validate MCC to CFVU level */
        VALIDATE_CFVU,
        /** Invalidate MCC to CFVU level */
        INVALIDATE_CFVU,
        /** Reset MCC statut */
        RESET
    }
    
    /**
     * Determines if the MCC workflow action is available for the container
     * @param container The container
     * @param workflowAction the MCC workflow action
     * @return true if action is available
     */
    public boolean isAvailableAction(Container container, MCCWorkflowAction workflowAction)
    {
        return isAvailableAction(container, workflowAction, true);
    }
    
    /**
     * Determines if the MCC workflow action is available for the container
     * @param container The container
     * @param workflowAction the MCC workflow action
     * @param checkStructure set to false to skip checking of structure validity. Be careful, using "false" the result may not reflect the truly availability of the action.
     * @return true if action is available
     */
    public boolean isAvailableAction(Container container, MCCWorkflowAction workflowAction, boolean checkStructure)
    {
        boolean available = false;
        boolean needDataCheck = !_isRevertAction(workflowAction);
        
        Map<String, Object> inputs = new HashMap<>();
        if (!checkStructure || workflowAction == MCCWorkflowAction.VALIDATE_RULES)
        {
            // For rule validation, we don't want to check the container structure. So we put CHECK_TREE_KEY to false
            inputs.put(ValidateProgramItemTreeCondition.CHECK_TREE_KEY, false);
        }
        
        List<MCCWorkflowStep> currentSteps = getCurrentSteps(container);
        
        switch (workflowAction)
        {
            case INVALIDATE_CFVU:
                available = currentSteps.contains(MCCWorkflowStep.MCC_CFVU_VALIDATED);
                break;
            case VALIDATE_CFVU:
            case REFUSE_VALIDATION_CFVU:
                available = currentSteps.contains(MCCWorkflowStep.MCC_CFVU_CONTROLLED);
                break;
            case CONTROL_CFVU:
            case REFUSE_CONTROL_CFVU:
                available = currentSteps.contains(MCCWorkflowStep.MCC_ORGUNIT_VALIDATED);
                break;
            case VALIDATE_ORGUNIT:
            case REFUSE_VALIDATION_ORGUNIT:
                available = currentSteps.contains(MCCWorkflowStep.MCC_ORGUNIT_CONTROLLED);
                break;
            case CONTROL_ORGUNIT:
                available = currentSteps.contains(MCCWorkflowStep.MCC_VALIDATED) && (!RulesManager.isRulesEnabled() || currentSteps.contains(MCCWorkflowStep.RULES_VALIDATED));
                break;
            case VALIDATE_MCC:
                available = currentSteps.contains(MCCWorkflowStep.DRAFT) && !currentSteps.contains(MCCWorkflowStep.MCC_VALIDATED);
                break;
            case REFUSE_MCC_ORGUNIT_CONTROL:
                available = currentSteps.contains(MCCWorkflowStep.MCC_VALIDATED);
                break;
            case VALIDATE_RULES:
                available = currentSteps.contains(MCCWorkflowStep.DRAFT) && !currentSteps.contains(MCCWorkflowStep.RULES_VALIDATED);
                break;
            case REFUSE_RULES_ORGUNIT_CONTROL:
                available = currentSteps.contains(MCCWorkflowStep.RULES_VALIDATED);
                break;
            default:
                throw new IllegalArgumentException("Unexpected value for MCC workflow action : " + workflowAction);
        }
        
        if (needDataCheck && available)
        {
            // Check if container data or structure are valid (except if checkStructure is false)
            return _contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID, inputs);
        }
        
        return available;
    }
    
    private boolean _isRevertAction(MCCWorkflowAction workflowAction)
    {
        switch (workflowAction)
        {
            case REFUSE_RULES_ORGUNIT_CONTROL:
            case REFUSE_MCC_ORGUNIT_CONTROL:
            case REFUSE_VALIDATION_ORGUNIT:
            case REFUSE_CONTROL_CFVU:
            case REFUSE_VALIDATION_CFVU:
            case INVALIDATE_CFVU:
                return true;
            default:
                return false;
        }
    }
    
    /**
     * Determines if a MCC step is currently valid
     * @param container the container
     * @param mccWorkflowStep the MCC workflow step
     * @return true if step is active
     */
    public boolean isStepValid(Container container, MCCWorkflowStep mccWorkflowStep)
    {
        switch (mccWorkflowStep)
        {
            case RULES_VALIDATED:
                return isRulesValidated(container);
            case MCC_VALIDATED:
                return isMCCValidated(container);
            case MCC_ORGUNIT_CONTROLLED:
                return isMCCOrgUnitControlled(container);
            case MCC_ORGUNIT_VALIDATED:
                return isMCCOrgUnitValidated(container);
            case MCC_CFVU_CONTROLLED:
                return isMCCCFVUControlled(container);
            case MCC_CFVU_VALIDATED:
                return isMCCCFVUValidated(container);
            case DRAFT:
                return !isMCCOrgUnitControlled(container);
            default:
                throw new IllegalArgumentException("Unexpected value for MCC workflow step: " + mccWorkflowStep);
        }
    }
    
    /**
     * Get the MCC current steps
     * @param container the container
     * @return the current steps
     */
    public List<MCCWorkflowStep> getCurrentSteps(Container container)
    {
        if (isMCCCFVUValidated(container))
        {
            return List.of(MCCWorkflowStep.MCC_CFVU_VALIDATED);
        }
        else if (isMCCCFVUControlled(container))
        {
            return List.of(MCCWorkflowStep.MCC_CFVU_CONTROLLED);
        }
        else if (isMCCOrgUnitValidated(container))
        {
            return List.of(MCCWorkflowStep.MCC_ORGUNIT_VALIDATED);
        }
        else if (isMCCOrgUnitControlled(container))
        {
            return List.of(MCCWorkflowStep.MCC_ORGUNIT_CONTROLLED);
        }
        
        boolean mccValidated = isMCCValidated(container);
        boolean rulesValidated = isRulesValidated(container);
        boolean draft = !mccValidated || RulesManager.isRulesEnabled() && !rulesValidated;
        
        List<MCCWorkflowStep> currentSteps = new ArrayList<>();
        if (draft)
        {
            currentSteps.add(MCCWorkflowStep.DRAFT);
        }
        if (mccValidated)
        {
            currentSteps.add(MCCWorkflowStep.MCC_VALIDATED);
        }
        if (RulesManager.isRulesEnabled() && rulesValidated)
        {
            currentSteps.add(MCCWorkflowStep.RULES_VALIDATED);
        }
        
        return currentSteps;
    }
    
    /**
     * Get the first parent container with the higher MCC status
     * @param programItem the program item
     * @return the first parent program with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatus(ProgramItem programItem)
    {
        Cache<String, String> cache = _getContainerParentsCache();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container parentContainer = _computeParentContainerWithHigherMCCStatus(programItem);
            cache.put(programItem.getId(), parentContainer != null ? parentContainer.getId() : null);
            return parentContainer;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatus(ProgramItem programItem)
    {
        Set<Container> parentContainers = _getParentYearContainers(programItem);
        return parentContainers.stream()
            .sorted(new MCCStatusComparator().reversed())
            .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent program with the higher MCC status
     * @param coursePart the course part
     * @return the first parent program with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatus(CoursePart coursePart)
    {
        Cache<String, String> cache = _getContainerParentsCache();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container parentContainer = _computeParentContainerWithHigherMCCStatus(coursePart);
            cache.put(coursePart.getId(), parentContainer != null ? parentContainer.getId() : null);
            return parentContainer;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatus(CoursePart coursePart)
    {
        Set<Container> parentPrograms = _getParentYearContainers(coursePart);
        return parentPrograms.stream()
            .sorted(new MCCStatusComparator().reversed())
            .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param programItem the program item
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForRules(ProgramItem programItem)
    {
        Cache<String, String> cache = _getContainerParentsCacheForRules();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForRules(programItem);
            cache.put(programItem.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForRules(ProgramItem programItem)
    {
        Set<Container> yearContainers = _getParentYearContainers(programItem);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCRulesStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param coursePart the course part
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForRules(CoursePart coursePart)
    {
        Cache<String, String> cache = _getContainerParentsCacheForRules();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForRules(coursePart);
            cache.put(coursePart.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForRules(CoursePart coursePart)
    {
        Set<Container> yearContainers = _getParentYearContainers(coursePart);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCRulesStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param programItem the program item
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForMCCFields(ProgramItem programItem)
    {
        Cache<String, String> cache = _getContainerParentsCacheForMCC();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForMCCFields(programItem);
            cache.put(programItem.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForMCCFields(ProgramItem programItem)
    {
        Set<Container> yearContainers = _getParentYearContainers(programItem);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCFieldsStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param coursePart the course part
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForMCCFields(CoursePart coursePart)
    {
        Cache<String, String> cache = _getContainerParentsCacheForMCC();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForMCCFields(coursePart);
            cache.put(coursePart.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForMCCFields(CoursePart coursePart)
    {
        Set<Container> yearContainers = _getParentYearContainers(coursePart);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCFieldsStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Validate MCC to orgunit level
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC to orgunit level
     */
    public boolean validateMCCForOrgunit(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.VALIDATE_ORGUNIT))
        {
            throw new MCCWorkflowException("MCC can not be validated to orgunit level as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        return _setMCCStatus(container, __MCC_ORGUNIT_VALIDATION_PREFIX, date, user, comment);
    }
    
    /**
     * Refuse control of MCC to CFVU level
     * @param container the container
     * @param user the user
     * @param notify to notify author of precede action by mail
     * @param comment optional comment to add to mail
     * @param reset true to reset MCC workflow
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate MCC for orgunit level
     */
    public boolean refuseCFVUControl(Container container, UserIdentity user, boolean notify, String comment, boolean reset) throws MCCWorkflowException
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.REFUSE_CONTROL_CFVU))
        {
            throw new MCCWorkflowException("MCC can not be invalidate to orgunit level as current MCC statut does not allow to perform action for container " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        if (notify)
        {
            List<UserIdentity> recipients = new ArrayList<>();
            Optional.of(getMCCOrgunitValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
            
            if (reset)
            {
                Optional.of(getMCCOrgunitControlStep(container)).ifPresent(s -> recipients.add(s.author()));
                Optional.of(getMCCMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                
                if (RulesManager.isRulesEnabled())
                {
                    recipients.add(getRulesMentionValidationStep(container).author());
                }
            }
            _notifyContributor(container, user, MCCWorkflowAction.REFUSE_CONTROL_CFVU, recipients, comment);
        }
        
        return reset ? removeMCCWorkflow(container) : _removeMCCStatus(container, __MCC_ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if MCC are validated to orgunit level
     * @param container the container
     * @return <code>true</code> if MCC are validated to orgunit level
     */
    public boolean isMCCOrgUnitValidated(Container container)
    {
        return _getMCCStatus(container, __MCC_ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Validate MCC to orgunit level
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC to orgunit level
     */
    public boolean controlMCCForOrgunit(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.CONTROL_ORGUNIT))
        {
            throw new MCCWorkflowException("MCC can not be controlled to orgunit level as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        return _setMCCStatus(container, __MCC_ORGUNIT_CONTROL_PREFIX, date, user, comment);
    }
    
    /**
     * Refuse validation of MCC to orgunit level
     * @param container the container
     * @param user the user
     * @param notify true to notify author of precede action by mail
     * @param comment optional comment to add to mail
     * @param reset true to reset MCC workflow
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate MCC for orgunit level
     */
    public boolean refuseOrgunitValidation(Container container, UserIdentity user, boolean notify, String comment, boolean reset) throws MCCWorkflowException
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.REFUSE_VALIDATION_ORGUNIT))
        {
            throw new MCCWorkflowException("Can not refuse orgunit validation as MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        if (notify)
        {
            List<UserIdentity> recipients = new ArrayList<>();
            Optional.of(getMCCOrgunitControlStep(container)).ifPresent(s -> recipients.add(s.author()));
            if (reset)
            {
                Optional.of(getMCCMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                if (RulesManager.isRulesEnabled())
                {
                    Optional.of(getRulesMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                }
            }
            _notifyContributor(container, user, MCCWorkflowAction.REFUSE_VALIDATION_ORGUNIT, recipients, comment);
        }
        
        return reset ? removeMCCWorkflow(container) : _removeMCCStatus(container, __MCC_ORGUNIT_CONTROL_PREFIX);
    }
    
    /**
     * Determines if MCC have been controlled by orgunit
     * @param container the container
     * @return <code>true</code> if MCC have been controlled by orgunit
     */
    public boolean isMCCOrgUnitControlled(Container container)
    {
        return _getMCCStatus(container, __MCC_ORGUNIT_CONTROL_PREFIX);
    }
    
    /**
     * Validate MCC to mention level
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC
     */
    public boolean validateMCC(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.VALIDATE_MCC))
        {
            throw new MCCWorkflowException("MCC can not be validate as current MCC status does not allow to perform the action or container data or structure are not valid for " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        return _setMCCStatus(container, __MCC_VALIDATION_PREFIX, date, user, comment);
    }
    
    /**
     * Refuse control of MCC to orgunit level
     * @param container the container
     * @param user the user
     * @param notify to notify author of precede action by mail
     * @param comment optional comment to add to mail
     * @param reset true to reset MCC workflow
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate MCC
     */
    public boolean refuseMCCOrgunitControl(Container container, UserIdentity user, boolean notify, String comment, boolean reset) throws MCCWorkflowException
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.REFUSE_MCC_ORGUNIT_CONTROL))
        {
            throw new MCCWorkflowException("MCC can not be invalidate as current MCC status does not allow to perform the action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        if (notify)
        {
            List<UserIdentity> recipients = new ArrayList<>();
            Optional.of(getMCCMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
            if (reset && RulesManager.isRulesEnabled())
            {
                Optional.of(getRulesMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
            }
            _notifyContributor(container, user, MCCWorkflowAction.REFUSE_MCC_ORGUNIT_CONTROL, recipients, comment);
        }
        
        return reset ? removeMCCWorkflow(container) : _removeMCCStatus(container, __MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if MCC are validated to mention level
     * @param container the container
     * @return <code>true</code> if MCC are validated to mention level
     */
    public boolean isMCCValidated(Container container)
    {
        return _getMCCStatus(container, __MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Validate rules
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate rules
     */
    public boolean validateRules(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.VALIDATE_RULES))
        {
            throw new MCCWorkflowException("Rule can not be validated as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        return _setMCCStatus(container, __RULES_VALIDATION_PREFIX, date, user, comment);
    }
    
    /**
     * Refuse control of rules to orgunit level
     * @param container the container
     * @param user the user
     * @param notify to notify author of precede action by mail
     * @param comment optional comment to add to mail
     * @param reset true to reset MCC workflow
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate rules
     */
    public boolean refuseRulesOrgUnitControl(Container container, UserIdentity user, boolean notify, String comment, boolean reset) throws MCCWorkflowException
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.REFUSE_RULES_ORGUNIT_CONTROL))
        {
            throw new MCCWorkflowException("Rule can not be invalidated as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        if (notify)
        {
            List<UserIdentity> recipients = new ArrayList<>();
            Optional.of(getRulesMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
            if (reset)
            {
                Optional.of(getMCCMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
            }
            _notifyContributor(container, user, MCCWorkflowAction.REFUSE_RULES_ORGUNIT_CONTROL, recipients, comment);
        }
        
        return reset ? removeMCCWorkflow(container) : _removeMCCStatus(container, __RULES_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if rules are validated to mention level
     * @param container the container
     * @return <code>true</code> if rules are validated to mention level
     */
    public boolean isRulesValidated(Container container)
    {
        return _getMCCStatus(container, __RULES_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if MCC have been controlled by CFVU
     * @param container the container
     * @return <code>true</code> if MCC have been controlled by CFVU
     */
    public boolean isMCCCFVUControlled(Container container)
    {
        return _getMCCStatus(container, __MCC_CFVU_CONTROL_PREFIX);
    }
    
    /**
     * Determines if MCC are validated by CFVU
     * @param container the container
     * @return <code>true</code> if MCC are validated by CFVU
     */
    public boolean isMCCCFVUValidated(Container container)
    {
        return _getMCCStatus(container, __MCC_CFVU_VALIDATION_PREFIX);
    }
    
    /**
     * CFVU control of MCC
     * @param container the container
     * @param validationDate the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC to orgunit level
     */
    public boolean controlMCCForCVFU(Container container, LocalDate validationDate, UserIdentity user, String comment)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.CONTROL_CFVU))
        {
            throw new MCCWorkflowException("MCC can not be CFVU controlled as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        return _setMCCStatus(container, __MCC_CFVU_CONTROL_PREFIX, validationDate, user, comment);
    }
    
    /**
     * Refuse CFVU validation of MCC
     * @param container the container
     * @param user the user
     * @param notify to notify author of precede action by mail
     * @param comment optional comment to add to mail
     * @param reset true to reset MCC workflow
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate MCC for orgunit level
     */
    public boolean refuseCFVUValidation(Container container, UserIdentity user, boolean notify, String comment, boolean reset) throws MCCWorkflowException
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.REFUSE_VALIDATION_CFVU))
        {
            throw new MCCWorkflowException("Control of MCC can not be invalidate as current MCC status does not allow to perform this action for " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        if (notify)
        {
            List<UserIdentity> recipients = new ArrayList<>();
            Optional.of(getMCCCFVUControlStep(container)).ifPresent(s -> recipients.add(s.author()));
            if (reset)
            {
                Optional.of(getMCCOrgunitValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                Optional.of(getMCCOrgunitControlStep(container)).ifPresent(s -> recipients.add(s.author()));
                Optional.of(getMCCMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                if (RulesManager.isRulesEnabled())
                {
                    Optional.of(getRulesMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                }
            }
            _notifyContributor(container, user, MCCWorkflowAction.REFUSE_VALIDATION_CFVU, recipients, comment);
        }
        
        return reset ? removeMCCWorkflow(container) : _removeMCCStatus(container, __MCC_CFVU_CONTROL_PREFIX);
    }
    
    /**
     * Get minimun date for MCC orgunit control
     * @param container the container
     * @return the minimun date
     */
    public LocalDate getMinDateForMCCOrgUnitControl(Container container)
    {
        if (!isMCCValidated(container) || RulesManager.isRulesEnabled() && !isRulesValidated(container))
        {
            throw new MCCWorkflowException("MCC current status does not allow to control MCC to orgunit level");
        }
        
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        LocalDate mccMentionDate = composite.getValue(__MCC_VALIDATION_PREFIX + __DATE_SUFFIX);
        LocalDate rulesMentionDate = RulesManager.isRulesEnabled() ? composite.getValue(__RULES_VALIDATION_PREFIX + __DATE_SUFFIX) : null;
        
        return rulesMentionDate == null ? mccMentionDate : mccMentionDate.compareTo(rulesMentionDate) > 0 ? mccMentionDate : rulesMentionDate;
    }
    
    /**
     * Get minimun date for MCC orgunit validation
     * @param container the container
     * @return the minimun date
     */
    public LocalDate getMinDateForMCCOrgUnitValidation(Container container)
    {
        if (!isMCCOrgUnitControlled(container))
        {
            throw new MCCWorkflowException("MCC current status does not allow to validate MCC to orgunit level");
        }
        
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        return composite.getValue(__MCC_ORGUNIT_CONTROL_PREFIX + __DATE_SUFFIX);
    }
    
    /**
     * Get minimun date for MCC CFVU control
     * @param container the container
     * @return the minimun date
     */
    public LocalDate getMinDateForMCCCFVUControl(Container container)
    {
        if (!isMCCOrgUnitValidated(container))
        {
            throw new MCCWorkflowException("MCC current status does not allow to control MCC to CFVU level");
        }
        
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        return composite.getValue(__MCC_ORGUNIT_VALIDATION_PREFIX + __DATE_SUFFIX);
    }
    
    /**
     * Get minimun date for MCC CFVU validation
     * @param container the container
     * @return the minimun date
     */
    public LocalDate getMinDateForMCCCFVUValidation(Container container)
    {
        if (!isMCCCFVUControlled(container))
        {
            throw new MCCWorkflowException("MCC current status does not allow to validate MCC to CFVU level");
        }
        
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        return composite.getValue(__MCC_CFVU_CONTROL_PREFIX + __DATE_SUFFIX);
    }
    
    /**
     * Get the MCC CFVU validation date
     * @param container the container
     * @return the MCC CFVU validation date
     */
    public LocalDate getMCCCFVUValidationDate(Container container)
    {
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        return composite.getValue(__MCC_CFVU_VALIDATION_PREFIX + __DATE_SUFFIX);
    }
    
    /**
     * CFVU validation of MCC
     * @param container the container
     * @param validationDate the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC to orgunit level
     */
    public boolean validateMCCForCVFU(Container container, LocalDate validationDate, UserIdentity user, String comment)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.VALIDATE_CFVU))
        {
            throw new MCCWorkflowException("MCC can not be validated to CFVU level as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        return _setMCCStatus(container, __MCC_CFVU_VALIDATION_PREFIX, validationDate, user, comment);
    }
    
    /**
     * Remove validation attribute for 'CFVU MCC validated' state
     * @param container the container
     * @param user the user
     * @param notify to notify author of precede action by mail
     * @param comment optional comment to add to mail
     * @param reset true to reset MCC workflow
     * @return <code>true</code> if the container has changed.
     */
    public boolean invalidateMCCForCVFU(Container container, UserIdentity user, boolean notify, String comment, boolean reset)
    {
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isAvailableAction(container, MCCWorkflowAction.INVALIDATE_CFVU))
        {
            throw new MCCWorkflowException("MCC can not be invalidated to CFVU level as current MCC status does not allow to perform this action " + container, ExceptionType.UNAVAILABLE_ACTION);
        }
        
        if (notify)
        {
            List<UserIdentity> recipients = new ArrayList<>();
            Optional.of(getMCCCFVUControlStep(container)).ifPresent(s -> recipients.add(s.author()));
            if (reset)
            {
                
                Optional.of(getMCCOrgunitValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                Optional.of(getMCCOrgunitControlStep(container)).ifPresent(s -> recipients.add(s.author()));
                Optional.of(getMCCCFVUControlStep(container)).ifPresent(s -> recipients.add(s.author()));
                Optional.of(getMCCMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                if (RulesManager.isRulesEnabled())
                {
                    Optional.of(getRulesMentionValidationStep(container)).ifPresent(s -> recipients.add(s.author()));
                }
            }
            _notifyContributor(container, user, MCCWorkflowAction.INVALIDATE_CFVU, recipients, comment);
        }
        
        return reset ? removeMCCWorkflow(container) : _removeMCCStatus(container, __MCC_CFVU_VALIDATION_PREFIX);
    }
    
    private void _notifyContributor(Container container, UserIdentity issuer, MCCWorkflowAction revertAction, List<UserIdentity> recipients, String comment)
    {
        try
        {
            String defaultLanguage = StringUtils.defaultIfBlank(container.getLanguage(), _userLanguagesManager.getDefaultLanguage());
            Map<String, Set<String>> emailsByLanguage = recipients.stream()
                    .map(_userManager::getUser)
                    .filter(Objects::nonNull)
                    .map(user -> Pair.of(user.getLanguage(), user.getEmail()))
                    .filter(p -> StringUtils.isNotBlank(p.getRight()))
                    .collect(Collectors.groupingBy(
                            p -> {
                                return StringUtils.defaultIfBlank(p.getLeft(), defaultLanguage);
                            },
                            Collectors.mapping(
                                Pair::getRight,
                                Collectors.toSet()
                            )
                        )
                    );
            
            if (emailsByLanguage.isEmpty())
            {
                return;
            }
            
            User user = _userManager.getUser(issuer);
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("title", new I18nizableText(container.getTitle()));
            i18nParams.put("code", new I18nizableText(container.getDisplayCode()));
            i18nParams.put("user", new I18nizableText(_userManager.getUser(issuer).getFullName()));
            
            I18nizableText i18nSubject = new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MAIL_SUBJECT_" + revertAction.name(), i18nParams);
            I18nizableText i18nMessage = new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MAIL_BODY_" + revertAction.name(), i18nParams);
            
            MailBodyBuilder mailBody = StandardMailBodyHelper.newHTMLBody()
                    .withTitle(i18nSubject)
                    .withMessage(i18nMessage)
                    .withLink(_contentHelper.getContentBOUrl(container, Map.of()), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MAIL_BODY_GO_TO_CONTENT"));
            
            if (StringUtils.isNotEmpty(comment))
            {
                UserInput userInput = new UserInput(user, ZonedDateTime.now(), comment.replaceAll("\r?\n", "<br/>"));
                mailBody.withUserInputs(List.of(userInput));
            }
            
            for (String userLanguage : emailsByLanguage.keySet())
            {
                String subject = _i18nUtils.translate(i18nSubject, userLanguage);
                String htmlBody = mailBody.withLanguage(userLanguage)
                                          .build();
                
                List<String> emails = new ArrayList<>(emailsByLanguage.get(userLanguage));
                try
                {
                    SendMailHelper.newMail()
                        .withSubject(subject)
                        .withHTMLBody(htmlBody)
                        .withSender(user.getEmail())
                        .withRecipients(emails)
                        .withAsync(true)
                        .sendMail();
                }
                catch (MessagingException | IOException e)
                {
                    getLogger().warn("Could not send MCC workflow notification mails to " + emails, e);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().warn("Could not build MCC workflow notification mail", e);
        }
    }
    
    /**
     * Removes attributes linked to MCC workflow
     * @param container the container
     * @return <code>true</code> if the program has changed.
     */
    public boolean removeMCCWorkflow(Container container)
    {
        container.removeValue(__MCC_WORKFLOW_COMPOSITE);
        return saveContent(container);
    }
    
    /**
     * Removes attributes linked to MCC workflow
     * @param container the container
     * @return <code>true</code> if the program has changed.
     */
    public boolean removeMCCValidatedPDF(Container container)
    {
        container.removeValue(MCC_VALIDATED_PDF_REPEATER);
        return saveContent(container);
    }
    
    private boolean _setMCCStatus(Container container, String prefixAttributeName, LocalDate validationDate, UserIdentity user, String comment)
    {
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE, true);
        composite.setValue(prefixAttributeName + __STATUS_SUFFIX, true);
        setWorkflowStep(container, composite, prefixAttributeName, validationDate, user, comment);
        
        return saveContentAndNotify(container);
    }
    
    private boolean _removeMCCStatus(Container container, String prefixAttributeName)
    {
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE, true);
        composite.setValue(prefixAttributeName + __STATUS_SUFFIX, false);
        removeWorkflowStep(container, composite, prefixAttributeName);
        
        return saveContentAndNotify(container);
    }
    
    private boolean _getMCCStatus(Container container, String attributeName)
    {
        return container.getValue(__MCC_WORKFLOW_COMPOSITE + ModelItem.ITEM_PATH_SEPARATOR + attributeName + __STATUS_SUFFIX, false, false);
    }
    
    /**
     * Get the MCC workflow step for validation of rules to mention level
     * @param container the container
     * @return the MCC workflow step or null if rules are not validated to mention level
     */
    public ODFWorkflowStep getRulesMentionValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.RULES_VALIDATED, __RULES_VALIDATION_PREFIX);
    }
    
    /**
     * Get the MCC workflow step for validation of MCC to mention level
     * @param container the container
     * @return the MCC workflow step or null if MCC are not validated to mention level
     */
    public ODFWorkflowStep getMCCMentionValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_VALIDATED, __MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Get the MCC workflow step when MCC have been controlled and ready to be validated by orgunit
     * @param container the container
     * @return the MCC workflow step or null if MCC are not ready for orgunit validation
     */
    public ODFWorkflowStep getMCCOrgunitControlStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_ORGUNIT_CONTROLLED, __MCC_ORGUNIT_CONTROL_PREFIX);
    }
    
    /**
     * Get the MCC workflow step for validation of MCC to orgunit level
     * @param container the container
     * @return the MCC workflow step or null if MCC are not validated to orgunit level
     */
    public ODFWorkflowStep getMCCOrgunitValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_ORGUNIT_VALIDATED, __MCC_ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Get the MCC workflow step when MCC have been controlled and ready to be validated by CFVU
     * @param container the container
     * @return the MCC workflow step or null if MCC are not ready for CFVY validation
     */
    public ODFWorkflowStep getMCCCFVUControlStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_CFVU_CONTROLLED, __MCC_CFVU_CONTROL_PREFIX);
    }
    
    /**
     * Get the MCC workflow step for CFVU validation of MCC
     * @param container the container
     * @return the MCC workflow step or null if MCC are not CFVU validated
     */
    public ODFWorkflowStep getMCCCFVUValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_CFVU_VALIDATED, __MCC_CFVU_VALIDATION_PREFIX);
    }
    
    private ODFWorkflowStep _getMCCWorkflowStep(Container container, MCCWorkflowStep step, String attributeNamePrefix)
    {
        if (container.hasValue(__MCC_WORKFLOW_COMPOSITE))
        {
            boolean status = _getMCCStatus(container, attributeNamePrefix);
            if (status)
            {
                ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
                return getWorkflowStep(composite, attributeNamePrefix, step.name());
            }
        }
        
        return null;
    }
    
    /**
     * Get the entry holding the most recent PDF for validated MCC on a year.
     * @param container the container
     * @return the entry or null if no PDF available
     */
    public ModelAwareRepeaterEntry getLastMCCValidatedEntry(Container container)
    {
        return Optional.ofNullable(container.getRepeater(MCC_VALIDATED_PDF_REPEATER))
            .map(ModelAwareRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .filter(e -> e.hasValue("pdf"))
            .max((entry1, entry2) -> entry1.<LocalDate>getValue("date").compareTo(entry2.<LocalDate>getValue("date")))
            .orElse(null);
    }
    
    private final class MCCStatusComparator implements Comparator<Container>
    {
        public int compare(Container c1, Container c2)
        {
            boolean mccValidated1 = isMCCCFVUValidated(c1);
            boolean mccValidated2 = isMCCCFVUValidated(c2);
            
            if (mccValidated1)
            {
                return mccValidated2 ? 0 : 1;
            }
            
            if (mccValidated2)
            {
                return -1; // p2 has a higher mcc status
            }
            
            boolean mccCFVUControlled1 = isMCCCFVUControlled(c1);
            boolean mccCFVUControlled2 = isMCCCFVUControlled(c2);
            
            if (mccCFVUControlled1)
            {
                return mccCFVUControlled2 ? 0 : 1;
            }
            if (mccCFVUControlled2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccOrgunitValidated1 = isMCCOrgUnitValidated(c1);
            boolean mccOrgunitValidated2 = isMCCOrgUnitValidated(c2);
            
            if (mccOrgunitValidated1)
            {
                return mccOrgunitValidated2 ? 0 : 1;
            }
            if (mccOrgunitValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccOrgunitControlled1 = isMCCOrgUnitControlled(c1);
            boolean mccOrgunitControlled2 = isMCCOrgUnitControlled(c2);
            
            if (mccOrgunitControlled1)
            {
                return mccOrgunitControlled2 ? 0 : 1;
            }
            if (mccOrgunitControlled2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean rulesMentionValidated1 = RulesManager.isRulesEnabled() && isRulesValidated(c1);
            boolean mccMentionValidated1 = isMCCValidated(c1);
            boolean rulesValidated2 = RulesManager.isRulesEnabled() && isRulesValidated(c2);
            boolean mccMentionValidated2 = isMCCValidated(c2);
            
            if (rulesMentionValidated1 && mccMentionValidated1)
            {
                if (!mccMentionValidated2 || !rulesValidated2)
                {
                    return 1; // c1 has a higher mcc status
                }
                else
                {
                    return 0; // same status
                }
            }
            
            if (rulesMentionValidated1 || mccMentionValidated1)
            {
                if (rulesValidated2 && !mccMentionValidated2 || !rulesValidated2 && mccMentionValidated2)
                {
                    return 0; // same status
                }
                else if (rulesValidated2 && mccMentionValidated2)
                {
                    return -1; // c2 has a higher mcc status
                }
                else
                {
                    return 1; // c1 has a higher mcc status
                }
            }
            
            return !rulesValidated2 && !mccMentionValidated2 ? 0 : -1;
        }
    }
    
    private final class ContainerMCCRulesStatusComparator implements Comparator<Container>
    {
        public int compare(Container c1, Container c2)
        {
            boolean mccValidated1 = isMCCCFVUValidated(c1);
            boolean mccValidated2 = isMCCCFVUValidated(c2);
            
            if (mccValidated1)
            {
                return mccValidated2 ? 0 : 1;
            }
            
            if (mccValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccCFVUControlled1 = isMCCCFVUControlled(c1);
            boolean mccCFVUControlled2 = isMCCCFVUControlled(c2);
            
            if (mccCFVUControlled1)
            {
                return mccCFVUControlled2 ? 0 : 1;
            }
            if (mccCFVUControlled2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccOrgunitValidated1 = isMCCOrgUnitValidated(c1);
            boolean mccOrgunitValidated2 = isMCCOrgUnitValidated(c2);
            
            if (mccOrgunitValidated1)
            {
                return mccOrgunitValidated2 ? 0 : 1;
            }
            if (mccOrgunitValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccOrgunitControlled1 = isMCCOrgUnitControlled(c1);
            boolean mccOrgunitControlled2 = isMCCOrgUnitControlled(c2);
            
            if (mccOrgunitControlled1)
            {
                return mccOrgunitControlled2 ? 0 : 1;
            }
            if (mccOrgunitControlled2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean rulesValidated1 = RulesManager.isRulesEnabled() && isRulesValidated(c1);
            boolean rulesValidated2 = RulesManager.isRulesEnabled() && isRulesValidated(c2);
            
            if (rulesValidated1)
            {
                return rulesValidated2 ? 0 : 1;
            }
            
            return rulesValidated2 ? -1 : 0;
        }
    }
    
    private final class ContainerMCCFieldsStatusComparator implements Comparator<Container>
    {
        public int compare(Container c1, Container c2)
        {
            boolean mccValidated1 = isMCCCFVUValidated(c1);
            boolean mccValidated2 = isMCCCFVUValidated(c2);
            
            if (mccValidated1)
            {
                return mccValidated2 ? 0 : 1;
            }
            
            if (mccValidated2)
            {
                return -1; // p2 has a higher mcc status
            }
            
            boolean mccCFVUControlled1 = isMCCCFVUControlled(c1);
            boolean mccCFVUControlled2 = isMCCCFVUControlled(c2);
            
            if (mccCFVUControlled1)
            {
                return mccCFVUControlled2 ? 0 : 1;
            }
            if (mccCFVUControlled2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccOrgunitValidated1 = isMCCOrgUnitValidated(c1);
            boolean mccOrgunitValidated2 = isMCCOrgUnitValidated(c2);
            
            if (mccOrgunitValidated1)
            {
                return mccOrgunitValidated2 ? 0 : 1;
            }
            if (mccOrgunitValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccOrgunitControlled1 = isMCCOrgUnitControlled(c1);
            boolean mccOrgunitControlled2 = isMCCOrgUnitControlled(c2);
            
            if (mccOrgunitControlled1)
            {
                return mccOrgunitControlled2 ? 0 : 1;
            }
            if (mccOrgunitControlled2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccMentionValidated1 = isMCCValidated(c1);
            boolean mccMentionValidated2 = isMCCValidated(c2);
            
            if (mccMentionValidated1)
            {
                return mccMentionValidated2 ? 0 : 1;
            }
            
            return mccMentionValidated2 ? -1 : 0;
        }
    }
}
