/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.tool;

import java.io.File;
import java.io.FileFilter;
import java.time.ZonedDateTime;
import java.util.regex.Pattern;

import org.ametys.core.util.DateUtils;

/**
 * File filter for pilotage files and search of these files.
 */
public class PilotageFileFilter implements FileFilter
{
    private static final Pattern __FILENAME_PATTERN = Pattern.compile("^.*\\.zip$", Pattern.CASE_INSENSITIVE);
    
    private ZonedDateTime _lastModifiedAfter;
    private ZonedDateTime _lastModifiedBefore;
    
    /**
     * Constructor
     * @param lastModifiedAfter Filter on the date after
     * @param lastModifiedBefore Filter on the date before
     */
    public PilotageFileFilter(ZonedDateTime lastModifiedAfter, ZonedDateTime lastModifiedBefore)
    {
        _lastModifiedAfter = lastModifiedAfter;
        _lastModifiedBefore = lastModifiedBefore;
    }
    
    @Override
    public boolean accept(File pathname)
    {
        // Only accept files
        if (pathname.isDirectory())
        {
            return false;
        }
        
        // Only accept files with zip extension
        String fileName = pathname.getName().toLowerCase();
        if (!__FILENAME_PATTERN.matcher(fileName).find())
        {
            return false;
        }

        // Check the last modified date
        if (_lastModifiedAfter != null || _lastModifiedBefore != null)
        {
            ZonedDateTime lastModified = DateUtils.asZonedDateTime(pathname.lastModified());
            if (_lastModifiedAfter != null && lastModified.isBefore(_lastModifiedAfter))
            {
                return false;
            }
            if (_lastModifiedBefore != null && lastModified.isAfter(_lastModifiedBefore.plusDays(1)))
            {
                return false;
            }
        }
        
        return true;
    }
}
