/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.consistency;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.odf.ODFHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * The abstract class for {@link ConsistencyAnalysis} which parse the ID, label and short ID.
 */
public abstract class AbstractConsistencyAnalysis implements ConsistencyAnalysis, Configurable, Serviceable, PluginAware
{
    /** Plugin name. */
    protected String _pluginName;
    
    /** Analysis complete id. */
    protected String _id;
    
    /** Analysis short id. */
    protected String _shortId;
    
    /** Label. */
    protected I18nizableText _label;
    
    /** Description. */
    protected I18nizableText _description;
    
    /** Priority. */
    protected int _priority;

    /** The ODF helper. */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        // Needed in almost all analyses
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _id = id;
    }

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), "plugin." + _pluginName);
        _description = I18nizableText.parseI18nizableText(configuration.getChild("description"), "plugin." + _pluginName);
        _shortId = configuration.getChild("shortId").getValue();
        _priority = configuration.getChild("priority").getValueAsInteger(50);
    }
    
    @Override
    public String getId()
    {
        return _id;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }

    @Override
    public I18nizableText getDescription()
    {
        return _description;
    }
    
    @Override
    public String getShortId()
    {
        return _shortId;
    }

    @Override
    public int getPriority()
    {
        return _priority;
    }
}
