/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.catalog;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.data.holder.group.IndexableRepeaterEntry;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.catalog.CopyCatalogUpdater;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The thematic catalog copy updater.
 * Also used on deletion.
 */
public class ThematicsUpdater extends AbstractLogEnabled implements CopyCatalogUpdater, Serviceable
{
    private AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        // Copy metadata to build rules number from the initial content to the copied content
        for (Content initialContent : copiedContents.keySet())
        {
            if (initialContent instanceof Container)
            {
                Content copiedContent = copiedContents.get(initialContent);
                
                // ametys-internal:TA
                _copyInternalValue(initialContent, copiedContent, RulesManager.ADDITIONAL_THEMATICS_PREFIX);
                
                // ametys-internal:TA[nb]-ruleNumber
                for (ModelAwareRepeaterEntry thematic : _getRepeaterEntries(copiedContent, RulesManager.CONTAINER_THEMATICS))
                {
                    String code = thematic.getValue(RulesManager.THEMATIC_CODE);
                    _copyInternalValue(initialContent, copiedContent, code + RulesManager.RULES_NB_SUFFIX);
                }
                
                // ametys-internal:TS[nb]-ruleNumber
                for (ModelAwareRepeaterEntry rule : _getRepeaterEntries(copiedContent, RulesManager.CONTAINER_RULES))
                {
                    if (rule.hasValue(RulesManager.RULE_THEMATIC_CODE))
                    {
                        String code = rule.getValue(RulesManager.RULE_THEMATIC_CODE);
                        _copyInternalValue(initialContent, copiedContent, code + RulesManager.RULES_NB_SUFFIX);
                    }
                }
                
                // Save the copied content
                ((ModifiableContent) copiedContent).saveChanges();
            }
        }
    }
    
    private List<? extends ModelAwareRepeaterEntry> _getRepeaterEntries(Content content, String repeaterName)
    {
        return Optional.ofNullable(content.getRepeater(repeaterName))
                .map(r -> r.getEntries())
                .orElseGet(List::of);
    }
    
    private void _copyInternalValue(Content sourceContent, Content targetContent, String metadataName)
    {
        targetContent.getInternalDataHolder().setValue(metadataName, sourceContent.getInternalDataHolder().getValue(metadataName, 0L));
    }
    
    public List<Content> getAdditionalContents(String catalogName)
    {
        return _getContents(catalogName);
    }
    
    public Map<Content, Content> copyAdditionalContents(String initialCatalogName, String newCatalogName)
    {
        Map<Content, Content> copiedContents = new HashMap<>();
        
        // Get the RG of the catalog to copy
        List<DefaultContent> contentsToCopy = _getContents(initialCatalogName);
        
        for (DefaultContent thematic : contentsToCopy)
        {
            // Log if the targeted catalog already contains the thematic
            if (_thematicExists(thematic, newCatalogName))
            {
                getLogger().info("A thematic already exists with the same code, catalog and language [{}, {}, {}]", thematic.getValue("code"), newCatalogName, thematic.getLanguage());
            }
            // Copy the thematic in the targeted catalog
            else
            {
                ModifiableContent newThematic = thematic.copyTo((ModifiableTraversableAmetysObject) thematic.getParent(), NameHelper.filterName(thematic.getTitle()));
                
                // Update ruleNumber value ametys-internal:TS[nb]-ruleNumber
                _copyInternalValue(thematic, newThematic, thematic.getValue(RulesManager.THEMATIC_CODE) + RulesManager.RULES_NB_SUFFIX);
                newThematic.saveChanges();
                
                // Prepare values for EditContentFunction
                
                Map<String, Object> values = new HashMap<>();
                values.put("catalog", newCatalogName);
                
                List<Integer> positions = new ArrayList<>();
                List<Map<String, Object>> repeaterValues = new ArrayList<>();
                ModifiableIndexableRepeater repeater = newThematic.getRepeater("rules");
                if (repeater != null)
                {
                    for (IndexableRepeaterEntry entry : repeater.getEntries())
                    {
                        String status = entry.getValue("status");
                        if (!"NON_MODIFIED".equals(status) && !"SUSPENDED".equals(status))
                        {
                            positions.add(entry.getPosition());
                            repeaterValues.add(Map.of("status", "NON_MODIFIED"));
                        }
                    }
                }
                
                if (!repeaterValues.isEmpty())
                {
                    values.put("rules", SynchronizableRepeater.replace(repeaterValues, positions));
                }
                
                newThematic.synchronizeValues(values);
                newThematic.saveChanges();
                
                copiedContents.put(thematic, newThematic);
            }
        }
        
        return copiedContents;
    }
    
    private boolean _thematicExists(Content thematic, String newCatalogName)
    {
        return _getContents(newCatalogName, thematic).findAny().isPresent();
    }

    private <T extends Content> List<T> _getContents(String catalogName)
    {
        return this.<T>_getContents(catalogName, null).toList();
    }
    
    private <T extends Content> Stream<T> _getContents(String catalogName, Content thematic)
    {
        AndExpression expression = new AndExpression();
        expression.add(new ContentTypeExpression(Operator.EQ, RulesManager.THEMATIC_CONTENT_TYPE));
        expression.add(new StringExpression("catalog", Operator.EQ, catalogName));
        if (thematic != null)
        {
            expression.add(new LanguageExpression(Operator.EQ, thematic.getLanguage()));
            expression.add(new StringExpression("code", Operator.EQ, thematic.getValue("code")));
        }
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        return _resolver.<T>query(query).stream();
    }
}
