/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.search;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.model.impl.AbstractStaticSearchModelCriterionDefinition;
import org.ametys.cms.search.query.ContentIdQuery;
import org.ametys.cms.search.query.MatchAllQuery;
import org.ametys.cms.search.query.MatchNoneQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * UI criteria to retrieve micro skills attached to a program
 */
public class MicroSkillsAttachedToCourseCriterionDefinition extends AbstractStaticSearchModelCriterionDefinition<String>
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The odf skills helper. */
    protected ODFSkillsHelper _odfSkillsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfSkillsHelper = (ODFSkillsHelper) manager.lookup(ODFSkillsHelper.ROLE);
    }
    
    @Override
    public Operator getOperator()
    {
        // Not suppose to be used
        return Operator.EQ;
    }
    
    @Override
    protected String getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    @Override
    public Query getQuery(Object value, Operator customOperator, Map<String, Object> allValues, String language, Map<String, Object> contextualParameters)
    {
        if (value instanceof EducationalPath educationalPath)
        {
            List<String> programItemIds = educationalPath.getProgramItemIds();
            return _getQueryForCourseAndProgram(programItemIds);
        }
        else if (value instanceof String educationalPath)
        {
            List<String> programItemIds = List.of(educationalPath.split(EducationalPath.PATH_SEGMENT_SEPARATOR));
            return _getQueryForCourseAndProgram(programItemIds);
        }
        
        return new MatchAllQuery();
    }
    
    private Query _getQueryForCourseAndProgram(List<String> programItemIds)
    {
        // If the path given contains at least a program and a course
        if (programItemIds.size() >= 2)
        {
            String programId = programItemIds.getFirst();
            String courseId = programItemIds.getLast();
            
            try
            {
                Content content = _resolver.resolveById(courseId);
                if (content instanceof Course course)
                {
                    // Retrieve the acquired skills of this course for the program
                    ContentValue[] acquiredSkills = course.getAcquiredSkills(programId);
                    if (acquiredSkills != null)
                    {
                        String[] courseMicroSkillsIds = Arrays.stream(acquiredSkills)
                                                                  .map(ContentValue::getContentId)
                                                                  .toArray(String[]::new);
                        if (courseMicroSkillsIds.length > 0)
                        {
                            // Add the ids to the queries
                            return new ContentIdQuery(courseMicroSkillsIds);
                        }
                        else
                        {
                            // Add empty query to not retrieve skills not linked with course
                            return new MatchNoneQuery();
                        }
                    }
                }
            }
            catch (AmetysRepositoryException e)
            {
                _logger.warn("The search of microskills could not be done because the requested course of id '{}' does not exist.", courseId, e);
            }
        }
        
        return new MatchAllQuery();
    }
}
