/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.validator;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.IndexableRepeater;
import org.ametys.cms.data.holder.group.IndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.rights.ODFRightHelper;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedContent;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.UntouchedValue;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.disableconditions.DefaultDisableConditionsEvaluator;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Checks that the mcc skills are compatible with the course path in the mccSessions
 */
public class CourseMccSkillsValidator extends AbstractContentValidator implements Serviceable, Initializable, Contextualizable
{
    private static final String __MCC_SESSION1 = "mccSession1";
    private static final String __MCC_SESSION2 = "mccSession2";
    
    private static final String __MCC_SESSION_COMMON = "common";
    private static final String __MCC_SESSION_PATH = "path";
    private static final String __MCC_SESSION_NOTES = "notes";
    private static final String __MCC_SESSION_NOTES_SKILLS = "skills";
    private static final String __MCC_SESSION_NOTES_SKILLS_SKILL = "skill";
    private static final String __MCC_SESSION_NOTES_SKILLS_PATH = "path";
    
    private DisableConditionsEvaluator _disableConditionsEvaluator;
    private ODFHelper _odfHelper;
    private RightManager _rightManager;
    private CurrentUserProvider _currentUserProvider;
    
    private boolean _isSkillsEnabled;
    private Context _context;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _disableConditionsEvaluator = (DisableConditionsEvaluator) manager.lookup(DefaultDisableConditionsEvaluator.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void initialize() throws ConfigurationException
    {
        super.initialize();
        _isSkillsEnabled = Config.getInstance().getValue("odf.skills.enabled");
    }
    
    public ValidationResult validate(Content content)
    {
        ValidationResult result = new ValidationResult();
        
        if (_isSkillsEnabled && content instanceof Course course)
        {
            // Get the educational path of the course if it is not shared
            Optional<EducationalPath> courseEducationalPath = _getUniqueEducationalPath(course);
            
            // Check the unicity of the skills under each note by educational path and check that every skills is attached to the right program
            result.addResult(_checkSession(course, __MCC_SESSION1, courseEducationalPath));
            result.addResult(_checkSession(course, __MCC_SESSION2, courseEducationalPath));
        }
        
        return result;
    }
    
    private ValidationResult _checkSession(Course course, String mccSession, Optional<EducationalPath> courseEducationalPath)
    {
        ValidationResult result = new ValidationResult();
        
        RepeaterDefinition mccSessionDefinition = (RepeaterDefinition) course.getDefinition(mccSession);
        
        // Check if the mcc session is not disabled
        if (!_disableConditionsEvaluator.evaluateDisableConditions(mccSessionDefinition, mccSession, course))
        {
            // Build common i18n parameters
            RepeaterDefinition noteDefinition = (RepeaterDefinition) mccSessionDefinition.getModelItem(__MCC_SESSION_NOTES);
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("sessionRepeater", mccSessionDefinition.getLabel());
            i18nParams.put("noteRepeater", noteDefinition.getLabel());
            i18nParams.put("skillRepeater", noteDefinition.getModelItem(__MCC_SESSION_NOTES_SKILLS).getLabel());
            
            // Process each mcc session entry
            Optional.ofNullable(course.getRepeater(mccSession))
                .map(IndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .map(entry -> _checkSessionEntry(course, entry, courseEducationalPath, i18nParams))
                .forEach(result::addResult);
        }
        
        return result;
    }
    
    private ValidationResult _checkSessionEntry(Course course, IndexableRepeaterEntry sessionEntry, Optional<EducationalPath> courseEducationalPath, Map<String, I18nizableTextParameter> commonI18nParams)
    {
        ValidationResult result = new ValidationResult();
        
        // Build common i18n parameters
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
        i18nParams.put("sessionPosition", new I18nizableText(Integer.toString(sessionEntry.getPosition())));
        
        // Get the educational path of the entry if it is not common and if the course educational path is empty
        Optional<EducationalPath> sessionEntryEducationalPath = courseEducationalPath
            .or(() ->
                Optional.of(sessionEntry)
                    // Don't get the educational path for common entries
                    .filter(entry -> !entry.getValue(__MCC_SESSION_COMMON, true, true))
                    .map(entry -> entry.getValue(__MCC_SESSION_PATH))
            );
        
        // Process each note entry
        Optional.ofNullable(sessionEntry.getRepeater(__MCC_SESSION_NOTES))
            .map(IndexableRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .map(noteEntry -> _checkNoteEntry(course, noteEntry, sessionEntryEducationalPath, i18nParams))
            .forEach(result::addResult);
        
        return result;
    }
    
    private ValidationResult _checkNoteEntry(Course course, IndexableRepeaterEntry noteEntry, Optional<EducationalPath> sessionEntryEducationalPath, Map<String, I18nizableTextParameter> commonI18nParams)
    {
        ValidationResult result = new ValidationResult();
        
        // Map used to detect duplicated skill for each educational path in the note entry
        Map<EducationalPath, Set<ContentValue>> skillsByPath = new HashMap<>();
        
        List<? extends IndexableRepeaterEntry> skillEntries = Optional.ofNullable(noteEntry.getRepeater(__MCC_SESSION_NOTES_SKILLS))
            .map(IndexableRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .toList();
        
        // Process each skill entry
        for (IndexableRepeaterEntry skillEntry : skillEntries)
        {
            ContentValue skill = skillEntry.getValue(__MCC_SESSION_NOTES_SKILLS_SKILL);
            EducationalPath educationalPath = sessionEntryEducationalPath.orElseGet(() -> skillEntry.getValue(__MCC_SESSION_NOTES_SKILLS_PATH));
            
            // If one of this data is null, the mandatory validator should react
            if (skill != null && educationalPath != null)
            {
                // Build common i18n parameters
                Map<String, I18nizableTextParameter> noteI18nParams = new HashMap<>(commonI18nParams);
                noteI18nParams.put("notePosition", new I18nizableText(Integer.toString(noteEntry.getPosition())));
                
                // Get the skills already present for the current educational path
                Set<ContentValue> skillsForPath = skillsByPath.computeIfAbsent(educationalPath, __ -> new HashSet<>());
                
                // The skill is already defined for this educational path in this note entry
                if (!skillsForPath.add(skill))
                {
                    // Error because duplication
                    result.addError(_buildDuplicationMessage(noteI18nParams, skill, educationalPath));
                }
                // Size of educational path can be zero if the course has no parent program
                else if (educationalPath.getProgramItemIds().size() > 0)
                {
                    // Check if the skill belong to the course acquired skills for the given program of the educational path
                    String programId = educationalPath.getProgramItemIds().get(0);
                    ContentValue[] acquiredSkills = course.getAcquiredSkills(programId);
                    if (!Arrays.asList(acquiredSkills).contains(skill))
                    {
                        // Skill is not in the course acquired skills, add a warning
                        result.addError(_buildInconsistentMessage(noteI18nParams, skillEntry.getPosition(), skill, course));
                    }
                }
            }
        }
        
        return result;
    }
    
    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        if (_isSkillsEnabled && content instanceof Course course)
        {
            // Get the educational path of the course if it is not shared
            Optional<EducationalPath> courseEducationalPath = _getUniqueEducationalPath(course);
            // Get the current educational paths to check only this context
            List<EducationalPath> currentEducationPaths = _getCurrentEducationPaths(course);

            // Check the unicity of the skills under each note by educational path and check that every skills is attached to the right program
            // Check will be done only for entries that part of the current edited paths
            result.addResult(_checkSession(course, values, __MCC_SESSION1, courseEducationalPath, currentEducationPaths));
            result.addResult(_checkSession(course, values, __MCC_SESSION2, courseEducationalPath, currentEducationPaths));
        }
        
        return result;
    }
    
    private ValidationResult _checkSession(Course course, Map<String, Object> values, String mccSession, Optional<EducationalPath> courseEducationalPath, List<EducationalPath> currentEducationPaths)
    {
        ValidationResult result = new ValidationResult();
        
        List<Map<String, Object>> mccSessionEntries = _getRepeaterEntries(values, mccSession);
        if (!mccSessionEntries.isEmpty())
        {
            // Build common i18n parameters
            RepeaterDefinition mccSessionDefinition = (RepeaterDefinition) course.getDefinition(mccSession);
            RepeaterDefinition noteDefinition = (RepeaterDefinition) mccSessionDefinition.getModelItem(__MCC_SESSION_NOTES);
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("sessionRepeater", mccSessionDefinition.getLabel());
            i18nParams.put("noteRepeater", noteDefinition.getLabel());
            i18nParams.put("skillRepeater", noteDefinition.getModelItem(__MCC_SESSION_NOTES_SKILLS).getLabel());
            
            // Process each mcc session entry
            int entryPosition = 1;
            for (Map<String, Object> sessionEntry : mccSessionEntries)
            {
                String prefix = mccSession + "[" + entryPosition + "]";
                result.addResult(_checkSessionEntry(course, prefix, sessionEntry, entryPosition, courseEducationalPath, currentEducationPaths, i18nParams));
                entryPosition++;
            }
        }
        
        return result;
    }

    private ValidationResult _checkSessionEntry(Course course, String prefixPath, Map<String, Object> sessionEntry, int sessionEntryPosition, Optional<EducationalPath> courseEducationalPath, List<EducationalPath> currentEducationPaths, Map<String, I18nizableTextParameter> commonI18nParams)
    {
        ValidationResult result = new ValidationResult();
        
        List<Map<String, Object>> noteEntries = _getRepeaterEntries(sessionEntry, __MCC_SESSION_NOTES);
        if (!noteEntries.isEmpty())
        {
            // Build common i18n parameters
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
            i18nParams.put("sessionPosition", new I18nizableText(Integer.toString(sessionEntryPosition)));
            
            // Get the educational path of the entry if it is not common and if the course educational path is empty
            Optional<EducationalPath> sessionEntryEducationalPath = courseEducationalPath
                .or(() ->
                    Optional.of(sessionEntry)
                        .filter(entry -> !Objects.requireNonNullElse((Boolean) _getValue(course, entry, prefixPath, __MCC_SESSION_COMMON), true)) // filter non common entries
                        .map(entry -> _getValue(course, entry, prefixPath, __MCC_SESSION_PATH))
                        .map(EducationalPath.class::cast)
                );
            
            // Process each note entry
            int entryPosition = 1;
            for (Map<String, Object> noteEntry : noteEntries)
            {
                String prefix = prefixPath + "/" + __MCC_SESSION_NOTES + "[" + entryPosition + "]";
                result.addResult(_checkNoteEntry(course, prefix, noteEntry, entryPosition, sessionEntryEducationalPath, currentEducationPaths, i18nParams));
                entryPosition++;
            }
        }
        
        return result;
    }
    
    private ValidationResult _checkNoteEntry(Course course, String prefixPath, Map<String, Object> noteEntry, int noteEntryPosition, Optional<EducationalPath> sessionEntryEducationalPath, List<EducationalPath> currentEducationPaths, Map<String, I18nizableTextParameter> commonI18nParams)
    {
        ValidationResult result = new ValidationResult();

        List<Map<String, Object>> skillEntries = _getRepeaterEntries(noteEntry, __MCC_SESSION_NOTES_SKILLS);
        if (!skillEntries.isEmpty())
        {
            // Map used to detect duplicated skill for each educational path in the note entry
            Map<EducationalPath, Set<ContentValue>> skillsByPath = new HashMap<>();
            
            // Process each skill entry
            int entryPosition = 1;
            for (Map<String, Object> skillEntry : skillEntries)
            {
                String prefix = prefixPath + "/" + __MCC_SESSION_NOTES_SKILLS + "[" + entryPosition + "]";
                ContentValue skill = (ContentValue) _getValue(course, skillEntry, prefix, __MCC_SESSION_NOTES_SKILLS_SKILL);
                // Get educational from the session entry if not null or from the skill entry
                EducationalPath educationalPath = sessionEntryEducationalPath.orElseGet(() -> (EducationalPath) _getValue(course, skillEntry, prefix, __MCC_SESSION_NOTES_SKILLS_PATH));
                
                // If one of this data is null, the mandatory validator should react
                if (skill != null && educationalPath != null)
                {
                    // Check only if there is no current educational path (global edition from form) or if the educational path of the skill is part of the educational paths beeing edited
                    if (currentEducationPaths.isEmpty() || currentEducationPaths.contains(educationalPath))
                    {
                        // Build common i18n parameters for errors/warnings
                        Map<String, I18nizableTextParameter> noteI18nParams = new HashMap<>(commonI18nParams);
                        noteI18nParams.put("notePosition", new I18nizableText(Integer.toString(noteEntryPosition)));
                        
                        // Get the skills already present for the current educational path
                        Set<ContentValue> skillsForPath = skillsByPath.computeIfAbsent(educationalPath, __ -> new HashSet<>());
                        
                        if (!skillsForPath.add(skill))
                        {
                            // The skill is already defined for this educational path in this note entry
                            result.addWarning(_buildDuplicationMessage(noteI18nParams, skill, educationalPath));
                        }
                        // Size of educational path can be zero if the course has no parent program
                        else if (educationalPath.getProgramItemIds().size() > 0)
                        {
                            // Check if the skill belong to the course acquired skills for the given program of the educational path
                            String programId = educationalPath.getProgramItemIds().get(0);
                            ContentValue[] acquiredSkills = course.getAcquiredSkills(programId);
                            if (!Arrays.asList(acquiredSkills).contains(skill))
                            {
                                // Skill is not in the course acquired skills, add a warning
                                result.addWarning(_buildInconsistentMessage(noteI18nParams, entryPosition, skill, course));
                            }
                        }
                    }
                }
                entryPosition++;
            }
        }
        
        return result;
    }
    
    private List<EducationalPath> _getCurrentEducationPaths(Course course)
    {
        // Get the educational paths if this course concerned by this edition
        // If empty, all educational paths are concerned
        Request request = ContextHelper.getRequest(_context);
        
        UserIdentity user = _currentUserProvider.getUser();
        boolean hasRightToEdit = _rightManager.hasRight(user, "ODF_Rights_Course_Edit", course) == RightResult.RIGHT_ALLOW;
        
        @SuppressWarnings("unchecked")
        List<EducationalPath> educationalPaths = (List<EducationalPath>) request.getAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS);
        return educationalPaths == null 
                    ? List.of() 
                    : educationalPaths.stream()
                        .filter(path -> hasRightToEdit || _hasRightToEditOnPath(course, path, user))
                        .toList();
    }
    
    private boolean _hasRightToEditOnPath(Course course, EducationalPath path, UserIdentity user)
    {
        return _rightManager.hasRight(user, "ODF_Rights_Course_Edit", new ContextualizedContent(course, path)) == RightResult.RIGHT_ALLOW;
    }
    
    private Optional<EducationalPath> _getUniqueEducationalPath(Course course)
    {
        List<EducationalPath> educationalPaths = _odfHelper.getEducationalPaths(course, true, true);
        return educationalPaths.size() == 1 ? Optional.of(educationalPaths.get(0)) : Optional.empty();
    }
    
    @SuppressWarnings("unchecked")
    private List<Map<String, Object>> _getRepeaterEntries(Map<String, Object> syncValues, String attributeName)
    {
        Object value = syncValues.get(attributeName);
        return value == null || value instanceof UntouchedValue 
                ? List.of() // If value is null or untouched, return an empty list to not check the repeater
                : value instanceof List ? (List<Map<String, Object>>) value : ((SynchronizableRepeater) value).getEntries();
    }
    
    private Object _getValue(Course course, Map<String, Object> syncValues, String prefixPath, String attributeName)
    {
        // Get attribute value from the synchronizable values map
        Object value = syncValues.get(attributeName);
        switch (value)
        {
            case UntouchedValue untouchedValue:
                // The value is untouched so get the value in the course
                return course.getValue(prefixPath + "/" + attributeName);
            case SynchronizableValue syncValue:
                return syncValue.getValue(Optional.ofNullable(syncValue.getExternalizableStatus()));
            default:
                return value;
        }
    }
    
    private I18nizableText _buildDuplicationMessage(Map<String, I18nizableTextParameter> commonI18nParams, ContentValue skill, EducationalPath educationalPath)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
        i18nParams.put("skill", new I18nizableText(skill.getContentIfExists().map(Content::getTitle).orElseGet(() -> skill.getContentId())));
        i18nParams.put("educationalPath", new I18nizableText(_odfHelper.getEducationalPathAsString(educationalPath)));
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCSKILLS_VALIDATOR_ERROR_DUPLICATED", i18nParams);
    }
    
    private I18nizableText _buildInconsistentMessage(Map<String, I18nizableTextParameter> commonI18nParams, int skillPosition, ContentValue skill, Course course)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
        i18nParams.put("skillPosition", new I18nizableText(Integer.toString(skillPosition)));
        i18nParams.put("skill", new I18nizableText(skill.getContentIfExists().map(Content::getTitle).orElseGet(() -> skill.getContentId())));
        i18nParams.put("course", new I18nizableText(course.getTitle()));
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCSKILLS_VALIDATOR_ERROR_INCONSISTENT", i18nParams);
    }
}
