/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.disableconditions.DefaultDisableConditionsEvaluator;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;

/**
 * Class to generate a report based on MCC.
 */
public abstract class AbstractMCCReport extends AbstractReport implements Initializable
{
    /** The key to define if we want a result by program or year */
    public static final String PARAMETER_ONE_FILE_BY_PROGRAM = "oneFileByProgram";
    
    /** The key to define if we want only year belonging to the current orgUnit */
    public static final String PARAMETER_FILTER_ON_CURRENT_ORGUNIT = "filterOnCurrentOrgUnit";
    
    /** The key to define the orgUnit to to get only year belonging to it */
    public static final String PARAMETER_FILTER_BY_ORGUNIT = "filterByOrgUnit";
    
    /** Prefix of the sessions name */
    private static final String __SESSION_NAME_PREFIX = "mccSession";
    
    /** Name of the first session */
    protected static final String FIRST_SESSION_NAME = __SESSION_NAME_PREFIX + "1";
    
    /** Name of the second session */
    protected static final String SECOND_SESSION_NAME = __SESSION_NAME_PREFIX + "2";
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** The disable conditions evaluator */
    protected DisableConditionsEvaluator _disableConditionsEvaluator;
    
    /** The content type EP */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The MCC sessions view */
    protected View _sessionsView;
    
    private View _programView;
    private View _subProgramView;
    private View _containerView;
    private View _courseListView;
    private View _courseView;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _disableConditionsEvaluator = (DisableConditionsEvaluator) manager.lookup(DefaultDisableConditionsEvaluator.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _programView = getView(ProgramFactory.PROGRAM_CONTENT_TYPE, "report-mcc");
        _subProgramView = getView(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, "report-mcc");
        _containerView = getView(ContainerFactory.CONTAINER_CONTENT_TYPE, "report-mcc");
        _courseListView = getView(CourseListFactory.COURSE_LIST_CONTENT_TYPE, "report-mcc");
        _courseView = getView(CourseFactory.COURSE_CONTENT_TYPE, "report-mcc");
        _sessionsView = getView(CourseFactory.COURSE_CONTENT_TYPE, "report-mcc-sessions");
    }
    
    @Override
    protected boolean isGeneric()
    {
        return false;
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_PDF;
    }
    
    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_DOC, OUTPUT_FORMAT_XLS, OUTPUT_FORMAT_PDF);
    }
    
    /**
     * Get the view from the content type.
     * @param contentTypeId The content type id
     * @param viewName The view name
     * @return The view
     */
    protected View getView(String contentTypeId, String viewName)
    {
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        return contentType.getView(viewName);
    }
    
    @Override
    protected PilotageReportContent getReportContentForOrgUnit(String outputFormat, Map<String, String> reportParameters)
    {
        return _getReportContentForProgramItemsInOrgUnit(outputFormat, reportParameters);
    }
    
    @Override
    protected Stream<? extends ProgramItem> _getProgramItemsFromProgram(Program program, Map<String, String> reportParameters)
    {
        if (!Boolean.parseBoolean(reportParameters.get(PARAMETER_ONE_FILE_BY_PROGRAM)))
        {
            Set<Container> years = _odfHelper.getYears(program);
            // Test if we want to filter on current orgUnit
            boolean filterOnOrgunit = Boolean.parseBoolean(reportParameters.get(PARAMETER_FILTER_ON_CURRENT_ORGUNIT));
            if (filterOnOrgunit)
            {
                // If yes, get the current orgunit in parameters and filter on it if exists
                String currentOrgUnitId = reportParameters.get(PARAMETER_ORGUNIT);
                return StringUtils.isNotBlank(currentOrgUnitId)
                        ? years.stream().filter(c -> this._isFromOrgUnit(c, currentOrgUnitId))
                                : years.stream();
            }
            else
            {
                // If not, test if we have selected an orgunit to filter on it
                String selectedOrgUnitId = reportParameters.get(PARAMETER_FILTER_BY_ORGUNIT);
                return StringUtils.isNotBlank(selectedOrgUnitId)
                        ? years.stream().filter(c -> this._isFromOrgUnit(c, selectedOrgUnitId))
                                : years.stream();
            }
        }
        return super._getProgramItemsFromProgram(program, reportParameters);
    }
    
    private boolean _isFromOrgUnit(Content programItem, String orgUnitId)
    {
        return ContentDataHelper.getContentIdsListFromMultipleContentData(programItem, "parentOrgunits").contains(orgUnitId);
    }
    
    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        // The MCC (diff too) report can accept every program item
        ProgramItem programItem = _resolver.resolveById(programItemId);
        
        ProgramItemTree programTree = createMCCTreeFromProgramItem(programItem, reportParameters);
        
        try
        {
            handler.startDocument();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType(reportParameters, true));
            XMLUtils.startElement(handler, "report", attrs);

            // SAX other global informations
            saxGlobalInformations(handler, programItem, reportParameters);
            
            // SAX other informations
            saxOtherInformations(handler, programItem, reportParameters);
            
            // SAX tree
            saxTree(handler, programTree);
            
            XMLUtils.endElement(handler, "report");
            
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'MCC' report for program item '{}' ({})", ((Content) programItem).getTitle(), programItem.getCode(), e);
        }
    }

    @Override
    protected void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        throw new UnsupportedOperationException("Impossible to launch the report '" + getType(reportParameters, true) + "' on a orgunit.");
    }
    
    /**
     * SAX the MCC tree for the report
     * @param handler The handler
     * @param tree The MCC tree
     * @throws SAXException if an error occurs
     */
    protected void saxTree(ContentHandler handler, ProgramItemTree tree) throws SAXException
    {
        ProgramItem programItem = tree.getCurrent();
        
        if (programItem instanceof Program program)
        {
            saxContent(handler, "program", program, _programView, tree);
        }
        else if (programItem instanceof SubProgram subProgram)
        {
            saxContent(handler, "subProgram", subProgram, _subProgramView, tree);
        }
        else if (programItem instanceof Container container)
        {
            saxContent(handler, "container", container, _containerView, tree);
        }
        else if (programItem instanceof CourseList courseList)
        {
            saxContent(handler, "courseList", courseList, _courseListView, tree);
        }
        else if (programItem instanceof Course course)
        {
            if (!isExcludedFromMCC(tree))
            {
                saxContent(handler, "course", course, _courseView, tree);
            }
        }
    }
    
    /**
     * Check if the current program item is excluded from MCC
     * @param tree the current program item
     * @return <code>true</code> if the current program item is excluded from MCC
     */
    protected abstract boolean isExcludedFromMCC(ProgramItemTree tree);
    
    /**
     * Sax a content on a view to the handler and sax children.
     * @param handler The handler
     * @param tagName The root tag name
     * @param content The content
     * @param view The view
     * @param tree The tree for children
     * @throws SAXException if an error occurs
     */
    protected void saxContent(ContentHandler handler, String tagName, Content content, View view, ProgramItemTree tree) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("path", tree.getPath().toString());
        if (_pilotageHelper.canWriteMccRestrictions(content, "mccSession2"))
        {
            attrs.addCDATAAttribute("session2", "available");
        }
        XMLUtils.startElement(handler, tagName, attrs);
        
        // Sax content data
        content.dataToSAX(handler, view);
        
        // Sax MCC for courses
        if (content instanceof Course)
        {
            saxMCCs(handler, tree);
        }
        
        // Sax children
        for (ProgramItemTree child : tree.getChildren())
        {
            saxTree(handler, child);
        }
        
        XMLUtils.endElement(handler, tagName);
    }
    
    /**
     * Sax the MCC sessions.
     * @param handler The handler
     * @param tree The MCC tree
     * @throws SAXException If an error occurs
     */
    protected void saxMCCs(ContentHandler handler, ProgramItemTree tree) throws SAXException
    {
        XMLUtils.startElement(handler, "mcc");
        
        // Generate SAX events for MCC sessions
        saxSession(handler, tree, FIRST_SESSION_NAME);
        saxSession(handler, tree, SECOND_SESSION_NAME);
        
        XMLUtils.endElement(handler, "mcc");
    }
    
    /**
     * Sax the MCC session (1 or 2).
     * @param handler The handler
     * @param tree The MCC tree
     * @param sessionName The session name
     * @throws SAXException If an error occurs
     */
    protected abstract void saxSession(ContentHandler handler, ProgramItemTree tree, String sessionName) throws SAXException;
    
    /**
     * Generates SAX events for the details of a MCC session entry.
     * @param handler The handler
     * @param sessionEntry The session repeater entry to SAX
     * @param rootContent The root content of the session
     * @throws SAXException If an error occurs
     */
    protected void saxSessionEntryDetails(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent) throws SAXException
    {
        String repeaterName = sessionEntry.getHoldingRepeater().getModel().getName();
        sessionEntry.dataToSAX(
            handler,
            (ViewItemAccessor) _sessionsView.getModelViewItem(repeaterName),
            RepositoryDataContext.newInstance()
                .withObject(rootContent)
                .withDataPath(repeaterName + "[" + sessionEntry.getPosition() + "]")
        );
    }

    /**
     * Add and populate the program item to the {@link ProgramItemTree}
     * @param programItem The program item to add
     * @param reportParameters The report parameters
     * @return a tree of the program
     */
    protected ProgramItemTree createMCCTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        ProgramItemTree programItemTree = new ProgramItemTree(programItem);
        populateMCCTree(programItemTree);
        return programItemTree;
    }
    
    /**
     * Populate the MCC tree.
     * @param tree The MCC tree
     */
    protected abstract void populateMCCTree(ProgramItemTree tree);
    
    /**
     * Sax the additional global informations of the report.
     * @param handler The handler
     * @param programItem The program item on which the report is launched
     * @param reportParameters The report parameters
     * @throws SAXException If an error occurs
     */
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        String catalog = programItem.getCatalog();
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", catalog);
        XMLUtils.createElement(handler, "catalog", attrs, _catalogsManager.getCatalog(catalog).getTitle());
        XMLUtils.createElement(handler, "title", ((Content) programItem).getTitle());
        XMLUtils.createElement(handler, "lang", programItem.getLanguage());
        for (EducationalPath path : _odfHelper.getEducationalPaths(programItem, false, true))
        {
            String readablePath = path.resolveProgramItems(_resolver).map(Content.class::cast).map(Content::getTitle).collect(Collectors.joining(" > "));
            XMLUtils.createElement(handler, "path", readablePath);
        }
    }
    
    /**
     * Sax various informations.
     * @param handler The handle
     * @param programItem The program item on which the report is laucnhed
     * @param reportParameters The report parameters
     * @throws SAXException if an error occurs
     */
    protected void saxOtherInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        _reportHelper.saxNaturesEnseignement(handler, getLogger());
    }
}
