  /*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

const OdfHelper = Ametys.serviceManager.lookup("org.ametys.odf.ODFHelper");
const PilotageHelper = Ametys.serviceManager.lookup("org.ametys.plugins.odfpilotage.helper.PilotageHelper");

logger.info("Move MCC pilotage status attributes on programs to mcc_workflow composite on years");

let programs = Repository.query("//element(*, ametys:programContent)");
let total = programs.getSize();
let updated = 0;
let handled = 0;
let hasChanges = false;
let updatedPrograms = [];

programs.forEach(
    program =>
    {
        hasChanges = false;
        
        if (program.hasValue("pilotage"))
	    {
	        var pilotageNode = program.getComposite("pilotage").getRepositoryData().getNode();
	        var pilotageStatus = pilotageNode.getProperty("ametys:pilotage_status").getString();
	        
	        if (pilotageStatus == "CFVU_MCC_VALIDATED")
	        {
	            var mccValidationDate = null,
                mccValidationComment = null,
                mccValidationAuthorLogin = null,
                mccValidationAuthorPopulation = null;
            
	            if (pilotageNode.hasProperty("ametys:cfvu_mcc_validation_date"))
	            {
	                mccValidationDate = pilotageNode.getProperty("ametys:cfvu_mcc_validation_date").getDate();
	            }
	            if (pilotageNode.hasProperty("ametys:cfvu_mcc_validation_comment"))
	            {
	                mccValidationComment = pilotageNode.getProperty("ametys:cfvu_mcc_validation_comment").getString();
	            }
	            if (pilotageNode.hasNode("ametys:cfvu_mcc_validation_author"))
	            {
	                var mccValidationAuthorNode = pilotageNode.getNode("ametys:cfvu_mcc_validation_author");
	                mccValidationAuthorLogin = mccValidationAuthorNode.getProperty("ametys:login").getString();
	                mccValidationAuthorPopulation = mccValidationAuthorNode.getProperty("ametys:population").getString();
	            }
                
                let years = Object.values(_getYears(program));
                years.forEach(
		            year =>
		            {
		                Content.migrate( 
		                    year, 
		                    [_migrateMCCStatusOnYear.bind(null, mccValidationDate, mccValidationComment, mccValidationAuthorLogin, mccValidationAuthorPopulation, year)], 
		                    true, /* old version still comptatible */ 
		                    null, /* no tag */ 
		                    false, /* verbose */
		                    true /* synchronize live */
		                );
		                
		            }
		        );
                
                if (hasChanges)
                {
                    updated++;
                }
	        }
            
            updatedPrograms.push(program);
        }
	        
        handled++;
        
        if (handled % 200 == 0)
        {
            logger.info(`${handled}/${total} handled programs`);
        }
    }
);

logger.info(`MCC status has been moved on years for ${updated} programs on a total of ${total} programs.`);

total = updatedPrograms.length;
handled = 0;
updated = 0;

updatedPrograms.forEach(
    program =>
    {
        hasChanges = false;
        
        Content.migrate( 
            program, 
            [_migrateMccStatus], 
            true, /* old version still comptatible */ 
            null, /* no tag */ 
            false, /* verbose */
            true /* synchronize live */
        );
        
        if (hasChanges)
        {
            updated++;
        }
                
        handled++;
        
        if (handled % 200 == 0)
        {
            logger.info(`${handled} handled programs`);
        }
    }
);

logger.info(`Pilotage status has been updated for ${updated} programs on a total of ${total} programs.`);

/* Functions */

function _migrateMccStatus(content)
{
    var node = content.getNode();
    if (content.hasValue("pilotage"))
    {
        var pilotageNode = content.getComposite("pilotage").getRepositoryData().getNode();
      
        if (pilotageNode.hasProperty("ametys:cfvu_mcc_validation_date"))
        {
            var mccValidationDateProperty = pilotageNode.getProperty("ametys:cfvu_mcc_validation_date").remove();
            hasChanges = true;
        }
        
        if (pilotageNode.hasProperty("ametys:cfvu_mcc_validation_comment"))
        {
            var mccValidationCommentProperty = pilotageNode.getProperty("ametys:cfvu_mcc_validation_comment").remove();
            hasChanges = true;
        }
        
        if (pilotageNode.hasNode("ametys:cfvu_mcc_validation_author"))
        {
            pilotageNode.getNode("ametys:cfvu_mcc_validation_author").remove();
            hasChanges = true;
        }
        
        // Update status of program
        var pilotageStatus = pilotageNode.getProperty("ametys:pilotage_status").getString();
        if (pilotageStatus == "CFVU_MCC_VALIDATED")
        {
            pilotageNode.setProperty("ametys:pilotage_status", "CFVU_VALIDATED");
            hasChanges = true;
        }
    }
}

function _migrateMCCStatusOnYear(date, comment, authorLogin, authorPopulation, year)
{
    var mccComposite = year.getComposite("mcc_workflow", true);
    mccComposite.setValue("cfvu_mcc_validation_status", true);
    mccComposite.setValue("mcc_orgunit_validation_status", true);
    mccComposite.setValue("rules_validation_status", true);
    mccComposite.setValue("mcc_validation_status", true);
    
    if (date != null)
    {
        let mccNode = mccComposite.getRepositoryData().getNode();
        mccNode.setProperty("ametys:rules_validation_date", date);
        mccNode.setProperty("ametys:mcc_validation_date", date);
        mccNode.setProperty("ametys:mcc_orgunit_validation_date", date);
        mccNode.setProperty("ametys:cfvu_mcc_validation_date", date);
    }
    
    mccComposite.setValue("rules_validation_comment", comment || "");
    mccComposite.setValue("mcc_validation_comment", comment || "");
    mccComposite.setValue("mcc_orgunit_validation_comment", comment || "");
    mccComposite.setValue("cfvu_mcc_validation_comment", comment || "");
    
    if (authorLogin != null)
    {
        mccComposite.setValue("rules_validation_author", new org.ametys.core.user.UserIdentity(authorLogin, authorPopulation));
        mccComposite.setValue("mcc_validation_author", new org.ametys.core.user.UserIdentity(authorLogin, authorPopulation));
        mccComposite.setValue("mcc_orgunit_validation_author", new org.ametys.core.user.UserIdentity(authorLogin, authorPopulation));
        mccComposite.setValue("cfvu_mcc_validation_author", new org.ametys.core.user.UserIdentity(authorLogin, authorPopulation));
    }
    
    hasChanges = true;
}

function _getYears(content)
{
    var years = {};
    var childProgramItems = OdfHelper.getChildProgramItems(content);
    
    for (var i in childProgramItems)
    {
        var child = childProgramItems[i];
        if (child instanceof org.ametys.odf.program.Container && PilotageHelper.isContainerOfTypeYear(child))
        {
            years[child.getId()] = child;
        }
        
        Object.assign(years, _getYears(child));
    }
  
    return years;
}