/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.AbstractProgramPart;
import org.ametys.odf.program.Container;
import org.ametys.runtime.config.Config;

/**
 * Client side element for MCC workflow buttons on a {@link OrgUnit}
 */
public class OrgunitMCCWorkflowClientSideElement extends MCCWorkflowClientSideElement
{
    /**
     * Validate the MCC to orgunit level for all years of a orgunit
     * @param ouId the id of root program
     * @param catalogName The catalogue name
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @return the result
     */
    @Callable (rights = "ODF_Pilotage_Global_MCC_Orgunit_Validated_Rights", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> validateOrgunitMCC(String ouId, String catalogName, String validationDate, String comment)
    {
        List<String> containerIds = _getYears(ouId, catalogName);
                
        // Do not check rights on years
        return validateOrgunitMCC(containerIds, null, validationDate, comment, Map.of());
    }
    
    /**
     * Validate the MCC to CFVU level for all years of a orgunit
     * @param ouId the id of root program
     * @param catalogName The catalogue name
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = "ODF_Pilotage_Global_CFVU_MCC_Validated_Rights", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> validateMCCForCVFU(String ouId, String catalogName, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        List<String> containerIds = _getYears(ouId, catalogName);
        
        // Do not check rights on years
        return validateMCCForCFVU(containerIds, null, validationDate, comment, contextualParameters);
    }
    
    private List<String> _getYears(String ouId, String catalogName)
    {
        OrgUnit ou = _resolver.resolveById(ouId);
        return _odfHelper.getProgramsFromOrgUnit(ou, catalogName, Config.getInstance().getValue("odf.programs.lang"), false)
            .stream()
            .map(_odfHelper::getYears)
            .flatMap(Collection::stream)
            .filter(y -> _checkOrgUnit(y, ouId)) // check year belongs to orgunit
            .map(Container::getId)
            .toList();
    }
    
    private boolean _checkOrgUnit(AbstractProgramPart programPart, String ouId)
    {
        if (programPart.getOrgUnits().contains(ouId))
        {
            return true;
        }
        
        List<ProgramItem> parentProgramItems = _odfHelper.getParentProgramItems(programPart);
        for (ProgramItem parentProgramItem : parentProgramItems)
        {
            if (parentProgramItem instanceof AbstractProgramPart parentProgramPart && _checkOrgUnit(parentProgramPart, ouId))
            {
                return true;
            }
        }
        
        return false;
    }
}
