/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.cms.search.cocoon.ContentGridComponent;
import org.ametys.cms.search.content.ContentValuesExtractorFactory;
import org.ametys.cms.search.content.ContentValuesExtractorFactory.SearchModelContentValuesExtractor;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.cms.search.ui.model.SearchUIModelExtensionPoint;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.ui.Callable;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.rights.ODFRightHelper;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.odf.workflow.EditContextualizedDataFunction;
import org.ametys.runtime.model.ViewItemAccessor;

/**
 * The tree helper for grid displaying a view
 */
public class ContentsWithViewTreeGridHelper extends ODFContentsTreeHelper implements Contextualizable, Configurable
{
    /** The id of the request attribute where required content views are stored */
    protected static final String REQUEST_ATTRIBUTE_SEARCHMODELNAME = ContentsWithViewTreeGridHelper.class.getName() + "$ContentViews";
    
    /** The avalon context */
    protected Context _context;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;

    /** The workflow action that will be used for edition (to check if available) */
    protected int _workflowEditActionId;
    
    private SearchUIModelExtensionPoint _searchUIModelExtensionPoint;
    private ContentValuesExtractorFactory _contentValuesExtractorFactory;
    private ContentGridComponent _contentGridComponent;


    public void configure(Configuration configuration) throws ConfigurationException
    {
        _workflowEditActionId = configuration.getChild("workflowEditActionId").getValueAsInteger(2);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _searchUIModelExtensionPoint = (SearchUIModelExtensionPoint) smanager.lookup(SearchUIModelExtensionPoint.ROLE);
        _contentValuesExtractorFactory = (ContentValuesExtractorFactory) smanager.lookup(ContentValuesExtractorFactory.ROLE);
        _contentGridComponent = (ContentGridComponent) smanager.lookup(ContentGridComponent.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }

    /**
     * Determines if current user can edit repeater with educational path as a consummer of content.
     * @param contentId the id of content
     * @param path the path of content in the current tree
     * @return true if user is allowed to edit data for given path
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public boolean canEditRepeaterWithPath(String contentId, List<String> path)
    {
        WorkflowAwareContent content = _ametysResolver.resolveById(contentId);
        
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS, List.of(EducationalPath.of(path.toArray(String[]::new))));
        
        return _contentWorkflowHelper.isAvailableAction(content, EditContextualizedDataFunction.EDIT_WORKFLOW_ACTION_ID);
    }
    
    /**
     * Get the root node informations
     * @param contentId The content
     * @param treeId the tree configuration
     * @param searchModel The search model involved
     * @return The informations
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getRootNodeInformations(String contentId, String treeId, String searchModel)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME, searchModel);
        
        return super.getRootNodeInformations(contentId, treeId);
    }
    
    /**
     * Get the node informations
     * @param contentId The content
     * @param path The path to the content in the current tree
     * @param searchModel The search model involved
     * @return The informations
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getNodeInformations(String contentId, List<String> path, String searchModel)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME, searchModel);
        
        return getNodeInformations(contentId, path);
    }
    
    /**
     * Get the children contents according the tree configuration
     * @param contentId the parent content
     * @param path the path to the content in the tree
     * @param treeId the tree configuration
     * @param searchModelName The name of the search model to use to send data
     * @return the children content
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> getChildrenContent(String contentId, List<String> path, String treeId, String searchModelName)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME, searchModelName);
        
        return super.getChildrenContent(contentId, path, treeId);
    }
    
    @Override
    protected Map<String, Object> content2Json(Content content, List<String> path)
    {
        Map<String, Object> content2Json = super.content2Json(content, path);
        
        Request request = ContextHelper.getRequest(_context);
        String searchModelName = (String) request.getAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME);
        
        if (searchModelName != null)
        {
            SearchUIModel searchUIModel = _searchUIModelExtensionPoint.getExtension(searchModelName);
            
            Set<String> contentTypesOfModel = searchUIModel.getContentTypes(Map.of());
            
            // Let's see if a view is provided for one of the content types of the content
            boolean jsonContentData = contentTypesOfModel.stream().anyMatch(cType -> _contentTypesHelper.isInstanceOf(content, cType));
            
            if (jsonContentData)
            {
                fillContentData(content, searchUIModel, path, content2Json);
            }
            else
            {
                // Non supported contents are not editable in the tree grid
                content2Json.put("notEditableData", true);
            }
        }
        else
        {
            // Rootnode contents are not editable in the tree grid (since we do not transfer the search model... yet)
            content2Json.put("notEditableData", true);
        }
        
        return content2Json;
    }
    
    /**
     * Fill JSON data for content of the tree
     * @param content the content
     * @param searchUIModel the search ui model
     * @param path The path of the content in the tree
     * @param content2Json content as JSON
     */
    protected void fillContentData(Content content, SearchUIModel searchUIModel, List<String> path, Map<String, Object> content2Json)
    {
        content2Json.put("data", getContentData(content, searchUIModel));
        
        if (_contentWorkflowHelper.isAvailableAction((WorkflowAwareContent) content, _workflowEditActionId))
        {
            content2Json.put("notEditableDataIndex", canEdit(searchUIModel.getResultItems(Map.of()), content));
        }
        else
        {
            // workflow does not allow to edit content
            content2Json.put("notEditableData", true);
        }
    }
    
    /**
     * Get content data
     * @param content the content
     * @param searchUIModel the search UI model
     * @return the content data
     */
    @SuppressWarnings("unchecked")
    protected Map<String, Object> getContentData(Content content, SearchUIModel searchUIModel)
    {
        // This method returns the repeaters recursively while a simple content.dataToJSON would not
        SearchModelContentValuesExtractor valuesExtractor = _contentValuesExtractorFactory.create(searchUIModel);
        return (Map<String, Object>) _contentGridComponent.getContentData(content, valuesExtractor, null, Map.of()).get("properties");
    }
    
    /**
     * List the path that cannot be edited. Not following the content attribute children (since it could imply some items modifiable and others not in a repeater)
     * @param viewItemAcessor The view with definition
     * @param content The content to study
     * @return The list of unmodifiable definition path
     */
    @SuppressWarnings("unchecked")
    protected List<String> canEdit(ViewItemAccessor viewItemAcessor, Content content)
    {
        List<String> results = new ArrayList<>();
        
        org.ametys.plugins.repository.model.ViewHelper.visitView(viewItemAcessor,
            (element, definition) -> {
                // attribute
                if (definition instanceof RestrictedModelItem restrictedModelItem && !restrictedModelItem.canWrite(content))
                {
                    results.add(definition.getPath());
                }
                
                // content attribute children not handled
            },
            (group, definition) -> {
                // composite
                results.addAll(canEdit(group, content));
            },
            (group, definition) -> {
                // repeater
                if (definition instanceof RestrictedModelItem restrictedModelItem && !restrictedModelItem.canWrite(content))
                {
                    results.add(definition.getPath());
                }
                else
                {
                    results.addAll(canEdit(group, content));
                }
            },
            group -> {
                // group
                results.addAll(canEdit(group, content));
            }
        );
        
        return results;
    }
}
