/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.IndexableRepeater;
import org.ametys.cms.data.holder.group.IndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.InvalidInputWorkflowException;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Program;
import org.ametys.odf.rights.ODFRightHelper;
import org.ametys.odf.workflow.EditContextualizedDataFunction;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.disableconditions.AbstractRelativeDisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.parameter.ValidationResults;

import com.opensymphony.workflow.WorkflowException;

/**
 * This component handle the content of the micro skill weight tool
 */
public class MicroSkillsWeightTreeHelper extends MicroSkillsTreeHelper
{
    /** The name of the mcc session 1 data */
    protected static final String MCC_SESSION_1_ATTRIBUTE_NAME = "mccSession1";
    /** The name of the mcc session 2 data */
    protected static final String MCC_SESSION_2_ATTRIBUTE_NAME = "mccSession2";
    /** Helper for ODF contents */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    protected Map<String, Object> content2Json(Content content, List<String> path)
    {
        Map<String, Object> json = super.content2Json(content, path);
        
        if (json != null && content instanceof Course course)
        {
            Program program = _resolver.resolveById(path.get(0));
            
            // Send "model"
            json.putAll(_sessionToJSON(course.getRepeater("mccSession1"), null));
            json.putAll(_sessionToJSON(course.getRepeater("mccSession2"), null));
            
            // Make non editable the skills that were not chosen
            json.computeIfAbsent("notEditableDataIndex", l -> new ArrayList<>());
            @SuppressWarnings("unchecked")
            List<String> notEditableDataIndex = (List<String>) json.get("notEditableDataIndex");

            
            ContentValue[] acquiredSkills = course.getAcquiredSkills(program.getId());
            List<String> acquiredSkillsIds = Arrays.stream(acquiredSkills).map(ContentValue::getContentId).toList();
            
            List<ContentValue> macroSkills = getMacroskills(program);
            for (ContentValue macroSkillContentValue : macroSkills)
            {
                Optional<ModifiableContent> optionalContent = macroSkillContentValue.getContentIfExists();
                if (optionalContent.isPresent())
                {
                    Content macroSkill = optionalContent.get();
                    
                    ContentValue[] microSkills = macroSkill.getValue("microSkills");
                    if (microSkills != null)
                    {
                        for (ContentValue microSkill : microSkills)
                        {
                            if (!acquiredSkillsIds.contains(microSkill.getContentId()))
                            {
                                notEditableDataIndex.add(microSkill.getContentId());
                            }
                        }
                    }
                }
            }
        }
        
        return json;
    }
    
    @Override
    protected Object _skill2Json(Course course, String skillId)
    {
        Map<String, Object> json = new LinkedHashMap<>();
        json.putAll(_sessionToJSON(course.getRepeater("mccSession1"), skillId));
        json.putAll(_sessionToJSON(course.getRepeater("mccSession2"), skillId));
        return json;
    }

    private Map< ? extends String, ? extends Object> _sessionToJSON(IndexableRepeater session, String skillId)
    {
        Map<String, Object> json = new LinkedHashMap<>();
        
        if (session != null && session.getSize() > 0)
        {
            Map<String, Object> evalsJson = new LinkedHashMap<>();

            for (IndexableRepeaterEntry evaluations : session.getEntries())
            {
                IndexableRepeater notes = evaluations.getRepeater("notes");
                
                boolean isCommon = !evaluations.hasDefinition("common") || evaluations.getValue("common") != Boolean.FALSE;

                List<Object> notesJson = new ArrayList<>();
                    
                if (notes != null && notes.getSize() > 0)
                {
                    for (IndexableRepeaterEntry note : notes.getEntries())
                    {
                        notesJson.add(_noteToJSON(isCommon, note, skillId));
                    }
                }
                else
                {
                    // Simulate one empty note
                    notesJson.add(_noteToJSON(isCommon, null, skillId));
                }
                
                Map<String, Object> evalJson = new LinkedHashMap<>();
                if (StringUtils.isBlank(skillId))
                {
                    evalJson.put("label", StringUtils.defaultIfBlank(evaluations.getValue("label"), ""));
                }
                
                evalJson.put("notes", notesJson);
                evalJson.put("common", isCommon);
                evalJson.put("path", _toString((EducationalPath) evaluations.getValue("path")));
                
                evalsJson.put(Integer.toString(evaluations.getPosition()), evalJson);
            }
            
            if (evalsJson.size() > 0)
            {
                Map<String, Object> sessionJson = new LinkedHashMap<>();
                if (StringUtils.isBlank(skillId))
                {
                    sessionJson.put("label", session.getModel().getLabel());
                }
                sessionJson.put("data", evalsJson);
                
                return Map.of(
                    session.getModel().getName(), sessionJson
                );
            }
        }

        return json;
    }

    private Map<String, ? extends Object> _noteToJSON(boolean commonParentEvaluation, IndexableRepeaterEntry note, String skillId)
    {
        if (StringUtils.isBlank(skillId))
        {
            return Map.of (
                "label", StringUtils.defaultIfBlank(note != null ? note.getValue("label") : null, ""),
                "ponderation", note != null ? (Double) note.getValue("ponderation", true, 100.0) : 100.0
            );
        }
        else
        {
            Map<String, Double> skillPonderations = new HashMap<>();
            
            IndexableRepeater skills = note != null ? note.getRepeater("skills") : null;
            if (skills != null && skills.getSize() > 0)
            {
                for (IndexableRepeaterEntry skill : skills.getEntries())
                {
                    if (skillId.equals(((ContentValue) skill.getValue("skill")).getContentId()))
                    {
                        EducationalPath p = skill.getValue("path");
                        skillPonderations.put(commonParentEvaluation ? _toString(p) : "", skill.getValue("ponderation"));
                    }
                }
            }
            
            return skillPonderations;
        }
    }

    private String _toString(EducationalPath value)
    {
        return value != null ? value.toString() : "";
    }
    
    /**
     * Save skills changes on course's MCC
     * @param courseId The id of the course to edit
     * @param changes The changes made from grid
     * @param programRootId The id of the root program
     * @return The result map with 'success' key to true or false.
     */
    @Override
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> saveEdition(String courseId, Map<String, Object> changes, String programRootId)
    {
        Content content = _resolver.resolveById(courseId);
        if (content instanceof Course course)
        {
            Map<String, Object> values = new HashMap<>();
            
            RepeaterDefinition mccSession1Def = (RepeaterDefinition) content.getDefinition(MCC_SESSION_1_ATTRIBUTE_NAME);
            List<Map<String, Object>> mccSession1Values = _getRepeaterEntries(content, mccSession1Def);
            if (mccSession1Values != null && !mccSession1Values.isEmpty())
            {
                _applyChanges(mccSession1Values, MCC_SESSION_1_ATTRIBUTE_NAME, changes);
                values.put(MCC_SESSION_1_ATTRIBUTE_NAME, mccSession1Values);
            }
            
            
            RepeaterDefinition mccSession2Def = (RepeaterDefinition) content.getDefinition(MCC_SESSION_2_ATTRIBUTE_NAME);
            List<Map<String, Object>> mccSession2Values = _getRepeaterEntries(content, mccSession2Def);
            if (mccSession2Values != null && !mccSession2Values.isEmpty())
            {
                _applyChanges(mccSession2Values, MCC_SESSION_2_ATTRIBUTE_NAME, changes);
                values.put(MCC_SESSION_2_ATTRIBUTE_NAME, mccSession2Values);
            }
            
            try
            {
                List<EducationalPath> educationPaths = _odfHelper.getEducationalPaths(course, true, true).stream()
                        .filter(p -> p.getProgramItemIds().contains(programRootId))
                        .toList();
                
                Request request = ContextHelper.getRequest(_context);
                request.setAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS, educationPaths);

                if (_contentWorkflowHelper.isAvailableAction(course, EDIT_ACTION_ID))
                {
                    _contentWorkflowHelper.editContent(course, values, EDIT_ACTION_ID);
                }
                else
                {
                    // Fallback to action 20
                    _contentWorkflowHelper.editContent(course, values, EditContextualizedDataFunction.EDIT_WORKFLOW_ACTION_ID);
                }
                
                return Map.of("success", true);
            }
            catch (WorkflowException e)
            {
                Map<String, Object> result = new HashMap<>();
                result.put("success", false);
                result.put("errorMsg", "An error occured while trying to save the changes of the course \"" + course.getTitle() + "\" (" + course.getId() + ") : " + e.getMessage());
                if (e instanceof InvalidInputWorkflowException iiwe)
                {
                    ValidationResults results = iiwe.getValidationResults();
                    result.put("errors", results.getAllErrors());
                }
                
                return result;
            }
        }
        else
        {
            throw new IllegalArgumentException("The content with id \"" + courseId + "\" is not a course or does not exist.");
        }
    }

    //  Parse changes and update current values
    //  Changes are like:
    //  {
    //     "microSkillId1" : {
    //          "mccSession1": {
    //              "data": {
    //                  "1" : {
    //                      "notes": [{
    //                          "path1/to/elp": 15.0,
    //                          "path2/to/elp": 20.0,
    //                      }, {
    //                          "path1/to/elp": 30.0,
    //                          "path2/to/elp": 15.0,
    //                          "path3/to/elp": null,
    //                      }],
    //                      "common": true,
    //                  },
    //                  "2" : {
    //                      "notes": [{
    //                          "": 50.0
    //                      }, {
    //                          "": 30.0
    //                      }],
    //                      "common": false,
    //                      "path": "path1/to/elp"
    //                  }
    //              }
    //          },
    //          "mccSession2": {
    //              "data": {...}
    //          }
    //     },
    //     "microSkillId2" : {...},
    //     "microSkillId3" : {...}
    //  }
    // --------------------------------------------------
    // Current MCC entries are like:
    // [
    //  {
    //      "common": true,
    //      "notes": [
    //          {
    //              "label": "Note A",
    //              "ponderation: 100.0
    //              "skills": [
    //                  { path : "path1/to/elp", "skill": "microSkillId1", "ponderation" : 30.0},
    //                  { path : "path2/to/elp", "skill": "microSkillId1", "ponderation" : 20.0}
    //              ]
    //          },
    //          {
    //              "label": "Note B",
    //              "ponderation: 100.0
    //              "skills": [
    //                  { path : "path1/to/elp", "skill": "microSkillId2", "ponderation" : 10.0},
    //                  { path : "path2/to/elp", "skill": "microSkillId3", "ponderation" : 10.0}
    //              ]
    //          }
    //      ]
    //  },
    //  {
    //      "common": false,
    //      "path": "path1/to/elp"
    //      "notes": [
    //          {
    //              "label": "Note C",
    //              "ponderation: 100.0
    //              "skills": [
    //                  { "skill": "microSkillId1", "ponderation" : 35.0},
    //                  { "skill": "microSkillId2", "ponderation" : 15.0}
    //              ]
    //          }
    //      ]
    //  }
    // ]
    @SuppressWarnings("unchecked")
    private void _applyChanges(List<Map<String, Object>> currentMccEntries, String mccSessionName, Map<String, Object> changes)
    {
        for (String microSkillId : changes.keySet())
        {
            Map<String, Object> microSkillChanges = (Map<String, Object>) changes.get(microSkillId);
            
            if (microSkillChanges.containsKey(mccSessionName))
            {
                Map<String, Object>  microSkillChangesForMccSession = (Map<String, Object>) ((Map<String, Object>) microSkillChanges.get(mccSessionName)).getOrDefault("data", Map.of());
                
                for (String mccEntryPos : microSkillChangesForMccSession.keySet())
                {
                    int entryPos = Integer.valueOf(mccEntryPos);
                    Map<String, Object> currentMCCEntry = currentMccEntries.get(entryPos - 1); // current values of MCC entry
                    if (!currentMCCEntry.containsKey("notes"))
                    {
                        currentMCCEntry.put("notes", new ArrayList<>());
                    }
                    List<Map<String, Object>> currentNotes = (List<Map<String, Object>>) currentMCCEntry.get("notes"); // current notes of MCC entry
                    
                    Map<String, Object> mccEntryChanges = (Map<String, Object>) microSkillChangesForMccSession.get(mccEntryPos);
                    List<Map<String, Object>> notesChanges = (List<Map<String, Object>>) mccEntryChanges.getOrDefault("notes", List.of());
                    
                    boolean common = (boolean) currentMCCEntry.getOrDefault("common", true); // is common MCC entry ?
                    
                    int noteIndex = 0;
                    for (Map<String, Object> noteChanges : notesChanges)
                    {
                        if (currentNotes.size() < noteIndex + 1)
                        {
                            // new entry for notes repeater
                            Map<String, Object> newNoteEntry = new HashMap<>();
                            newNoteEntry.put("ponderation", 100.0);
                            currentNotes.add(newNoteEntry);
                        }
                        Map<String, Object> currentNote = currentNotes.get(noteIndex);
                        if (!currentNote.containsKey("skills"))
                        {
                            currentNote.put("skills", new ArrayList<>());
                        }
                        List<Map<String, Object>> currentSkills = (List<Map<String, Object>>) currentNote.get("skills");
                        
                        for (String elpPath : noteChanges.keySet())
                        {
                            Number ponderation = (Number) noteChanges.get(elpPath); // new ponderation for micro skill
                            
                            if (common && StringUtils.isNotEmpty(elpPath))
                            {
                                _applyChangesOnCommonEntry(currentSkills, microSkillId, elpPath, ponderation);
                            }
                            else if (!common)
                            {
                                _applyChangesOnNotCommonEntry(currentSkills, microSkillId, ponderation);
                            }
                        }
                        noteIndex++;
                    }
                }
            }
        }
    }
    
    private void _applyChangesOnCommonEntry(List<Map<String, Object>> currentSkills, String microSkillId, String elpPath, Number ponderation)
    {
        // Find skill entry with same micro skill and ELP path
        Optional<Map<String, Object>> matchedSkillEntry = currentSkills
                .stream()
                .filter(e -> e.get("skill") instanceof ContentValue skillContent && microSkillId.equals(skillContent.getContentId())  // stored skills are ContentValue
                        || e.get("skill") instanceof String skillId && microSkillId.equals(skillId)) // new skills are String
                .filter(e -> elpPath.equals(e.get("path").toString()))
                .findFirst();
            
        if (matchedSkillEntry.isPresent())
        {
            if (ponderation == null)
            {
                // remove skill entry
                currentSkills.remove(matchedSkillEntry.get());
            }
            else
            {
                // update ponderation
                matchedSkillEntry.get().put("ponderation", ponderation.doubleValue());
            }
        }
        else if (ponderation != null)
        {
            // Create new skill entry (DO NOT use immutable map, same changes could be applied twice because of data sent by client side)
            Map<String, Object> newSkillEntry = new HashMap<>();
            newSkillEntry.put("skill", microSkillId);
            newSkillEntry.put("ponderation", ponderation.doubleValue());
            newSkillEntry.put("path", elpPath);
            currentSkills.add(newSkillEntry);
        }
    }
    
    private void _applyChangesOnNotCommonEntry(List<Map<String, Object>> currentSkills, String microSkillId, Number ponderation)
    {
        // Find skill entry with same micro skill
        Optional<Map<String, Object>> matchedSkillEntry = currentSkills
                .stream()
                .filter(e -> e.get("skill") instanceof ContentValue skillContent && microSkillId.equals(skillContent.getContentId()) // stored skills are ContentValue
                                || e.get("skill") instanceof String skillId && microSkillId.equals(skillId)) // new skills are String
                .findFirst();
        
        if (matchedSkillEntry.isPresent())
        {
            if (ponderation == null)
            {
                // remove skill entry
                currentSkills.remove(matchedSkillEntry.get());
            }
            else
            {
                // update ponderation
                matchedSkillEntry.get().put("ponderation", ponderation.doubleValue());
            }
        }
        else if (ponderation != null)
        {
            // Create new skill entry (DO NOT use immutable map, same changes could be applied twice because of data sent by client side)
            Map<String, Object> newSkillEntry = new HashMap<>();
            newSkillEntry.put("skill", microSkillId);
            newSkillEntry.put("ponderation", ponderation.doubleValue());
            currentSkills.add(newSkillEntry);
        }
    }
    
    private List<Map<String, Object>> _getRepeaterEntries(Content content, RepeaterDefinition repeaterDefinition)
    {
        List<String> itemPaths = new ArrayList<>();
        itemPaths.add(repeaterDefinition.getPath());
        
        // Add items needed to evaluate disabled conditions
        itemPaths.addAll(_getDisabledConditionPaths(repeaterDefinition));
        
        View view = View.of(content.getModel(), itemPaths.toArray(String[]::new));
        
        Map<String, Object> dataToMap = content.dataToMap(view, DataContext.newInstance().withDisabledValues(true).withEmptyValues(false));
        
        return _getRepeaterEntries(dataToMap, repeaterDefinition.getPath());
    }
    
    
    @SuppressWarnings("unchecked")
    private List<Map<String, Object>> _getRepeaterEntries(Map<String, Object> data, String path)
    {
        String[] pathSegments = StringUtils.split(path, ModelItem.ITEM_PATH_SEPARATOR);
        if (pathSegments.length == 1)
        {
            return (List<Map<String, Object>>) data.get(pathSegments[0]);
        }
        else
        {
            Map<String, Object> subData = (Map<String, Object>) data.get(pathSegments[0]);
            String remainPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 1, pathSegments.length);
            return _getRepeaterEntries(subData, remainPath);
        }
    }
    
    private Set<String> _getDisabledConditionPaths(ModelItem modelItem)
    {
        DisableConditions<? extends DisableCondition> disableConditions = modelItem.getDisableConditions();
        if (disableConditions != null)
        {
            Set<String> disableConditionPaths = disableConditions.getConditions()
                    .stream()
                    .filter(AbstractRelativeDisableCondition.class::isInstance)
                    .map(c -> ModelHelper.getDisableConditionAbsolutePath(c, modelItem.getPath()))
                    .collect(Collectors.toSet());
            
            if (modelItem instanceof ModelItemGroup group)
            {
                for (ModelItem childModelItem : group.getChildren())
                {
                    disableConditionPaths.addAll(_getDisabledConditionPaths(childModelItem));
                }
            }
            
            return disableConditionPaths;
        }
        return Set.of();
    }
    
    @Override
    protected boolean _canEditCourse(Course course, List<String> path)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS, List.of(EducationalPath.of(path.toArray(String[]::new))));
        
        return super._canEditCourse(course, path)
                || _contentWorkflowHelper.isAvailableAction(course, EditContextualizedDataFunction.EDIT_WORKFLOW_ACTION_ID);
    }
    
    /**
     * Determines if current user can edit repeater with educational path as a consummer of content.
     * @param contentId the id of content
     * @param path the path of content in the current tree
     * @return true if user is allowed to edit data for given path
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public boolean canEditRepeaterWithPath(String contentId, List<String> path)
    {
        WorkflowAwareContent content = _ametysResolver.resolveById(contentId);
        
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS, List.of(EducationalPath.of(path.toArray(String[]::new))));
        
        return _contentWorkflowHelper.isAvailableAction(content, EditContextualizedDataFunction.EDIT_WORKFLOW_ACTION_ID);
    }
}
